/*
 * Copyright (c) 2002, 2003 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <glib.h>
#include <sys/types.h>
#include <regex.h>
#include <stdarg.h>

/*** API implementation ******************************************************/

/**
 * st_re_parse:
 * @regexp: a regular expression to match.
 * @str: a string to match @regexp against.
 * @...: a list of char ** pointers for storing substrings of
 * @str. The number of char ** pointers provided must be equal to
 * @regexp->re_nsub. The strings stored in the pointers should be freed
 * when no longer needed.
 *
 * Checks if @str matches @regexp, and if it does, distributes the
 * substrings of @str matched by the parenthesized subexpressions of
 * @regexp.
 *
 * <informalexample><programlisting>
 * int status;
 * regex_t re;
 * gboolean matches;
 * char *value1, *value2;
 * &nbsp;
 * status = regcomp(&amp;re, "name1=(.*) name2=(.*)", REG_EXTENDED);
 * g_assert(status == 0);
 * &nbsp;
 * matches = st_re_parse(&amp;re, "name1=foo name2=bar", &amp;value1, &amp;value2);
 * </programlisting></informalexample>
 *
 * Return value: %TRUE if @str matches @regexp.
 **/
gboolean
st_re_parse (const regex_t *regexp, const char *str, ...)
{
  g_return_val_if_fail(regexp != NULL, FALSE);
  g_return_val_if_fail(str != NULL, FALSE);
  
  {
    regmatch_t pmatch[regexp->re_nsub + 1];
    
    if (regexec(regexp, str, regexp->re_nsub + 1, pmatch, 0) == 0)
      {
	va_list ap;
	
	char **sub;
	int i;
	
	va_start(ap, str);
	
	for (i = 0; i < regexp->re_nsub; i++)
	  {
	    int index = i + 1;
	    
	    sub = va_arg(ap, char **);
	    
	    g_return_val_if_fail(sub != NULL, FALSE);
	    
	    *sub = g_strndup(&str[pmatch[index].rm_so],
			     pmatch[index].rm_eo - pmatch[index].rm_so);
	  }
	
	va_end(ap);
	
	return TRUE;
      }
  }

  return FALSE;
}
