#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "log.h"
#include "socklib.h"
#include "udpcast.h"
#include "udpc-protoc.h"
#include "fifo.h"
#include "udp-receiver.h"
#include "util.h"
#include "produconsum.h"
#include "statistics.h"

#ifndef O_BINARY
# define O_BINARY 0
#endif

static int sendConnectReq(struct client_config *client_config,
			  struct net_config *net_config,
			  int haveServerAddress) {
    struct connectReq connectReq;
    int endianness = client_config->endianness;    

    if(net_config->flags & FLAG_PASSIVE)
	return 0;

    connectReq.opCode = htoxs(CMD_CONNECT_REQ);
    connectReq.reserved = 0;
    connectReq.capabilities = htonl(RECEIVER_CAPABILITIES);
    connectReq.rcvbuf = htonl(getRcvBuf(client_config->S_UCAST));
    if(haveServerAddress)
      return SSEND(connectReq);
    else
      return BCAST_CONTROL(client_config->S_UCAST, connectReq);
}

int sendGo(struct client_config *client_config) {
    struct go go;
    int endianness = client_config->endianness;
    go.opCode = htoxs(CMD_GO);
    go.reserved = 0;
    return SSEND(go);
}

#ifndef __CYGWIN__
static void sendDisconnectWrapper(int exitStatus,
				  void *args) {    
    sendDisconnect(exitStatus, (struct client_config *) args);
}
#endif

void sendDisconnect(int exitStatus,
		    struct client_config *client_config) {    
    int endianness = client_config->endianness;
    struct disconnect disconnect;
    disconnect.opCode = htoxs(CMD_DISCONNECT);
    disconnect.reserved = 0;
    SSEND(disconnect);
    if (exitStatus == 0)
	udpc_flprintf("Transfer complete.\007\n");
}


struct startTransferArgs {
    int fd;
    int pipeFd;
    struct client_config *client_config;
    int doWarn;
};

int startReceiver(int doWarn,
		  struct disk_config *disk_config,
		  struct net_config *net_config)
{
    char ipBuffer[16];
    union serverControlMsg Msg;
    int connectReqSent=0;
    struct client_config client_config;
    int outFile=1;
    int pipedOutFile;
    struct sockaddr myIp;
    int pipePid = 0;
    int origOutFile;
    int haveServerAddress;

    client_config.sender_is_newgen = 0;
    if(disk_config->fileName != NULL) {
	int oflags = O_CREAT | O_WRONLY;
	if(!(disk_config->flags & FLAG_NOSYNC)) {
	    oflags |= O_SYNC;
	}
	outFile = open(disk_config->fileName, oflags | O_BINARY, 0644);
	if(outFile < 0) {
#ifdef NO_BB
	    extern int errno;
#endif
	    udpc_fatal(1, "open outfile %s: %s\n",
		       disk_config->fileName, strerror(errno));
	}
    }

    net_config->net_if = getIfName(net_config->net_if);
    zeroSockArray(client_config.socks, NR_CLIENT_SOCKS);

    client_config.S_UCAST = makeSocket(ADDR_TYPE_UCAST,
				       net_config->net_if,
				       0, RECEIVER_PORT(net_config->portBase));
    client_config.S_BCAST = makeSocket(ADDR_TYPE_BCAST,
				       net_config->net_if,
				       0, RECEIVER_PORT(net_config->portBase));

    if(net_config->ttl == 1 && net_config->mcastAll == NULL) {
	getBroadCastAddress(client_config.S_UCAST, 
			    net_config->net_if,
			    &net_config->controlMcastAddr,
			    SENDER_PORT(net_config->portBase));
	setSocketToBroadcast(client_config.S_UCAST);
    } else {
	getMcastAllAddress(client_config.S_UCAST, 
			   net_config->net_if,
			   &net_config->controlMcastAddr,
			   net_config->mcastAll,
			   SENDER_PORT(net_config->portBase));
	setMcastDestination(client_config.S_UCAST, net_config->net_if,
			    &net_config->controlMcastAddr);
	setTtl(client_config.S_UCAST, net_config->ttl);

	client_config.S_MCAST_CTRL =
	  makeSocket(ADDR_TYPE_MCAST,
		     net_config->net_if,
		     &net_config->controlMcastAddr,
		     RECEIVER_PORT(net_config->portBase));
	// TODO: subscribe address as receiver to!
    }
    clearIp(&net_config->dataMcastAddr);
    udpc_flprintf("%sUDP receiver for %s at ", 
		  disk_config->pipeName == NULL ? "" :  "Compressed ",
		  disk_config->fileName == NULL ? "(stdout)":disk_config->fileName);
    printMyIp(net_config->net_if, client_config.S_UCAST);
    udpc_flprintf(" on %s\n", net_config->net_if);

    connectReqSent = 0;
    haveServerAddress = 0;

    client_config.endianness = PC_ENDIAN;
    if (sendConnectReq(&client_config, net_config, haveServerAddress) < 0) {
	perror("sendto to locate server");
	exit(1);
    }

    client_config.endianness = NET_ENDIAN;
    client_config.clientNumber= 0; /*default number for asynchronous transfer*/
    /*flprintf("Endian = NET\n");*/
    while(1) {
	// int len;
	int msglen;
	int sock;

	if (!connectReqSent) {
	    if (sendConnectReq(&client_config, net_config,
			       haveServerAddress) < 0) {
		perror("sendto to locate server");
		exit(1);
	    }
	    connectReqSent = 1;
	}

	haveServerAddress=0;

	sock = udpc_selectSock(client_config.socks, NR_CLIENT_SOCKS);

	// len = sizeof(server);
	msglen=RECV(sock, 
		    Msg, client_config.serverAddr, net_config->portBase);
	if (msglen < 0) {
	    perror("recvfrom to locate server");
	    exit(1);
	}
	
	if(getPort(&client_config.serverAddr) != 
	   SENDER_PORT(net_config->portBase))
	    /* not from the right port */
	    continue;

	switch(ntohs(Msg.opCode)) {
	    case CMD_CONNECT_REPLY:
		client_config.endianness = NET_ENDIAN;
		/*flprintf("Endian = NET\n");*/
		client_config.clientNumber = ntohl(Msg.connectReply.clNr);
		net_config->blockSize = ntohl(Msg.connectReply.blockSize);

		udpc_flprintf("received message, cap=%08lx\n",
			      (long) ntohl(Msg.connectReply.capabilities));
		if(ntohl(Msg.connectReply.capabilities) & CAP_NEW_GEN) {
		    client_config.sender_is_newgen = 1;
		    copyFromMessage(&net_config->dataMcastAddr,
				    Msg.connectReply.mcastAddr);
		}
		if (client_config.clientNumber == -1) {
		    udpc_fatal(1, "Too many clients already connected\n");
		}
		goto break_loop;

	    case CMD_HELLO:
		client_config.endianness = NET_ENDIAN;
		/*flprintf("Endian = NET\n");*/
		connectReqSent = 0;
		if(ntohl(Msg.hello.capabilities) & CAP_NEW_GEN) {
		    client_config.sender_is_newgen = 1;
		    copyFromMessage(&net_config->dataMcastAddr,
				    Msg.hello.mcastAddr);
		    net_config->blockSize = pctohl(Msg.hello.blockSize);
		    if(ntohl(Msg.hello.capabilities) & CAP_ASYNC)
			net_config->flags |= FLAG_PASSIVE;
		    if(net_config->flags & FLAG_PASSIVE)
			goto break_loop;
		}
		haveServerAddress=1;
		continue;
	    case CMD_CONNECT_REQ:
		client_config.endianness = NET_ENDIAN;
		/*flprintf("Endian = NET\n");*/
		continue;
	    default:
		break;
	}


	switch(pctohs(Msg.opCode)) {
	    case CMD_CONNECT_REPLY:
		client_config.endianness = PC_ENDIAN;
		/*flprintf("Endian = PC\n");*/
		client_config.clientNumber = pctohl(Msg.connectReply.clNr);
		net_config->blockSize = pctohl(Msg.connectReply.blockSize);
		if (client_config.clientNumber == -1) {
		    udpc_fatal(1, "Too many clients already connected\n");
		}
		goto break_loop;

	    case CMD_HELLO:
		if(msglen > 4)
		    client_config.endianness = NET_ENDIAN;
		else
		    client_config.endianness = PC_ENDIAN;
		/*flprintf("Endian = %s\n",
		  client_config.endianness == NET_ENDIAN ? "NET" : "PC");*/
		if(ntohl(Msg.hello.capabilities) & CAP_NEW_GEN) {
		    client_config.sender_is_newgen = 1;
		    copyFromMessage(&net_config->dataMcastAddr,
				    Msg.hello.mcastAddr);
		    net_config->blockSize = ntohs(Msg.hello.blockSize);
		    if(ntohl(Msg.hello.capabilities) & CAP_ASYNC)
			net_config->flags |= FLAG_PASSIVE;
		    if(net_config->flags & FLAG_PASSIVE)
			goto break_loop;
		}
		haveServerAddress=1;
		connectReqSent = 0;
		continue;
	
	    case CMD_CONNECT_REQ:
		client_config.endianness = PC_ENDIAN;
		/*flprintf("Endian = PC\n");*/
		continue;
	    default:
		break;
	}

	udpc_fatal(1, 
		   "Bad server reply %04x. Other transfer in progress?\n",
		   (unsigned short) ntohs(Msg.opCode));
    }

 break_loop:
    udpc_flprintf("Connected as #%d to %s\n", 
		  client_config.clientNumber, 
		  getIpString(&client_config.serverAddr, ipBuffer));

    getMyAddress(client_config.S_UCAST, net_config->net_if, &myIp);

    if(!ipIsZero(&net_config->dataMcastAddr)  &&
       !ipIsEqual(&net_config->dataMcastAddr, &myIp)) {
	udpc_flprintf("Listening to multicast on %s\n",
		      getIpString(&net_config->dataMcastAddr, ipBuffer));
	client_config.S_MCAST_DATA = 
	  makeSocket(ADDR_TYPE_MCAST, net_config->net_if, 
		     &net_config->dataMcastAddr, 
		     RECEIVER_PORT(net_config->portBase));
    }


    if(net_config->requestedBufSize) {
      int i;
      for(i=0; i<NR_CLIENT_SOCKS; i++)
	if(client_config.socks[i] != -1)
	  setRcvBuf(client_config.socks[i],net_config->requestedBufSize);
    }

    origOutFile = outFile;
    pipedOutFile = openPipe(outFile, disk_config, &pipePid);

#ifndef __CYGWIN__
    on_exit(sendDisconnectWrapper, &client_config);
#endif
    {
	struct fifo fifo;
	void *returnValue;
	receiver_stats_t stats = allocReadStats(origOutFile);
	
	udpc_initFifo(&fifo, net_config->blockSize);

	fifo.data = pc_makeProduconsum(fifo.dataBufSize, "receive");

	client_config.isStarted = 0;

	if((net_config->flags & (FLAG_PASSIVE|FLAG_NOKBD))) {
	  /* No console used */
	  client_config.console = NULL;
	} else {
	  if(doWarn)
	    udpc_flprintf("WARNING: This will overwrite the hard disk of this machine\n");
	  client_config.console = prepareConsole(0);
	  udpc_flprintf("Press any key to start receiving data!\n");
	}

	spawnNetReceiver(&fifo,&client_config, net_config, stats);
	writer(&fifo, pipedOutFile);
	if(pipePid) {
	    close(pipedOutFile);
	}
	pthread_join(client_config.thread, &returnValue);

	/* if we have a pipe, now wait for that too */
	if(pipePid) {
	    waitForProcess(pipePid, "Pipe");
	}
	fsync(origOutFile);
	displayReceiverStats(stats);

    }
    return 0;
}
