/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Abner Jose de Faria Silva <abner.silva@indt.org.br>
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 * @author  Marcio Macedo <marcio.macedo@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "tpa-channel-target-priv.h"
#include "tpa-channel.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-debug.h>

enum{
    PROP_0,
    PROP_HANDLE
};

struct _TpaChannelTargetPrivate
{
    TpaHandle *handle;
    gboolean disposed;
};

static void             tpa_channel_target_get_property       (GObject * object,
                                                               guint prop_id,
                                                               GValue * value,
                                                               GParamSpec * pspec);
static void             tpa_channel_target_set_property       (GObject * object,
                                                               guint prop_id,
                                                               const GValue * value,
                                                               GParamSpec * pspec);

G_DEFINE_TYPE(TpaChannelTarget, tpa_channel_target, TPA_TYPE_OBJECT)

static void
tpa_channel_target_dispose (GObject *object)
{
    TpaChannelTarget *self = TPA_CHANNEL_TARGET (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    if (self->priv->handle)
        g_object_unref (self->priv->handle);

    G_OBJECT_CLASS (tpa_channel_target_parent_class)->dispose (object);
}

static void
tpa_channel_target_class_init (TpaChannelTargetClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_channel_target_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaChannelTargetPrivate));

    gobject_class->set_property = tpa_channel_target_set_property;
    gobject_class->get_property = tpa_channel_target_get_property;

    gobject_class->dispose = tpa_channel_target_dispose;

    g_object_class_install_property (gobject_class,
                                     PROP_HANDLE,
                                     g_param_spec_object ("handle",
                                     "Handle",
                                     "Handle",
                                     TPA_TYPE_HANDLE,
                                     G_PARAM_CONSTRUCT_ONLY |G_PARAM_READWRITE));
}

static void
tpa_channel_target_init (TpaChannelTarget *self)
{
    self->priv = TPA_CHANNEL_TARGET_GET_PRIVATE (self);
    self->priv->disposed = FALSE;
    self->priv->handle = NULL;
}

static void
tpa_channel_target_get_property (GObject * object,
                                 guint prop_id,
                                 GValue * value,
                                 GParamSpec * pspec)
{
    TpaChannelTarget *self = TPA_CHANNEL_TARGET(object);

    switch (prop_id) {
        case PROP_HANDLE:
            g_value_set_object (value, self->priv->handle);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
tpa_channel_target_set_property (GObject * object,
                                 guint prop_id,
                                 const GValue * value,
                                 GParamSpec * pspec)
{
    TpaChannelTarget *self = TPA_CHANNEL_TARGET(object);

    switch (prop_id) {
        case PROP_HANDLE:
            self->priv->handle = g_value_get_object (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

TpaChannelTarget *
tpa_channel_target_new (TpaHandle *handle)
{
    TpaChannelTarget *self;

    self = TPA_CHANNEL_TARGET (g_object_new (TPA_TYPE_CHANNEL_TARGET,
                               "handle", handle,
                               NULL));
    return self;
}

/**
 * tpa_channel_target_get_uri:
 * @self: #TpaChannelTarget instance.
 * @returns: String uri.
 *
 * Get uri of the target.
 */
const gchar *
tpa_channel_target_get_uri (TpaChannelTarget *self)
{
    const gchar *uri;
    VERBOSE ("(%p)", self);
    g_assert (self);

    uri = tpa_handle_get_name (self->priv->handle);
    VERBOSE ("return %s", uri);
    return uri;
}

/**
 * tpa_channel_target_get_uri:
 * @self: #TpaChannelTarget instance
 * @returns: #TpaHandle handle
 *
 * Get handle of the target.
 */
TpaHandle *
tpa_channel_target_get_handle (TpaChannelTarget *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %p", self->priv->handle);
    return self->priv->handle;
}

