/*
 * Tapioca library
 * Copyright (C) 2007 INdT.
 * @author  Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "tpa-video-stream.h"
#include "tpa-video-stream-priv.h"
#include "tpa-channel-priv.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-debug.h>

struct _TpaVideoStreamPrivate {
    DBusGProxy *stream_engine_proxy;
    gboolean disposed;
    guint window_id;
};

G_DEFINE_TYPE(TpaVideoStream, tpa_video_stream, TPA_TYPE_STREAM)

static GObject*
tpa_video_stream_constructor (GType type,
                              guint n_construct_params,
                              GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaVideoStream *self;
    DBusGConnection *bus;
    GError *error = NULL;

    object = G_OBJECT_CLASS (tpa_video_stream_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_VIDEO_STREAM (object);

    bus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);

    if (!bus) {
        ERROR ("failed to open connection to dbus");
        g_error_free (error);
        return NULL;
    }

    self->priv->stream_engine_proxy = dbus_g_proxy_new_for_name (bus,
                                        "org.freedesktop.Telepathy.StreamEngine",
                                        "/org/freedesktop/Telepathy/StreamEngine",
                                        "org.freedesktop.Telepathy.StreamEngine");
    tpa_object_add_proxy (TPA_OBJECT (self), self->priv->stream_engine_proxy);

    return object;
}

static void
tpa_video_stream_dispose (GObject *object)
{
    TpaVideoStream *self = TPA_VIDEO_STREAM (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    if (self->priv->stream_engine_proxy)
        g_object_unref (self->priv->stream_engine_proxy);

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    G_OBJECT_CLASS (tpa_video_stream_parent_class)->dispose (object);
}

static void
tpa_video_stream_class_init (TpaVideoStreamClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_video_stream_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaVideoStreamPrivate));

    gobject_class->dispose = tpa_video_stream_dispose;
    gobject_class->constructor = tpa_video_stream_constructor;
}

static void
tpa_video_stream_init (TpaVideoStream *self)
{
    self->priv = TPA_VIDEO_STREAM_GET_PRIVATE (self);
    self->priv->stream_engine_proxy = NULL;
    self->priv->window_id = 0;
    self->priv->disposed = FALSE;
}

TpaVideoStream *
tpa_video_stream_new (TpaChannel *channel,
                      guint id,
                      guint contact,
                      guint type,
                      guint state,
                      guint direction,
                      guint flags)
{
    TpaVideoStream *self = NULL;

    g_return_val_if_fail (channel != NULL, NULL);

    /* Pass givin arguments to constructor */
    self = TPA_VIDEO_STREAM (g_object_new (TPA_TYPE_VIDEO_STREAM,
                                           "channel", channel,
                                           "id", id,
                                           "contact", contact,
                                           "type", type,
                                           "state", state,
                                           "direction", direction,
                                           "flags", flags,
                                           NULL));

    return self;
}

TpaVideoStream *
tpa_video_stream_new_incoming (TpaChannel *channel,
                               guint id,
                               guint contact,
                               guint type)
{
    TpaVideoStream *self = NULL;

    g_return_val_if_fail (channel != NULL, NULL);

    /* Pass givin arguments to constructor */
    self = TPA_VIDEO_STREAM (g_object_new (TPA_TYPE_VIDEO_STREAM,
                                           "channel", channel,
                                           "id", id,
                                           "contact", contact,
                                           "type", type,
                                           NULL));

    return self;
}

void
tpa_video_stream_set_window_id (TpaVideoStream *self,
                                guint id)
{
    const gchar *path;
    GError *error = NULL;

    VERBOSE ("(%p, %d)", self, id);
    g_assert (self);

    if (self->priv->window_id != id) {
        INFO ("setting video output of %d to stream %d", id, tpa_stream_get_id (TPA_STREAM (self)));
        path = tpa_channel_get_object_path (tpa_stream_get_channel (TPA_STREAM (self)));
        dbus_g_proxy_call (self->priv->stream_engine_proxy, "SetOutputWindow", &error,
                           DBUS_TYPE_G_OBJECT_PATH, path,
                           G_TYPE_UINT, tpa_stream_get_id (TPA_STREAM (self)),
                           G_TYPE_UINT, id,
                           G_TYPE_INVALID, G_TYPE_INVALID);
        if (error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
        else
            self->priv->window_id = id;
    }

    VERBOSE ("return");
}

guint
tpa_video_stream_get_window_id (TpaVideoStream *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return");

    return self->priv->window_id;
}

void
tpa_video_stream_add_preview_window_id (TpaVideoStream *self,
                                        guint id)
{
    GError *error = NULL;

    VERBOSE ("(%p, %d)", self, id);
    g_assert (self);

    INFO ("adding video input %d", id);
    dbus_g_proxy_call (self->priv->stream_engine_proxy, "AddPreviewWindow", &error,
                       G_TYPE_UINT, id,
                       G_TYPE_INVALID, G_TYPE_INVALID);
    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

   VERBOSE ("return");
}

void
tpa_video_stream_remove_preview_window_id (TpaVideoStream *self,
                                           guint id)
{
    GError *error = NULL;

    VERBOSE ("(%p, %d)", self, id);
    g_assert (self);

    INFO ("removing video input %d", id);
    dbus_g_proxy_call (self->priv->stream_engine_proxy, "RemovePreviewWindow", &error,
                       G_TYPE_UINT, id,
                       G_TYPE_INVALID, G_TYPE_INVALID);
    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

    VERBOSE ("return");
}
