/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <tapioca/core/tpa-manager.h>
#include <tapioca/core/tpa-account.h>
#include <tapioca/core/tpa-session.h>
#include <tapioca/core/tpa-media.h>
#include <tapioca/core/tpa-ice.h>
#include <tapioca/core/tpa-group.h>
#include <tapioca/core/tpa-ifaces.h>
#include <tapioca/base/tpa-errors.h>
#include <tapioca/base/tpa-debug.h>

#include <glib.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <signal.h>
#include <sys/types.h>

#define _DBUS_SERVICE "org.freedesktop.Telepathy"
G_BEGIN_DECLS


/* TYPE MACROS */
#define ACCOUNT_TYPE             (account_get_type())
#define ACCOUNT(obj)             (G_TYPE_CHECK_INSTANCE_CAST((obj), ACCOUNT_TYPE, Account))
#define ACCOUNT_CLASS(klass)     (G_TYPE_CHECK_CLASS_CAST((klass), ACCOUNT_TYPE, AaccountClass))
#define IS_ACCOUNT(obj)          (G_TYPE_CHECK_INSTANCE_TYPE((obj), ACCOUNT_TYPE))
#define IS_ACCOUNT_CLASS(klass)  (G_TYPE_CHECK_CLASS_TYPE((klass), ACCOUNT_TYPE))
#define ACCOUNT_GET_CLASS(obj)   (G_TYPE_INSTANCE_GET_CLASS ((obj), ACCOUNT_TYPE, AccountClass))
#define ACCOUNT_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), ACCOUNT_TYPE, AccountPrivate))

typedef struct _Account      Account;
typedef struct _AccountClass AccountClass;

struct _AccountClass {
    TpaAccountClass parent_class;
};

struct _Account {
    TpaAccount parent;
};

GType account_get_type (void);

G_END_DECLS

G_DEFINE_TYPE(Account, account, TPA_TYPE_ACCOUNT)

static void
account_dispose (GObject *object)
{
}

static TpaError
account_connect (GObject *obj)
{
    g_debug ("%s: (%p)", G_STRFUNC, obj);
    tpa_connection_signal_status_changed (obj,
        TPA_CONNECTION_STATUS_CONNECTED, TPA_CONNECTION_STATUS_REASON_REQUESTED);
    return TPA_ERROR_NONE;
}

static TpaError
account_disconnect (GObject *obj)
{
    g_debug ("%s: (%p)", G_STRFUNC, obj);
    tpa_connection_signal_status_changed (obj,
        TPA_CONNECTION_STATUS_DISCONNECTED, TPA_CONNECTION_STATUS_REASON_REQUESTED);
    return TPA_ERROR_NONE;
}

static TpaError
account_get_interfaces (GObject *obj,
                        gchar ***ret)
{
    const gchar *interfaces[] = { DBUS_ALIASING_IFACE, DBUS_AVATARS_IFACE,
        DBUS_AVATARS_IFACE, DBUS_CAPABILITIES_IFACE,  DBUS_FORWARDING_IFACE,
        DBUS_PRESENCE_IFACE, DBUS_PROPERTIES_IFACE, NULL };

    g_debug ("%s: (%p, %p)", G_STRFUNC, obj, ret);

    *ret = g_strdupv ((gchar **) interfaces);

    return TPA_ERROR_NONE;
}

static TpaError
account_request_session (GObject *obj,
                         const gchar *type,
                         const gchar *string,
                         gboolean suppress_handler)
{
    g_debug ("%s: (%p, %s, %s, %d)", G_STRFUNC, obj, type, string, suppress_handler);
    return TPA_ERROR_NONE;
}

static void
account_class_init (AccountClass *klass)
{
    GObjectClass *object_class;

    object_class = (GObjectClass *) klass;

    object_class->dispose = account_dispose;
}

static void
account_init (Account *obj)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);

    iface->connect = account_connect;
    iface->disconnect = account_disconnect;
    iface->get_interfaces = account_get_interfaces;
    iface->request_session = account_request_session;
}


/* TYPE MACROS */
#define SESSION_TYPE             (session_get_type())
#define SESSION(obj)             (G_TYPE_CHECK_INSTANCE_CAST((obj), SESSION_TYPE, Session))
#define SESSION_CLASS(klass)     (G_TYPE_CHECK_CLASS_CAST((klass), SESSION_TYPE, SessionClass))
#define IS_SESSION(obj)          (G_TYPE_CHECK_INSTANCE_TYPE((obj), SESSION_TYPE))
#define IS_SESSION_CLASS(klass)  (G_TYPE_CHECK_CLASS_TYPE((klass), SESSION_TYPE))
#define SESSION_GET_CLASS(obj)   (G_TYPE_INSTANCE_GET_CLASS ((obj), SESSION_TYPE, SessionClass))
#define SESSION_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), SESSION_TYPE, SessionPrivate))

typedef struct _Session Session;
typedef struct _SessionClass SessionClass;

struct _SessionClass {
    TpaSessionClass parent_class;
};

struct _Session {
    TpaSession parent;
};

G_END_DECLS

G_DEFINE_TYPE(Session, session, TPA_TYPE_SESSION)

static void
session_dispose (GObject *object)
{
}

static TpaError
session_close (GObject *obj)
{
    g_debug ("%s: (%p)", G_STRFUNC, obj);
    tpa_channel_signal_closed (obj);
    tpa_session_unref (TPA_SESSION (obj));
    return TPA_ERROR_NONE;
}

static void
session_class_init (SessionClass *klass)
{
    GObjectClass *object_class;

    object_class = (GObjectClass *) klass;

    object_class->dispose = session_dispose;
}

static void
session_init (Session *obj)
{
    TpaIChannel *channel_iface = TPA_ICHANNEL (obj);

    channel_iface->close = session_close;
}

/* variables */
static GMainLoop *mainloop;
static Account *account;
static Session *session;

int main (int argc, char **argv)
{
    gchar **protocols = NULL;

    protocols = g_new0 (gchar*, 2);
    protocols[0] = g_strdup ("manager");

    g_type_init ();
    mainloop = g_main_loop_new (NULL, FALSE);

    tpa_debug_set_flags_from_env ();

    account = ACCOUNT (tpa_account_new (ACCOUNT_TYPE, NULL, NULL, "protocol", "account"));
    g_debug ("(%p) Creating %s.Connection service... ", account, _DBUS_SERVICE);
    if (account) {
        g_debug ("OK");
        TpaHandle *handle = tpa_connection_get_contact_handle (G_OBJECT (account),
            "eu@eu");
        session = SESSION (tpa_media_session_new (SESSION_TYPE, TPA_ACCOUNT (account),
            handle, "manager", "name"));
        g_debug ("(%p) Creating %s.Channel service... ", session, _DBUS_SERVICE);
        if (session) {
            g_debug ("OK");
            g_debug ("Listening");
            g_main_run (mainloop);
        }
    }

    return 0;
}
