/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#include "config.h"

#include <QDebug>

#include "QtTapioca/UserContact"
#include "QtTapioca/Handle"
#include "QtTapioca/Avatar"

#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusConnectionInterface>
#include <QtTelepathy/Client/Connection>

namespace QtTapioca {

class UserContactPrivate {
public:
    UserContactPrivate(org::freedesktop::Telepathy::Connection *telepathyConn)
        :bus(QDBusConnection::sessionBus())
    {
        this->telepathyConn = telepathyConn;
    }
    ~UserContactPrivate()
    {
    }

    QDBusConnection bus;
    org::freedesktop::Telepathy::Connection *telepathyConn;
};

}

using namespace QtTapioca;

UserContact::UserContact(org::freedesktop::Telepathy::Connection *telepathyConn,
                           org::freedesktop::Telepathy::ConnectionAvatarsInterface  *iAvatar,
                           org::freedesktop::Telepathy::ConnectionPresenceInterface *iPresence,
                           org::freedesktop::Telepathy::ConnectionAliasingInterface *iAliasing,
                           org::freedesktop::Telepathy::ConnectionCapabilitiesInterface *iCapabilities,
                           Handle *handle,
                           QObject *parent)
    : ContactBase(telepathyConn, iAvatar, iPresence, iAliasing, iCapabilities, handle, parent),
      d(new UserContactPrivate(telepathyConn))
{
}

UserContact::~UserContact()
{
    delete d;
}

bool UserContact::setAvatar(Avatar *avatar)
{
    if ((!avatar) || (!telepathyIAvatar))
        return false;

    QDBusReply<QString> reply = telepathyIAvatar->SetAvatar(avatar->data(), avatar->mimeType());

    if (!reply.isValid()) {
        qDebug() << "error on setting the avatar:" << reply.error().message();
        return false;
    }

    avatar->setToken(reply.value());

    return true;
}

bool UserContact::setPresence(ContactBase::Presence status)
{
    if (!telepathyIPresence)
        return false;

    QString presenceStr = presenceEnumToStr(status);

    setPresenceStatus(presenceStr, presenceMessage());

    m_presence = status;

    return true;
}

bool UserContact::setPresenceMessage(const QString &message)
{
    if (!telepathyIPresence)
        return false;

    QString presenceStr = presenceEnumToStr(presence());

    setPresenceStatus(presenceStr, message);

    m_presenceMessage = message;

    return true;
}

bool UserContact::setPresenceWithMessage(ContactBase::Presence status, const QString &message)
{
    if (!telepathyIPresence)
        return false;

    QString presenceStr = presenceEnumToStr(status);

    setPresenceStatus(presenceStr, message);

    m_presence = status;
    m_presenceMessage = message;

    return true;
}

bool UserContact::setAlias(const QString &alias)
{
    if (!telepathyIAliasing)
        return false;

    QMap<uint, QString> map;
    map.insert(handle()->id(), alias);

    QDBusReply<void> reply = telepathyIAliasing->SetAliases(map);

    if (!reply.isValid()) {
        qDebug() << "error on setting alias:" << reply.error().message();
        return false;
    }

    return true;
}

bool UserContact::setCapabilities(QList<ContactBase::Capability> caps)
{
    if (!telepathyICapabilities)
        return false;

    QStringList remove;
    org::freedesktop::Telepathy::LocalCapabilityInfoList add;
    QDBusReply<org::freedesktop::Telepathy::LocalCapabilityInfoList> reply;

    /* Reset all caps */
    remove << "org.freedesktop.Telepathy.Channel.Type.Text";
    remove << "org.freedesktop.Telepathy.Channel.Type.StreamedMedia";

    reply = telepathyICapabilities->AdvertiseCapabilities(add, remove);

    if (!reply.isValid()) {
        qDebug() << "error on advertise capabilities:" << reply.error().message();
        return false;
    }

    //Check if Text is supported
    if (caps.contains(ContactBase::Text)) {
        org::freedesktop::Telepathy::LocalCapabilityInfo capInfo;
        capInfo.channelType =  "org.freedesktop.Telepathy.Channel.Type.Text";
        add << capInfo;
    }

    //Check if Audio and video are supported
    if ((caps.contains(ContactBase::Audio)) ||
        (caps.contains(ContactBase::Video))) {
        org::freedesktop::Telepathy::LocalCapabilityInfo capInfo;
        capInfo.channelType =  "org.freedesktop.Telepathy.Channel.Type.StreamedMedia";

        if (caps.contains(ContactBase::Audio))
            capInfo.typeSpecificFlags |= 1;
        if (caps.contains(ContactBase::Video))
            capInfo.typeSpecificFlags |= 2;

        add << capInfo;
    }

    remove.clear();

    reply = telepathyICapabilities->AdvertiseCapabilities(add, remove);

    if (!reply.isValid()) {
        qDebug() << "error on advertise capabilities:" << reply.error().message();
        return false;
    }

    return true;
}

void UserContact::setPresenceStatus(const QString &presenceStr, const QString &message)
{
    QMap<QString, QVariant> opt;
    org::freedesktop::Telepathy::PresenceState presenceStatus;

    opt.insert("message", message);

    presenceStatus.insert(presenceStr, opt);

    QDBusReply<void> reply = telepathyIPresence->SetStatus(presenceStatus);

    if (!reply.isValid())
        qDebug() << "error setting presence status:" << reply.error().message();
}
