/*
 *	tardy - a tar post-processor
 *	Copyright (C) 1998, 1999, 2001, 2002, 2004 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate tar input remove prefixes
 */

#include <ac/assert.h>
#include <ac/string.h>

#include <cannonical.h>
#include <tar/input/filter/remov_prefi.h>


tar_input_filter_remove_prefix::~tar_input_filter_remove_prefix()
{
}


static rcstring
cannonicalize_plus_slash(const rcstring &arg)
{
    rcstring tmp = cannonicalize(arg);
    if (tmp.length() == 0 || tmp[tmp.length() - 1] != '/')
	tmp += "/";
    return tmp;
}


tar_input_filter_remove_prefix::tar_input_filter_remove_prefix(
	tar_input *arg_fp, const rcstring &arg_prefix) :
    tar_input_filter(arg_fp),
    prefix(cannonicalize_plus_slash(arg_prefix))
{
}


static rcstring
remove_prefix_inner(const rcstring &prfx, const rcstring &name)
{
    /*
     * The name and the prefix have both been cannonicalized at this
     * point (i.e. no extra slashes, no extra dot dirs).  The prefix
     * is also guaranteed to have a '/' on the end.
     */
    int len_p = prfx.length();
    assert(len_p >= 1);
    assert(prfx[len_p - 1] == '/');

    /*
     * if the prefix matches the name exactly,
     * replace it with "./"
     *
     * Note that the prefix has been canonicalized to have a slash, and
     * the name as been canonicalized NOT to have a slash.
     */
    int len_n = name.length();
    if
    (
	len_p == len_n + 1
    &&
	0 == memcmp(prfx.to_c_string(), name.to_c_string(), len_n)
    )
    {
	return ".";
    }

    /*
     * if the prefix is longer than the name,
     * it can't be a prefix 
     */
    if (len_p > len_n)
	return name;

    /*
     * if it is a leading prefix,
     * remove it from the name
     */
    if (0 == memcmp(prfx.to_c_string(), name.to_c_string(), len_p))
	return rcstring(name.to_c_string() + len_p, len_n - len_p);

    /*
     * prefix does not match the start of the name
     */
    return name;
}


static rcstring
apply_remove_prefix(const rcstring &prefix, const rcstring &name)
{
    return remove_prefix_inner(prefix, cannonicalize(name));
}


int
tar_input_filter_remove_prefix::read_header(tar_header &h)
{
    int ok = tar_input_filter::read_header(h);
    if (ok)
    {
	h.name = apply_remove_prefix(prefix, h.name);
	if (h.type == tar_header::type_link_hard)
    	    h.linkname = apply_remove_prefix(prefix, h.linkname);
    }
    return ok;
}
