/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-conn-manager.c:
 * ConnManager - GObject wrapper for D-Bus method calls to
 *               org.freedesktop.Telepathy.ConnectionManager
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <string.h>
#include <glib.h>
#include <dbus/dbus-glib.h>

#include "ti-conn-manager.h"

G_DEFINE_TYPE (TIConnManager, ti_conn_manager, G_TYPE_OBJECT);

/***
 * Instance private data.
 */
struct _TIConnManagerPrivate {
    gboolean disposed;

    gchar* name;
    DBusGProxy* dbus_proxy;
};
typedef struct _TIConnManagerPrivate TIConnManagerPrivate;


#define TI_CONN_MANAGER_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_CONN_MANAGER, TIConnManagerPrivate))


// Prototypes of private, local, functions.
static void _create_connection_manager_proxy (DBusGProxy** dbus_proxy, DBusGConnection* dbus_conn, const gchar* name);
static gboolean check_parameters (GPtrArray* parameters);


/***
 * Drop all references to other objects.
 */
static void
ti_conn_manager_dispose (GObject *object)
{
    TIConnManager* conn_manager = TI_CONN_MANAGER (object);
    TIConnManagerPrivate* priv = TI_CONN_MANAGER_GET_PRIVATE (conn_manager);

    if (priv->disposed) {
        return;
    } else {
        priv->disposed = TRUE;
    }

    if (priv->dbus_proxy != NULL) {
        g_object_unref (priv->dbus_proxy);
        priv->dbus_proxy = NULL;
    }

    G_OBJECT_CLASS (ti_conn_manager_parent_class)->dispose (object);
}

/***
 * Finalizes the object, marking the memory as ready for reuse
 */
static void
ti_conn_manager_finalize (GObject *object)
{
	TIConnManager* conn_manager = TI_CONN_MANAGER (object);
    TIConnManagerPrivate* priv = TI_CONN_MANAGER_GET_PRIVATE (conn_manager);

    g_free (priv->name);
    priv->name = NULL;

	G_OBJECT_CLASS (ti_conn_manager_parent_class)->finalize (object);
}

/**
 * Class initialization.
 */
static void
ti_conn_manager_class_init (TIConnManagerClass *ti_conn_manager_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_conn_manager_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_conn_manager_dispose;
    gobject_class->finalize = ti_conn_manager_finalize;
		
	/* Add private */
	g_type_class_add_private (ti_conn_manager_class, sizeof (TIConnManagerPrivate));
}

/***
 * Instance initialization.
 */
static void
ti_conn_manager_init (TIConnManager *ti_conn_manager)
{
    TIConnManagerPrivate *priv = TI_CONN_MANAGER_GET_PRIVATE (ti_conn_manager);

    priv->disposed = FALSE;
    priv->dbus_proxy = NULL;
    priv->name = NULL;
}

/**
 * Returns a new instance.
 */
TIConnManager*
ti_conn_manager_new (const gchar* name)
{
    TIConnManager* conn_manager;
    TIConnManagerPrivate* priv;
    DBusGConnection* dbus_conn;
    GError* error;

    error = NULL;
    dbus_conn = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (dbus_conn == NULL)
    {
        g_printerr ("Failed to open connection to bus: %s\n", error->message);
        g_error_free (error);
        return NULL;
    }

	conn_manager = g_object_new (TI_TYPE_CONN_MANAGER, NULL);

    priv = TI_CONN_MANAGER_GET_PRIVATE(conn_manager);

    _create_connection_manager_proxy (&(priv->dbus_proxy), dbus_conn, name);

    if (priv->dbus_proxy == NULL) {
        g_object_unref (conn_manager);
        return NULL;
    }

    priv->name = g_strdup (name);

    return conn_manager;
}

/***
 * Creates a D-Bus connection manager proxy.
 */
static void
_create_connection_manager_proxy (DBusGProxy** dbus_proxy, DBusGConnection* dbus_conn, const gchar* name)
{
    gchar* dbus_name;
    gchar* dbus_path_name;

    *dbus_proxy = NULL;

    if (name == NULL) {
        return;
    }

    dbus_name = g_strdup_printf("org.freedesktop.Telepathy.ConnectionManager.%s", name);
    dbus_path_name = g_strdup_printf("/org/freedesktop/Telepathy/ConnectionManager/%s", name);

    *dbus_proxy = dbus_g_proxy_new_for_name (dbus_conn, dbus_name, dbus_path_name,
                                             "org.freedesktop.Telepathy.ConnectionManager");

    g_free(dbus_name);
    g_free(dbus_path_name);
}

/***
 * List Protocols
 */
GStrv
ti_conn_manager_list_protocols (TIConnManager* conn_manager)
{
    GError *error;
    char **str_list;
    TIConnManagerPrivate* priv = TI_CONN_MANAGER_GET_PRIVATE(conn_manager);

    error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "ListProtocols", &error,
                            G_TYPE_INVALID,
                            G_TYPE_STRV, &str_list, G_TYPE_INVALID)) {
        g_printerr ("Error: %s\n", error->message);
        g_error_free (error);
        return NULL;
    }

    return str_list;
}

/***
 * Get Parameters
 */
GPtrArray*
ti_conn_manager_get_parameters (TIConnManager* conn_manager, const gchar* protocol)
{
    GError *error;
    TIConnManagerPrivate* priv = TI_CONN_MANAGER_GET_PRIVATE(conn_manager);
    static GType protocol_param_gtype = G_TYPE_INVALID;
    GPtrArray* parameters = NULL;
    gboolean ok;
    GValueArray* val_array;
    guint i;

    if (protocol_param_gtype == G_TYPE_INVALID)
    {
        protocol_param_gtype = dbus_g_type_get_struct ("GValueArray",
                                            G_TYPE_STRING,
                                            G_TYPE_UINT,
                                            G_TYPE_STRING,
                                            G_TYPE_VALUE,
                                            G_TYPE_INVALID);
    }
    g_return_val_if_fail (protocol_param_gtype != G_TYPE_INVALID, NULL);

    error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "GetParameters", &error,
                            G_TYPE_STRING, protocol, G_TYPE_INVALID,
                            dbus_g_type_get_collection ("GPtrArray", protocol_param_gtype), &parameters, G_TYPE_INVALID)) {
        g_printerr ("Error: %s\n", error->message);
        g_error_free (error);
        return NULL;
    }

    if (parameters != NULL) {
        ok = check_parameters (parameters);
        if (!ok) {
            for (i = 0; i < parameters->len; i++) {
                val_array = g_ptr_array_index (parameters, i);
                g_value_array_free (val_array);
            }
            g_ptr_array_free (parameters, TRUE);
            parameters = NULL;
        }
    }

    return parameters;
}

/**
 * Check Parameters - private helper function
 * @param parameters An array of GValueArray (GValue(string), GValue(uint), GValue(String), GValue(?)) elements.
 * @return FALSE if the array does not comply to the above specified restriction and TRUE otherwise
 */
static gboolean
check_parameters (GPtrArray* parameters)
{
    GValueArray* param;
    guint i;
    gboolean ok = TRUE;

    for (i = 0; ok && i < parameters->len; i++) {
        param = g_ptr_array_index (parameters, i);

        ok = param->n_values == 4;

        if (ok) {
            ok = G_VALUE_HOLDS_STRING (&(param->values[0])) &&
                 G_VALUE_HOLDS_UINT   (&(param->values[1])) &&
                 G_VALUE_HOLDS_STRING (&(param->values[2])) &&
                 G_VALUE_HOLDS        (&(param->values[3]), G_TYPE_VALUE);
        }
    }

    return ok;
}

/**
 * Request Connection
 */
TIConnection*
ti_conn_manager_request_connection (TIConnManager* conn_manager,
                                    const gchar* protocol,
                                    GHashTable* parameters)
{
    gchar* service_name;
    gchar* obj_path;
    TIConnManagerPrivate* priv = TI_CONN_MANAGER_GET_PRIVATE(conn_manager);
    TIConnection* connection;
    GError* error;

    g_assert (error != NULL);

    error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "RequestConnection", &error,
                            G_TYPE_STRING, protocol, dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE), parameters, G_TYPE_INVALID,
                            G_TYPE_STRING, &service_name, DBUS_TYPE_G_OBJECT_PATH, &obj_path, G_TYPE_INVALID)) {
        if (error != NULL) {
            g_printerr ("Error: %s\n", error->message);
            g_error_free (error);
        }
        return NULL;
    }

    connection = ti_connection_new (service_name, obj_path);

    g_free (service_name);
    g_free (obj_path);

    return connection;
}
