/*
 * spc_col.c - a Color handling module for 'special' DVI instruction
 * by Hirotsugu Kakugawa
 *
 *  2 June 1997  Firsyt implementation
 *
 */
/*
 * Copyright (C) 1997-2001 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This file is part of the DVIlib Library.  This library is free
 * software; you can redistribute it and/or modify it under the terms of
 * the GNU Library General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  This library is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if HAVE_MALLOC_H
#  include <malloc.h>
#endif
#if defined(HAVE_STRING_H)
#  include  <string.h>
#endif
#if defined(HAVE_STRINGS_H)
#  include  <strings.h>
#endif
#include <ctype.h>

#include "libdvi29.h"
#include "defs.h"
#include "cache.h"
#include "private.h"
#include "spc_col.h"

#define COLOR_WHERE_FOREGROUND  0
#define COLOR_WHERE_BACKGROUND  1


static DVI_COLOR   dvi_color_stack_copy2(DVI,DVI_DEVICE,DVI_COLOR);
#if 0
static int         dvi_color_stack_equal(DVI_COLOR s1, DVI_COLOR s2);
#endif

static int   dvi_color_set(DVI,DVI_DEVICE,char*);
static int   dvi_color_push(DVI,DVI_DEVICE,char*);
static int   dvi_color_pop(DVI,DVI_DEVICE);

static int   color_parse(DVI_DEVICE,DVI_COLOR,char*);
static int   color_set(DVI,DVI_DEVICE,DVI_COLOR,char*,int);
static int   do_color_set(DVI,DVI_DEVICE,DVI_COLOR,char*,int,int);

static void  rgb2cmyk(double,double,double,double*,double*,double*,double*);
static void  rgb2gray(double,double,double,double*);
static void  cmyk2rgb(double,double,double,double,double*,double*,double*);
static void  print_color(DVI_COLOR);






Public int  
dvi_special_color(DVI dvi, DVI_DEVICE dev, char* cmd, long x, long y)
{
  int    res;
  char  *p;

  p = &cmd[strlen("color")];
  while (isspace((int)*p))
    p++;
  if (*p == '\0')
    return 0;

#if 0
  printf("** %s\n", cmd);
  dvi_color_stack_print(dvi, dev, PD(dvi,color_stack));
#endif

  if (strncmp(p, "push", 4) == 0){
    p = &p[4];
    while (isspace((int)*p))
      p++;
    if (*p == '\0')
      p = NULL;
    res = dvi_color_push(dvi, dev, p);
  } else if (strncmp(p, "pop", 3) == 0){
    res = dvi_color_pop(dvi, dev);
  } else {
    res = dvi_color_set(dvi, dev, p);
  }

#if 0
  dvi_color_stack_print(dvi, dev, PD(dvi,color_stack));
#endif

  return res;
}

Public int  
dvi_special_textcolor(DVI dvi, DVI_DEVICE dev, char* cmd, long x, long y)
{
  int   res;
  char *p;

  p = &cmd[strlen("textcolor:")];
  while (isspace((int)*p))
    p++;
  if (*p == '\0')
    return 0;

  if (PD(dvi,color_stack) == NULL){
    res = dvi_color_push(dvi, dev, p);
  } else {
    res = dvi_color_set(dvi, dev, p);
  }

  return res;
}

Public int  
dvi_special_color_pdftex(DVI dvi, DVI_DEVICE dev, char* cmd, long x, long y)
{
  int    res;
  char  *p;

  p = &cmd[strlen("pdf: /C")];
  while (isspace((int)*p))
    p++;
  if (*p == '\0')
    return 0;

  if (strncmp(p, ">>", 2) == 0){
    res = dvi_color_pop(dvi, dev);
  } else {
    res = dvi_color_push(dvi, dev, p);
  }

  return 0;
}



static int
dvi_color_set(DVI dvi, DVI_DEVICE dev, char *color_name)
{
  DVI_COLOR col;

  if ((col = PD(dvi,color_stack)) == NULL)
    return 0;

  if (col->color_name != NULL){
    free(col->color_name);
    col->color_name = NULL;
  }
  if (color_name != NULL){
    if ((col->color_name = (char*)malloc(strlen(color_name)+1)) == NULL)
      return 0;
    strcpy(col->color_name, color_name);
    if (color_parse(dev, col, col->color_name) < 0){
      free(col->color_name);
      col->color_name = NULL;
    }
  }

  return color_set(dvi, dev, col, color_name, COLOR_WHERE_FOREGROUND);
}

static int
dvi_color_push(DVI dvi, DVI_DEVICE dev, char *color_name)
{
  DVI_COLOR col;

  if ((col = dvi_color_stack_elem_alloc(dvi, dev, color_name)) == NULL)
    return -1;
  col->next = PD(dvi,color_stack);
  PD(dvi,color_stack) = col;

  if (color_name == NULL)
    return 0;

  while (isspace((int)*color_name))
    color_name++;
  if (*color_name == '\0')
    return 0;

  return dvi_color_set(dvi, dev, color_name);
}

static int
dvi_color_pop(DVI dvi, DVI_DEVICE dev)
{
  DVI_COLOR col;

  if ((col = PD(dvi,color_stack)) == NULL){
    /* Color stack underflow. This case happens when current page is 'jumped'
     * from another page, before poping color stack items.
     * We use default color. */
    DEV_CALL(dev,message_error)
      (dev, dvi, "Color stack underflow. Text color may be incorrect.");
#if 0
    color_set(dvi, dev, NULL, NULL, COLOR_WHERE_FOREGROUND);
#endif
    return 0;
  }    

  PD(dvi,color_stack) = col->next;
  dvi_color_stack_elem_free(dvi, dev, col);
  if ((col = PD(dvi,color_stack)) != NULL){
    return color_set(dvi, dev, col, col->color_name, COLOR_WHERE_FOREGROUND);
  } else {
    if ((col = dvi_color_stack_elem_alloc(dvi, dev, NULL)) == NULL)
      return -1;
    col->model = COLOR_MODEL_RGB;
    col->c1 = col->c2 = col->c3 = -1;
    color_set(dvi, dev, col, NULL, COLOR_WHERE_FOREGROUND); 
    dvi_color_stack_elem_free(dvi, dev, col);
    return 0;
  }
}




static int  
page_color(DVI dvi, DVI_DEVICE dev, char* color_name, long x, long y)
{
  int        res;
  DVI_COLOR  col;

  if (color_name == NULL)
    return 0;
  if (*color_name == '\0')
    return 0;

  if (PD(dvi,color_page_bg[dvi->current_page-1]) == NULL){
    STRDUP(PD(dvi,color_page_bg[dvi->current_page-1]), color_name);
  }

  if ((col = dvi_color_stack_elem_alloc(dvi, dev, color_name)) == NULL)
    return -1;
  res = color_set(dvi, dev, col, color_name, COLOR_WHERE_BACKGROUND); 
  dvi_color_stack_elem_free(dvi, dev, col);

  return res;
}


Public int  
dvi_special_bgcolor(DVI dvi, DVI_DEVICE dev, char* cmd, long x, long y)
{
  char       *color_name;

  color_name = &cmd[strlen("background")];
  while (isspace((int)*color_name))
    color_name++;
  if (*color_name == '\0')
    return 0;

  return  page_color(dvi, dev, color_name, x, y);
}

Public int  
dvi_special_bgcolor_pdftex(DVI dvi, DVI_DEVICE dev, char* cmd, long x, long y)
{
  char  *color_name;

  color_name = &cmd[strlen("pdf: /BG")];
  while (isspace((int)*color_name))
    color_name++;
  if (*color_name == '\0')
    return 0;

  return  page_color(dvi, dev, color_name, x, y);

  return 0;
}







Public int
dvi_color_init(DVI dvi, DVI_DEVICE dev)
{
  DVI_COLOR   *cps_fg;
  char       **cps_bg;
  char      **cps_bgh;
  int  i;

#if 0
  printf("!! dvi_color_init\n");
#endif

  PD(dvi,color_stack)         = NULL; 
  PD(dvi,color_background)    = NULL;
  PD(dvi,color_page_stack_fg) = NULL;
  PD(dvi,color_page_bg)       = NULL;
  PD(dvi,color_page_bg_hint)  = NULL;

  if (dvi->pages > 0){
    ALLOCN_IF_ERR(PD(dvi,color_page_stack_fg), DVI_COLOR, dvi->pages){
      dvi->error = DVI_ERR_NO_MEMORY;
      return -1;
    }
    ALLOCN_IF_ERR(PD(dvi,color_page_bg), char*, dvi->pages){
      dvi->error = DVI_ERR_NO_MEMORY;
      return -1;
    }
    ALLOCN_IF_ERR(PD(dvi,color_page_bg_hint), char*, dvi->pages){
      dvi->error = DVI_ERR_NO_MEMORY;
      return -1;
    }
  }
  cps_fg  = PD(dvi,color_page_stack_fg);
  cps_bg  = PD(dvi,color_page_bg);
  cps_bgh = PD(dvi,color_page_bg_hint);
  for (i = 0; i < dvi->pages; i++){
    cps_fg[i] = NULL;
    cps_bg[i] = NULL;
    cps_bgh[i] = NULL;
  }

  return 0;
}


Public void
dvi_color_deinit(DVI dvi, DVI_DEVICE dev)
{
  DVI_COLOR  *cps_fg;
  char      **cps_bg;
  char      **cps_bgh;
  int   i;

#if 0
  printf("!! dvi_color_deinit\n");
#endif

  dvi_color_stack_free(dvi, dev);

  if (PD(dvi,color_page_stack_fg) != NULL){
    cps_fg = PD(dvi,color_page_stack_fg);
    for (i = 0; i < dvi->pages; i++){
      if (cps_fg[i] != NULL)
	dvi_color_stack_release(dvi, dev, cps_fg[i]);
      cps_fg[i] = NULL;
    }
    PD(dvi,color_page_stack_fg) = NULL;
  } 
  if (PD(dvi,color_page_bg) != NULL){
    cps_bg = PD(dvi,color_page_bg);
    for (i = 0; i < dvi->pages; i++){
      if (cps_bg[i] != NULL)
	free(cps_bg[i]);
      cps_bg[i] = NULL;
    }
    PD(dvi,color_page_bg) = NULL;
  } 
  if (PD(dvi,color_page_bg_hint) != NULL){
    cps_bgh = PD(dvi,color_page_bg_hint);
    for (i = 0; i < dvi->pages; i++){
      if (cps_bgh[i] != NULL)
	free(cps_bgh[i]);
      cps_bgh[i] = NULL;
    }
    PD(dvi,color_page_bg_hint) = NULL;
  } 
}



Public int
dvi_color_reset_page(DVI dvi, DVI_DEVICE dev, char *color_name)
{
  dvi_color_stack_free(dvi, dev);
  return 1;
}



Public void
dvi_color_stack_free(DVI dvi, DVI_DEVICE dev)
{
  dvi_color_stack_release(dvi, dev, PD(dvi,color_stack));
  PD(dvi,color_stack) = NULL;
}

Public void
dvi_color_stack_release(DVI dvi, DVI_DEVICE dev, DVI_COLOR stack)
{
  DVI_COLOR  stack_next;

  while (stack != NULL){
    stack_next = stack->next;
    dvi_color_stack_elem_free(dvi, dev, stack);
    stack = stack_next;
  }
}

Public int
dvi_color_stack_empty(DVI dvi, DVI_DEVICE dev)
{
  /* Return 1 if stack is empty.
     Return 0 if stack is non-empty. */

  if (PD(dvi,color_stack) == NULL)
    return 1;
  return 0;
}

Public DVI_COLOR
dvi_color_stack_copy(DVI dvi, DVI_DEVICE dev)
{
  return dvi_color_stack_copy2(dvi, dev, PD(dvi,color_stack));
}

static DVI_COLOR
dvi_color_stack_copy2(DVI dvi, DVI_DEVICE dev, DVI_COLOR stack)
{
  DVI_COLOR  col, cp_stack, cp_col;

  if (stack == NULL)
    return NULL;

  col = stack;
  cp_stack = cp_col = dvi_color_stack_elem_copy(dvi, dev, col);
  col = col->next;
  if (cp_stack == NULL)
    return NULL;
  while (col != NULL){
    cp_col->next = dvi_color_stack_elem_copy(dvi, dev, col);
    cp_col = cp_col->next;
    col    = col->next;
  }

#if 0
  printf("@@ Stack copy (orig): ");
  dvi_color_stack_print(dvi,dev,stack);
  printf("@@ Stack copy (copy): ");
  dvi_color_stack_print(dvi,dev,cp_stack);
#endif

  return cp_stack;
}

Public int
dvi_color_stack_resume(DVI dvi, DVI_DEVICE dev, 
		       DVI_COLOR col_stack, char *col_name_bg)
{
  struct dvi_s_color  col_bg;
  
  dvi_color_stack_free(dvi, dev);

  if (col_stack != NULL){
    PD(dvi,color_stack) = dvi_color_stack_copy2(dvi, dev, col_stack);
    do_color_set(dvi, dev, PD(dvi,color_stack), 
		 PD(dvi,color_stack)->color_name,
		 COLOR_WHERE_FOREGROUND, 1);
  } else {
    dvi_color_init(dvi, dev);
  }

  if (col_name_bg != NULL){
    if (color_parse(dev, &col_bg, col_name_bg) < 0)
      return -1;
    do_color_set(dvi, dev, &col_bg, col_name_bg, COLOR_WHERE_BACKGROUND, 1);
  }
  
  return 0;
}



#if 0
static int
dvi_color_stack_depth(DVI dvi, DVI_DEVICE dev, DVI_COLOR col)
{
  int  d;

  d = 0;
  while (col != NULL){
    col = col->next;
    d++;
  }

  return d;
}
#endif


Public void
dvi_color_stack_restore_page(DVI dvi, DVI_DEVICE dev, int page)
{
  DVI_COLOR  *cps_fg;
  char      **cps_bg;
  char      **cps_bgh;
  struct dvi_s_color   col_bg;

  if ((page < 1) || (page > dvi->pages))
    return;
  if ((cps_fg = PD(dvi,color_page_stack_fg)) == NULL)
    return;
  if ((cps_bg = PD(dvi,color_page_bg)) == NULL)
    return;
  if ((cps_bgh = PD(dvi,color_page_bg_hint)) == NULL)
    return;

#if 0
  printf("@@ Restored Stack [%d]: ", page);
  dvi_color_stack_print(dvi,dev, cps_fg[page-1]);
#endif

  dvi_color_stack_free(dvi, dev);

  if (cps_fg[page-1] != NULL){
    PD(dvi,color_stack) = dvi_color_stack_copy2(dvi, dev, cps_fg[page-1]);
    color_set(dvi, dev, PD(dvi,color_stack), 
	      PD(dvi,color_stack)->color_name, COLOR_WHERE_FOREGROUND);
  } else {
    color_set(dvi, dev, NULL, NULL, COLOR_WHERE_FOREGROUND);
  }

  if (cps_bg[page-1] != NULL){
    if (color_parse(dev, &col_bg, cps_bg[page-1]) < 0)
      return;
    color_set(dvi, dev, &col_bg, cps_bg[page-1], COLOR_WHERE_BACKGROUND);
  } else if (cps_bgh[page-1] != NULL){
    if (color_parse(dev, &col_bg, cps_bgh[page-1]) < 0)
      return;
    color_set(dvi, dev, &col_bg, cps_bgh[page-1], COLOR_WHERE_BACKGROUND);
  } else {
    color_set(dvi, dev, NULL, NULL, COLOR_WHERE_BACKGROUND);
  }
}


Public void
dvi_color_stack_save_page(DVI dvi, DVI_DEVICE dev, int page)
{
  DVI_COLOR  *cps_fg;
  char      **cps_bg;
  char      **cps_bgh;
  int         next_page;

  next_page = page+1;

#if 0
  printf(">> %d\n", next_page);
#endif

  if ((next_page < 1) || (next_page > dvi->pages))
    return;
  if ((cps_fg = PD(dvi,color_page_stack_fg)) == NULL)
    return;
  if ((cps_bg = PD(dvi,color_page_bg)) == NULL)
    return;
  if ((cps_bgh = PD(dvi,color_page_bg_hint)) == NULL)
    return;

  if (cps_fg[next_page-1] != NULL){
    dvi_color_stack_release(dvi, dev, cps_fg[next_page-1]);
    cps_fg[next_page-1] = NULL;
  }
  cps_fg[next_page-1] = dvi_color_stack_copy(dvi, dev);

  if (cps_bgh[next_page-1] != NULL){
    free(cps_bgh[next_page-1]);
    cps_bgh[next_page-1] = NULL;
  }
  if (PD(dvi,color_background) != NULL){
    STRDUP(cps_bgh[next_page-1], PD(dvi,color_background));
  }

#if 0
  printf("@@ Saved Stack [%d]: ", next_page);
  dvi_color_stack_print(dvi,dev,cps_fg[next_page-1]);
#endif
}






Public DVI_COLOR
dvi_color_stack_elem_alloc(DVI dvi, DVI_DEVICE dev, char *color_name)
{
  DVI_COLOR   col;

  ALLOC_IF_ERR(col, struct dvi_s_color)
    return NULL;
  col->model      = COLOR_MODEL_NONE;
  col->next       = NULL;
  col->c1         = -1;
  col->c2         = -1;
  col->c3         = -1;
  col->c4         = -1;
  col->cname      = NULL;
  col->color_name = NULL;

  if (color_name != NULL){
    if ((col->color_name = (char*)malloc(strlen(color_name)+1)) == NULL){
      free(col);
      return NULL;
    }
    strcpy(col->color_name, color_name);
    if (color_parse(dev, col, col->color_name) < 0){
      free(col->color_name);
      free(col);
      return NULL;
    }
  }

  return col;
}

Public void
dvi_color_stack_elem_free(DVI dvi, DVI_DEVICE dev, DVI_COLOR col)
{
  if (col != NULL){
    free_if_non_null(col->cname);
    free_if_non_null(col->color_name);
    col->cname = NULL;
    col->color_name = NULL;
    free(col);
  }
}

Public DVI_COLOR
dvi_color_stack_elem_copy(DVI dvi, DVI_DEVICE dev, DVI_COLOR col)
{
  DVI_COLOR  cp_col;

  if (col == NULL)
    return NULL;

  if ((cp_col = dvi_color_stack_elem_alloc(dvi, dev, col->color_name)) == NULL)
    return NULL;
  cp_col->next       = NULL;
  cp_col->model      = col->model;
  cp_col->c1         = col->c1;
  cp_col->c2         = col->c2;
  cp_col->c3         = col->c3;
  cp_col->c4         = col->c4;
  if (col->cname != NULL){
    if (STRDUP(cp_col->cname, col->cname) == NULL){
      free(cp_col);
      return NULL;
    }
  }
  if (col->color_name != NULL){
    if (STRDUP(cp_col->color_name, col->color_name) == NULL){
      free(cp_col->cname);
      free(cp_col);
      return NULL;
    }
  }
  
  return cp_col;
}





static int
color_parse(DVI_DEVICE dev, DVI_COLOR col, char *color_name)
{
  char          *p;
  unsigned char  r, g, b;
  int            ir, ig, ib; 

  if (color_name == NULL){
    color_name = COLOR_DEFAULT_RGB;
    if (DEV_METHOD_DEF(dev,color_rgb))
      color_name = COLOR_DEFAULT_RGB;
    else if (DEV_METHOD_DEF(dev,color_cmyk))
      color_name = COLOR_DEFAULT_CMYK;
    else if (DEV_METHOD_DEF(dev,color_gray))
      color_name = COLOR_DEFAULT_GRAY;
  }

  col->model = COLOR_MODEL_NONE;
  p = color_name;
  while (isspace((int)*p))
    p++;
  if (*p == '\0')
    return -1;
  
  if ((strncmp(p, "rgb", 3) == 0) && isspace((int)p[3])){
    col->model = COLOR_MODEL_RGB;
    p = &p[3];
    sscanf(p, "%lf%lf%lf", &col->c1, &col->c2, &col->c3);
  } else if ((strncmp(p, "cmyk", 4) == 0) && isspace((int)p[4])){
    col->model = COLOR_MODEL_CMYK;
    p = &p[4];
    sscanf(p, "%lf%lf%lf%lf", &col->c1, &col->c2, &col->c3, &col->c4);
  } else if ((strncmp(p, "gray", 4) == 0) && isspace((int)p[4])){
    col->model = COLOR_MODEL_GRAY;
    p = &p[4];
    sscanf(p, "%lf", &col->c1);
  } else if (isdigit((int)*p) && (sscanf(p, "%i%i%i", &ir, &ig, &ib) == 3)){
    col->model = COLOR_MODEL_RGB;
    col->c1 = (double)ir/255.0;
    col->c2 = (double)ig/255.0;
    col->c3 = (double)ib/255.0;
  } else {
    if (DVI_rgb_lookup(color_name, &r, &g, &b) >= 0){
      col->model = COLOR_MODEL_RGB;
      col->c1 = (double) r / 255.0;
      col->c2 = (double) g / 255.0;
      col->c3 = (double) b / 255.0;
    } else {
      if ((col->cname = (char*)malloc(strlen(color_name)+1)) == NULL)
	return -1;
      col->model = COLOR_MODEL_NAMED;    
      strcpy(col->cname, color_name);
    }
  }
  return 0;
}
 



static int
color_set(DVI dvi, DVI_DEVICE dev, DVI_COLOR col, char *color_name, int where)
{
  return do_color_set(dvi, dev, col, color_name, where, 0);
}


static int
do_color_set(DVI dvi, DVI_DEVICE dev, DVI_COLOR col, char *color_name, 
	     int where, int page_resume_flag)
{
  double  c, m, y, k, r, g, b, gs;
  
#if 0
  printf("*** COLOR CHANGE:  ");
  print_color(col);
  printf("\n");
#endif
#if 0
  if (page_resume_flag == 1){
    printf("*** COLOR CHANGE:  ");
    print_color(col);
    printf("\n");
  }
#endif

#if 0
  if (page_resume_flag == 0){
    if (where == COLOR_WHERE_FOREGROUND){
      free_if_non_null(col->color_name);
      col->color_name = NULL;
      if (color_name != NULL)
	STRDUP(col->color_name, color_name);
    } else {
      free_if_non_null(PD(dvi,color_background));
      PD(dvi,color_background) = NULL;
      if (color_name != NULL)
	STRDUP(PD(dvi,color_background), color_name);
    }
  }
#endif

  if (where == COLOR_WHERE_BACKGROUND){
    free_if_non_null(PD(dvi,color_background));
    PD(dvi,color_background) = NULL;
    if (color_name != NULL)
      STRDUP(PD(dvi,color_background), color_name);
  }

  if ((col == NULL) && (color_name == NULL)){
    if (where == COLOR_WHERE_FOREGROUND){
      if (DEV_METHOD_DEF(dev,color_rgb)){
	DEV_CALL(dev,color_rgb)(dev, dvi, page_resume_flag, -1, -1, -1);
      } else if (DEV_METHOD_DEF(dev,color_cmyk)){
	rgb2cmyk(col->c1, col->c2, col->c3, &c, &m, &y, &k);
	DEV_CALL(dev,color_cmyk)(dev, dvi, page_resume_flag, -1, -1, -1, -1);
      } else if (DEV_METHOD_DEF(dev,color_gray)){
	DEV_CALL(dev,color_gray)(dev, dvi, page_resume_flag, -1);
      }
    }
    if (where == COLOR_WHERE_BACKGROUND){
      if (DEV_METHOD_DEF(dev,bgcolor_rgb)){
	DEV_CALL(dev,bgcolor_rgb)(dev, dvi, page_resume_flag, -1, -1, -1);
      } else if (DEV_METHOD_DEF(dev,bgcolor_cmyk)){
	DEV_CALL(dev,bgcolor_cmyk)(dev, dvi, page_resume_flag, -1, -1, -1, -1);
      } else if (DEV_METHOD_DEF(dev,bgcolor_gray)){
	DEV_CALL(dev,bgcolor_gray)(dev, dvi, page_resume_flag, -1);
      }
    }
    return 0;
  }

  switch (col->model){
  case COLOR_MODEL_RGB:
    if (where == COLOR_WHERE_FOREGROUND){
      if (DEV_METHOD_DEF(dev,color_rgb)){
	DEV_CALL(dev,color_rgb)(dev, dvi, page_resume_flag, 
				col->c1, col->c2, col->c3);
      } else if (DEV_METHOD_DEF(dev,color_cmyk)){
	rgb2cmyk(col->c1, col->c2, col->c3, &c, &m, &y, &k);
	DEV_CALL(dev,color_cmyk)(dev, dvi, page_resume_flag, c, m, y, k);
      } else if (DEV_METHOD_DEF(dev,color_gray)){
	rgb2gray(col->c1, col->c2, col->c3, &gs);
	DEV_CALL(dev,color_gray)(dev, dvi, page_resume_flag, gs);
      }
    } else if (where == COLOR_WHERE_BACKGROUND){
      if (DEV_METHOD_DEF(dev,bgcolor_rgb)){
	DEV_CALL(dev,bgcolor_rgb)(dev, dvi, page_resume_flag,
				  col->c1, col->c2, col->c3);
      } else if (DEV_METHOD_DEF(dev,bgcolor_cmyk)){
	rgb2cmyk(col->c1, col->c2, col->c3, &c, &m, &y, &k);
	DEV_CALL(dev,bgcolor_cmyk)(dev, dvi, page_resume_flag, c, m, y, k);
      } else if (DEV_METHOD_DEF(dev,bgcolor_gray)){
	rgb2gray(col->c1, col->c2, col->c3, &gs);
	DEV_CALL(dev,bgcolor_gray)(dev, dvi, page_resume_flag, gs);
      }
    }
    break;
  case COLOR_MODEL_CMYK:
    if (where == COLOR_WHERE_FOREGROUND){
      if (DEV_METHOD_DEF(dev,color_cmyk)){
	DEV_CALL(dev,color_cmyk)(dev, dvi, page_resume_flag,
				 col->c1, col->c2, col->c3, col->c4);
      } else if (DEV_METHOD_DEF(dev,color_rgb)){
	cmyk2rgb(col->c1, col->c2, col->c3, col->c4, &r, &g, &b);
	DEV_CALL(dev,color_rgb)(dev, dvi, page_resume_flag, r, g, b);
      } else if (DEV_METHOD_DEF(dev,color_gray)){
	cmyk2rgb(col->c1, col->c2, col->c3, col->c4, &r, &g, &b);
	rgb2gray(r, g, b, &gs);
	DEV_CALL(dev,color_gray)(dev, dvi, page_resume_flag, gs);
      }
    } else if (where == COLOR_WHERE_BACKGROUND){
      if (DEV_METHOD_DEF(dev,bgcolor_cmyk)){
	DEV_CALL(dev,bgcolor_cmyk)(dev, dvi, page_resume_flag,
				   col->c1, col->c2, col->c3, col->c4);
      } else if (DEV_METHOD_DEF(dev,bgcolor_rgb)){
	cmyk2rgb(col->c1, col->c2, col->c3, col->c4, &r, &g, &b);
	DEV_CALL(dev,bgcolor_rgb)(dev, dvi, page_resume_flag, r, g, b);
      } else if (DEV_METHOD_DEF(dev,bgcolor_gray)){
	cmyk2rgb(col->c1, col->c2, col->c3, col->c4, &r, &g, &b);
	rgb2gray(r, g, b, &gs);
	DEV_CALL(dev,bgcolor_gray)(dev, dvi, page_resume_flag, gs);
      }
    }
    break;
  case COLOR_MODEL_GRAY:
    if (where == COLOR_WHERE_FOREGROUND){
      if (DEV_METHOD_DEF(dev,color_gray)){
	DEV_CALL(dev,color_gray)(dev, dvi, page_resume_flag, col->c1);
      } else if (DEV_METHOD_DEF(dev,color_rgb)){
	DEV_CALL(dev,color_rgb)(dev, dvi, page_resume_flag,
				col->c1, col->c1, col->c1);
      } else if (DEV_METHOD_DEF(dev,color_cmyk)){
	DEV_CALL(dev,color_cmyk)(dev, dvi, page_resume_flag, 
				 0, 0, 0, col->c1);
      }
    } else if (where == COLOR_WHERE_BACKGROUND){
      if (DEV_METHOD_DEF(dev,bgcolor_gray)){
	DEV_CALL(dev,bgcolor_gray)(dev, dvi, page_resume_flag, col->c1);
      } else if (DEV_METHOD_DEF(dev,bgcolor_rgb)){
	DEV_CALL(dev,bgcolor_rgb)(dev, dvi, page_resume_flag,
				  col->c1, col->c1, col->c1);
      } else if (DEV_METHOD_DEF(dev,bgcolor_cmyk)){
	DEV_CALL(dev,bgcolor_cmyk)(dev, dvi, page_resume_flag,
				   0, 0, 0, col->c1);
      }
    } 
    break;
  case COLOR_MODEL_NAMED:
    if (where == COLOR_WHERE_FOREGROUND){
      if (DEV_METHOD_DEF(dev,color_named)){
	DEV_CALL(dev,color_named)(dev, dvi, page_resume_flag, col->cname);
      } else {
	; /* NEED CONVERSION */
	/* Read a configuration file in a runtime library directory 
	   and convert color name to rgb, cmyk or gray values. */
      }
    } else if (where == COLOR_WHERE_BACKGROUND){
      if (DEV_METHOD_DEF(dev,bgcolor_named)){
	DEV_CALL(dev,bgcolor_named)(dev, dvi, page_resume_flag, col->cname);
      } else {
	; /* NEED CONVERSION */
	/* Read a configuration file in a runtime library directory 
	   and convert color name to rgb, cmyk or gray values. */
      }
    }
    break;
  case COLOR_MODEL_NONE:
  default:
    break;
  }

  return 0;
}


static void
rgb2cmyk(double r, double g, double b, 
	 double *c, double *m, double *y, double *k) 
{
  *c = 1.0 - r;
  *m = 1.0 - g;
  *y = 1.0 - b;
  *k = 0.0;
}

static void
cmyk2rgb(double c, double m, double y, double k, 
	 double *r, double *g, double *b) 
{
  *r = 1.0 - c;
  *g = 1.0 - m;
  *b = 1.0 - y;
}

static void
rgb2gray(double r, double g, double b, double *gs)
{
  *gs = 0.299*r + 0.587*g + 0.114*b;
}





void
dvi_color_stack_print(DVI dvi, DVI_DEVICE dev, DVI_COLOR stack)
{
  DVI_COLOR  col;

  if (stack == NULL){
    printf("[]\n"); 
    return;
  }

  printf("["); 
  for (col = stack; col != NULL; col = col->next){
    printf(" [ "); 
    print_color(col);
    printf(" (%s)", (col->color_name != NULL) ? col->color_name : "?"); 
    printf(" ]"); 
  }
  printf(" ]\n"); 
}


static void
print_color(DVI_COLOR col)
{
  switch (col->model){
  case COLOR_MODEL_RGB:
    printf("RGB %.2f %.2f %.2f", col->c1, col->c2, col->c3);
    break;
  case COLOR_MODEL_CMYK:
    printf("CYMK %.2f %.2f %.2f %.2f", col->c1, col->c2, col->c3, col->c4);
    break;
  case COLOR_MODEL_GRAY:
    printf("GRAY %.2f", col->c1);
    break;
  case COLOR_MODEL_NAMED:
    printf("NAMED %s", col->cname);
    break;
  default:
    printf("???");
    break;
  }
}


/*EOF*/
