/*  spx11  -  A Simple Previewer on X11                                */
/*  Copyright (C) 1997-1999  Hirotsugu Kakugawa. All rights reserved.  */
/*  See "COPYING" for distribution of this software.                   */
#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRING_H
#  include <string.h>
#endif
#if HAVE_STRINGS_H
#  include <strings.h>
#endif
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>
#include "libdvi29.h"
#include "defs.h"

#define SPX11_VERSION  "1.2.3" 
int              param_dpi           = DEFAULT_DPI;
char            *param_kpathsea_mode = DEFAULT_KPATHSEA_MODE;
char            *param_vflibcap      = DEFAULT_VFLIBCAP;
char            *param_paper         = "A4"; 
int              param_orient        = 0;
char            *param_foregr        = "black";
char            *param_backgr        = "white";
DVI              dvi   = NULL;
DVI_DEVICE       dev   = NULL;
DVI_FRAME_BUFFER fb    = NULL;
int              thePage, ui_cmd;
double           shrink = 5;
Display         *x_disp;
Window           x_win;
XEvent           xevent;
Cursor           x_c_reading, x_c_ready, x_c_drawing, x_c_ps; 
int              x_w, x_h, x_ofsx, x_ofsy;
GC               x_gc_win, x_gc_pix, x_gc_pix_clr;
unsigned long    x_pix_char, x_pix_back;
#define EXIT_MSG1(a1)   {fprintf(stderr,a1); fprintf(stderr,"\n"); exit(1);}
#define EXIT_MSG2(a1,a2){fprintf(stderr,a1,a2); fprintf(stderr,"\n"); exit(1);}

int  window_size(char *p, int ori, int *w, int *h)
{
  double pw, ph;

  if (DVI_parse_paper_size(p, &pw, &ph) < 0)
    return -1;
  if (w != NULL) 
    *w = ((ori==1)?ph:pw) * param_dpi / shrink;
  if (h != NULL)
    *h = ((ori==1)?pw:ph) * param_dpi / shrink;
  return 0;
}

void  x_init(char *fgcol, char *bgcol)
{
  XColor   xc_char, xc_back, xc;

  if ((x_disp = XOpenDisplay(NULL)) == NULL)
    EXIT_MSG1("Can't open X display...");
  x_c_reading = XCreateFontCursor(x_disp, XC_shuttle);
  x_c_drawing = XCreateFontCursor(x_disp, XC_watch);
  x_c_ready   = XCreateFontCursor(x_disp, XC_circle);
  x_c_ps      = XCreateFontCursor(x_disp, XC_coffee_mug);
  XAllocNamedColor(x_disp, DefaultColormap(x_disp,0), fgcol, &xc_char, &xc); 
  XAllocNamedColor(x_disp, DefaultColormap(x_disp,0), bgcol, &xc_back, &xc); 
  x_pix_char = xc_char.pixel;
  x_pix_back = xc_back.pixel;
  x_win = XCreateSimpleWindow(x_disp, RootWindow(x_disp, 0), 0, 0, x_w, x_h,
			      2, XWhitePixel(x_disp,0), x_pix_back);
  XSelectInput(x_disp, x_win, XEVENTS);
  XStoreName(x_disp, x_win, "SPX11");
  x_gc_win = XCreateGC(x_disp, x_win, 0, 0);
  XSetForeground(x_disp, x_gc_win, x_pix_char);
  XSetBackground(x_disp, x_gc_win, x_pix_back);
  XMapWindow(x_disp, x_win);
}

void  x_change_cursor(Cursor new)
{
  XDefineCursor(x_disp, x_win, new);
  XFlush(x_disp); 
}

int  x_parse_event(DVI dvi, XEvent *xev)
{
  switch (xev->type){
  case Expose:     return CMD_EXPOSE;
  case ButtonPress:
    switch (xev->xbutton.button){
    case Button1:   return (1 < thePage) ?          CMD_PREV : CMD_NONE;
    case Button3:   return (thePage < dvi->pages) ? CMD_NEXT : CMD_NONE;
    }
    break;
  case KeyPress:
    switch (XLookupKeysym(&xev->xkey, 0)){
    case XK_space:  return (thePage < dvi->pages) ? CMD_NEXT : CMD_NONE;
    case XK_Delete: return (1 < thePage) ?          CMD_PREV : CMD_NONE;
    case XK_q:      exit(0);
    }
  }
  return CMD_NONE;
}

int  main_loop(DVI_DEVICE dev, DVI dvi)
{
  XEvent  xev;
  int  last = 0;

  ui_cmd = CMD_NONE;
  for (thePage = 1; ; last = thePage){
    while (ui_cmd == CMD_NONE){
      XNextEvent(x_disp, &xevent);
      ui_cmd = x_parse_event(dvi, &xevent);
    }
    switch (ui_cmd){
    case CMD_EXPOSE: 
      while (XCheckWindowEvent(x_disp, x_win, ExposureMask, &xev) == True)
	;
      break;
    case CMD_NEXT:  thePage++;  break;
    case CMD_PREV:  thePage--;  break;
    }
    if (last != thePage){
      x_change_cursor(x_c_drawing);
      XClearWindow(x_disp, x_win);
      DVI_fb_clear(fb);
    }
    if (DVI_DRAW_PAGE(dvi, dev, thePage, shrink) != DVI_DRAW_INTERRUPTED){
      ui_cmd = CMD_NONE;
      x_change_cursor(x_c_ready);
    }
  }
}

void x_draw_fb(long x, long y, unsigned int w, unsigned int h)
{
  int      i, j;
  XImage   *img;
  static unsigned char bits[] = {
    0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01};

  if ((img = XCreateImage(x_disp, DefaultVisual(x_disp,0), 
			  DefaultDepth(x_disp,0),
			  ZPixmap, 0, NULL, w, h, 8, 0)) == NULL)
    return;
  if ((img->data = (char*)malloc(img->bytes_per_line * h)) == NULL)
    return;
  for (i = 0; i < h; i++)
    for (j = 0; j < w; j++)
      if ((fb->data[fb->raster * (i+y) + (j+x)/8] & bits[(j+x)%8]) != 0)
	XPutPixel(img, j, i, XBlackPixel(x_disp,0));
      else
	XPutPixel(img, j, i, XWhitePixel(x_disp,0));
  XPutImage(x_disp, x_win, x_gc_win, img, 0, 0, x, y, w, h);
  XDestroyImage(img);
}

void  DEV_put_bitmap(DVI_DEVICE d, DVI v, DVI_BITMAP bm, int i, 
		     long k, long cp, long x, long y)
{
  DVI_fb_put_bitmap(fb, bm, x + x_ofsx, y + x_ofsy);
  x_draw_fb(x + x_ofsx, y + x_ofsy, bm->width, bm->height);
}

void  DEV_put_rectangle(DVI_DEVICE d, DVI v, long x, long y, long w, long h)
{
  DVI_fb_put_rectangle(fb, x + x_ofsx, y + x_ofsy, w, h);
  x_draw_fb(x + x_ofsx, y + x_ofsy, w, h);
}

int   DEV_before_ps_figure(DVI_DEVICE dev, DVI dvi, char *cmd, 
			   long x, long y, long dvipos)
{
  x_change_cursor(x_c_ps);
  return 1;
}

void  DEV_after_ps_figure(DVI_DEVICE dev, DVI dvi, char *cmd, 
			  long x, long y, long dvipos)
{
  x_change_cursor(x_c_drawing);
}

int  DEV_poll(DVI_DEVICE dev, DVI dvi, int poll_type)
{
  static int  t = 0;

  if (poll_type != DVI_POLL_PAGE)
    return 0;
  if ((--t) > 0)
    return 0;
  t = DEFAULT_POLLING_INTERVAL;
  if ((XCheckWindowEvent(x_disp, x_win, XEVENTS, &xevent) == False)
      || ((ui_cmd = x_parse_event(dvi, &xevent)) == CMD_NONE))
    return 0;
  return 1;
}

void  usage(void)
{
  printf("spx11 %s - Simple Previewer on X11 (for TeX DVI files)\n",
	 SPX11_VERSION);
  printf("Usage: spx11 [PaperSize] [Options] DVI-FILE\n");
  printf("PaperSize:\n");
  printf("  -A1,..,-A7,-B1,..,-B7,-LETTER,-US,-LEGAL\n");
  printf("  -l      Landscape mode\n");
  printf("Options:  (Each value enclosed by [ ] is the default.)\n");
  printf("  -m M         Magnification (M < %d) [1.0]\n", MAG_MAX);
  printf("  -v VFLIBCAP  A path name of vflibcap [%s]\n", DEFAULT_VFLIBCAP);
  printf("  -dpi DPI     Device resolution for font selection [%d]\n", 
	 DEFAULT_DPI);
  printf("  -mode MODE   Device mode name for kpathsea [%s]\n", 
	 DEFAULT_KPATHSEA_MODE);
  printf("  -cx          Same as `-dpi 300 -mode cx'\n");
  printf("  -sparcptr    Same as `-dpi 400 -mode sparcptr'\n");
  printf("  -ljfour      Same as `-dpi 600 -mode ljfour'\n");
  printf("  -fg COLOR    Foreground color [black]\n");
  printf("  -bg COLOR    Background color [white]\n");
  printf("Key and Mouse operation:\n");
  printf("  Mouse Right Button    Next page\n");
  printf("  Mouse Left Button     Previous page\n");
  printf("  q                     Quit\n");
  exit(0);
}

char  *parse_args(int argc, char **argv)
{
  char     *dvi_file = NULL;
  double   mag;
  int      junk;

  mag = 1.0;
  for (--argc, argv++; argc > 0;  argc--, argv++){
    if ((strcmp(*argv, "-m") == 0) && (argc > 1)){
      argc--; argv++;
      mag = atof(*argv);
    } else if (strcmp(*argv, "-l") == 0){
      param_orient = 1;
    } else if ((strcmp(*argv, "-v") == 0)  && (argc > 1)){
      argc--;  argv++;
      param_vflibcap = *argv;
    } else if ((strcmp(*argv, "-dpi") == 0) && (argc > 1)){
      argc--;  argv++;
      param_dpi = atoi(*argv);
    } else if ((strcmp(*argv, "-mode") == 0) && (argc > 1)){
      argc--;  argv++;
      param_kpathsea_mode = *argv;
    } else if (strcmp(*argv, "-cx") == 0){
      param_dpi = 300; param_kpathsea_mode = "cx";
    } else if (strcmp(*argv, "-sparcptr") == 0){
      param_dpi = 400; param_kpathsea_mode = "sparcptr";
    } else if (strcmp(*argv, "-ljfour") == 0){
      param_dpi = 600; param_kpathsea_mode = "ljfour";
    } else if ((strcmp(*argv, "-fg") == 0) && (argc > 1)){
      argc--;  argv++;
      param_foregr = *argv;
    } else if ((strcmp(*argv, "-bg") == 0) && (argc > 1)){
      argc--;  argv++;
      param_backgr = *argv;
    } else if (**argv != '-'){
      dvi_file = *argv;
    } else if (window_size(&argv[0][1], param_orient, &junk, &junk) >= 0){
      param_paper = &argv[0][1];
    } else if ((strcmp(*argv, "-h") == 0) || (strcmp(*argv, "--help") == 0)){
      usage();
    } else 
      EXIT_MSG2("Unknow option: %s\n", *argv);
  }
  if (mag < MAG_MIN)
    EXIT_MSG1("Maginification is too small.");
  if (MAG_MAX < mag)
    EXIT_MSG1("Maginification is too large.");
  shrink = (DEFAULT_SHRINK * param_dpi) / (mag * DEFAULT_DPI);
  if (window_size(param_paper, param_orient, &x_w, &x_h) < 0)
    EXIT_MSG2("Unknown paper size: %s", param_paper);
  x_ofsx = DEFAULT_OFFSET_X * param_dpi / shrink;
  x_ofsy = DEFAULT_OFFSET_Y * param_dpi / shrink;
  return dvi_file;
}

DVI_DEVICE  create_dev(void)
{
  if ((dev = DVI_DEVICE_ALLOC()) == NULL)
    return NULL;
  dev->h_dpi = dev->v_dpi   = param_dpi;
  dev->device_polling       = DEV_poll;
  dev->put_rectangle        = DEV_put_rectangle;
  dev->put_bitmap           = DEV_put_bitmap;
  dev->message_advice       = NULL;
  dev->before_ps_figure     = DEV_before_ps_figure;
  dev->after_ps_figure      = DEV_after_ps_figure;
  return dev;
}

int  main(int argc, char **argv)
{
  char  params[3*256], *dvi_file;
  DVI_PROPERTY  dvi_props;

  DVI_setup();
  if ((dvi_file = parse_args(argc, argv)) == NULL)
    EXIT_MSG1("No DVI file.");
  x_init(param_foregr, param_backgr);
  x_change_cursor(x_c_reading);
  sprintf(params, "TeX_DPI=%d, TeX_KPATHSEA_MODE=%s, TeX_KPATHSEA_PROGRAM=%s",
	  param_dpi, param_kpathsea_mode, "xgdvi");
  if ((DVI_INIT(param_vflibcap, params) < 0) || (create_dev() == NULL))
    exit(1);
  if ((fb = DVI_fb_create((long)x_w, (long)x_h)) == NULL)
    EXIT_MSG1("No memory.");   
  DVI_fb_clear(fb);
  dvi_props = DVI_PROPERTY_ALLOC_DEFAULT();
  DVI_PROPERTY_SET(dvi_props, DVI_PROP_ASYNC_GS_INVOCATION);
  DVI_PROPERTY_SET(dvi_props, DVI_PROP_INCREMENTAL_EPS_DISPLAY);
  if ((dvi = DVI_CREATE(dev, dvi_file, dvi_props)) == NULL)
    EXIT_MSG2("Can't open DVI file: %s", dvi_file);
  if (DVI_OPEN_FONT(dvi, dev) < 0)
    EXIT_MSG1("Cannot find all fonts.");
  main_loop(dev, dvi);
  return 0;
}
