/***************************************************************************
 *   copyright           : (C) 2003 by Hendrik Sattler                     *
 *   mail                : pingos@hendrik-sattler.de                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "UrlSelectPanel.H"

enum {
  COMBO_PROTO = 1,
  BTN_PATH
};

BEGIN_EVENT_TABLE(UrlSelectPanel,wxPanel)
  EVT_COMBOBOX(COMBO_PROTO, UrlSelectPanel::onProtocol)
END_EVENT_TABLE()

UrlSelectPanel::UrlSelectPanel (wxWindow* parent, int id, UrlFile& defaultUrl,
				const wxString& pathLabel)
  : wxPanel(parent,id),
    protocolDescr(_("protocol:")),
    serverDescr(_("server:")),
    pathDescr(pathLabel)
{
  this->protocols.Add(wxString(_T("file")));
  this->protocols.Add(wxString(_T("ftp")));
  this->protocols.Add(wxString(_T("http")));
}

bool UrlSelectPanel::Enable (bool enable) {
  bool retval = true;
  retval &= this->protocol->Enable(enable);
  retval &= this->serverText->Enable(enable && this->protocol->GetSelection()!=0);
  retval &= this->server->Enable(enable && this->protocol->GetSelection()!=0);
  retval &= this->path->Enable(enable);
  retval &= this->pathBtn->Enable(enable && this->protocol->GetSelection()==0);
  retval &= wxPanel::Enable(enable);
  return retval;
}

bool UrlSelectPanel::allFieldsValid () {
  switch (this->protocol->GetSelection()) {
  case -1:
    return false;
    break;
  default:
    if (this->server->GetValue() == _T("")) {
      return false;
    }
    //no break
  case 0:
    if (this->path->GetValue() == _T("")) {
      return false;
    }
    break;    
  }
  return true;
}


wxSizer* UrlSelectPanel::getPanelSizer (UrlFile& defaultUrl) {
  wxBoxSizer* retval = new wxBoxSizer(wxVERTICAL);
  retval->Add(getCorePanelSizer(defaultUrl),1,wxEXPAND|wxALL,5);
  return (wxSizer*)retval;
}

wxSizer* UrlSelectPanel::getCorePanelSizer (UrlFile& defaultUrl) {
  wxFlexGridSizer* retval = new wxFlexGridSizer(3,3,10,10);

  //first row
  retval->Add(new wxStaticText(this,-1,this->protocolDescr),
	      0,wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL);
  this->protocol = getProtocolBox(defaultUrl.protocol);
  this->protocol->SetToolTip(_("Choose the protocol you want to use to access the data files"));
  retval->Add(this->protocol,0,wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL);
  retval->Add(0,0);

  //second row
  this->serverText = new wxStaticText(this,-1,this->serverDescr);
  retval->Add(this->serverText,0,wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL);
  this->server = new wxTextCtrl(this,-1,defaultUrl.server);
  this->server->SetToolTip(_("Enter the server address here (i.e. www.myserver.de or 127.0.0.1)"));
  retval->Add(this->server,1,wxEXPAND|wxALIGN_CENTRE_VERTICAL);
  retval->Add(0,0);

  //third row
  retval->Add(new wxStaticText(this,-1,this->pathDescr),
	      0,wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL);
  this->path = new wxTextCtrl(this,-1,defaultUrl.getFullPath(true));
  this->path->SetToolTip(_("Enter the relative path to the data files here (i.e. ./data/ )"));
  retval->Add(this->path,1,wxEXPAND|wxALIGN_CENTRE_VERTICAL);
  this->pathBtn = new wxButton(this,BTN_PATH,_("Select"),
			       wxDefaultPosition,wxDefaultSize,
			       wxBU_EXACTFIT);
  retval->Add(this->pathBtn,0,wxALIGN_CENTRE_VERTICAL);

  //only set the new minSize once, the others will adopt to it
//   int width = 0;
//   int height = 0;
//   this->server->GetSize(&width,&height);
//   this->server->SetSize(3*width,height);

  retval->AddGrowableCol(1);
  Enable(true);

  return (wxSizer*)retval;
}

#include "commonElements.H"

wxComboBox* UrlSelectPanel::getProtocolBox (const wxString& defaultUrl) {
  unsigned int selIndex = 0;
  for (unsigned int i=0; i<this->protocols.GetCount(); ++i) {
    if (defaultUrl == protocols[i]) {
      selIndex = i;
      break;
    }
  }
  wxComboBox* retval = new wxComboBox(this,COMBO_PROTO,_T(""),
				      wxDefaultPosition,wxDefaultSize,
				      this->protocols.GetCount(),
				      common::toStringPtr(this->protocols),
				      wxCB_DROPDOWN|wxCB_READONLY);
  retval->SetSelection(selIndex);
  return retval;
}


void UrlSelectPanel::onProtocol (wxCommandEvent& WXUNUSED(e)) {
  Enable(true);
}

BEGIN_EVENT_TABLE(FileUrlSelectPanel,UrlSelectPanel)
  EVT_BUTTON(BTN_PATH, FileUrlSelectPanel::onPathBtn)
END_EVENT_TABLE()

FileUrlSelectPanel::FileUrlSelectPanel (wxWindow* parent, int id, UrlFile defaultUrl)
  : UrlSelectPanel (parent,id,defaultUrl,_("file path:"))
{
}

#include <wx/filename.h>

bool FileUrlSelectPanel::allFieldsValid () {
  switch (this->protocol->GetSelection()) {
  case -1:
    return false;
    break;
  default:
    if (this->server->GetValue() == _T("")) {
      return false;
    }
    //no break
  case 0:
    if (this->path->GetValue() == _T("")) {
      return false;
    }
    break;    
  }
  return true;
}

wxString FileUrlSelectPanel::getURLstr () {
  if (allFieldsValid()) {
    UrlFile u;
    u.protocol=this->protocol->GetStringSelection();
    u.server=this->server->GetValue();
#ifdef __WXMSW__
    u.setFullPath(this->path->GetValue(),true);
#else
    u.setFullPath(wxString(_T("/")) + this->path->GetValue(),true);
#endif
    return u.getURLstr();
  } else {
    return _T("");
  }
}

#include  <wx/file.h>

void FileUrlSelectPanel::onPathBtn (wxCommandEvent& WXUNUSED(event)) {
  wxFileDialog dialog(this);
  if (this->path->GetValue().Length() > 0) {
    dialog.SetPath(this->path->GetValue());
    dialog.SetStyle(wxOPEN|wxFILE_MUST_EXIST);
  }
  int r;
  do { //loop to work-around wxWindows bug
    r = dialog.ShowModal();
  } while (r == wxID_OK && !wxFile::Exists(dialog.GetPath()));

  if (r == wxID_OK) {
    this->path->SetValue(dialog.GetPath());
  }
}

BinaryFileUrlSelectPanel::BinaryFileUrlSelectPanel (wxWindow* parent, int id,
						    UrlFile defaultUrl)
  : FileUrlSelectPanel(parent,id,defaultUrl)
{
  this->panelSizer = getPanelSizer(defaultUrl);
  
  SetSizer(panelSizer);
  panelSizer->SetSizeHints(this);
}

TextFileUrlSelectPanel::TextFileUrlSelectPanel (wxWindow* parent, int id,
						UrlFile defaultUrl)
  : FileUrlSelectPanel(parent,id,defaultUrl)
{
  this->charset = getCharsetBox(defaultUrl.getCharset());

  wxSizer* s  = getCorePanelSizer(defaultUrl);
  s->Add(new wxStaticText(this,-1,_("Character set")),
		  0,wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL);
  s->Add(this->charset,0,wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL);

  this->panelSizer = new wxBoxSizer(wxVERTICAL);
  panelSizer->Add(s,1,wxEXPAND|wxALL,5);
  
  SetSizer(panelSizer);
  panelSizer->SetSizeHints(this);
}

wxComboBox* TextFileUrlSelectPanel::getCharsetBox (const wxString& charset) {
  this->charsets.Add(wxString(_T("")));
  if (charset.Len() &&
      charset!=wxString(_T("UTF-8"))) {
    this->charsets.Add(charset);
  }
  this->charsets.Add(wxString(_T("UTF-8")));
  wxComboBox* retval = new wxComboBox(this,COMBO_PROTO,_T(""),
				      wxDefaultPosition,wxDefaultSize,
				      this->charsets.GetCount(),
				      common::toStringPtr(this->charsets),
				      wxCB_DROPDOWN);
  retval->SetSelection(1);
  return retval;
}


BEGIN_EVENT_TABLE(DirUrlSelectPanel,UrlSelectPanel)
  EVT_BUTTON(BTN_PATH, DirUrlSelectPanel::onPathBtn)
END_EVENT_TABLE()

DirUrlSelectPanel::DirUrlSelectPanel (wxWindow* parent, int id, UrlFile defaultUrl)
  : UrlSelectPanel (parent,id,defaultUrl,_("directory:"))
{
  this->panelSizer = getPanelSizer(defaultUrl);

  SetSizer(panelSizer);
  panelSizer->SetSizeHints(this);
}

bool DirUrlSelectPanel::allFieldsValid () {
  switch (this->protocol->GetSelection()) {
  case -1:
    return false;
    break;
  default:
    if (this->server->GetValue() == _T("")) {
      return false;
    }
    //no break
  case 0:
    break;    
  }
  return true;
}

wxString DirUrlSelectPanel::getURLstr () {
  if (allFieldsValid()) {
    UrlFile u;
    u.protocol=this->protocol->GetStringSelection();
    u.server=this->server->GetValue();
    u.setFullPath(this->path->GetValue() + wxString(_T("/")),true);
    return u.getURLstr();
  } else {
    return _T("");
  }
}

void DirUrlSelectPanel::onPathBtn (wxCommandEvent& WXUNUSED(event)) {
  wxDirDialog dialog(this);
  if (this->path->GetValue().Length() > 0) {
    dialog.SetPath(this->path->GetValue());
  }
  if (dialog.ShowModal() == wxID_OK) {
    this->path->SetValue(UrlFile::toUrlType(dialog.GetPath()));
  }
}
