#   Copyright (C) 1987-2005 by Jeffery P. Hansen
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Last edit by hansen on Thu Feb  5 14:43:33 2004
#

set undo_type undo

#############################################################################
#
# Use this for a basic action.  "name" is the name of the action that will
# appear in the undo/redo listboxes, and "body" is the undoable code to 
# execute.
#
proc action {name body} {
  gat_obBeginFrame $name
  namespace eval :: $body
  gat_obEndFrame
}

#############################################################################
#
# Use this for a action which will be appended to the most recent action.
#
proc continueAction {name body} {
  gat_obAppendFrame $name
  namespace eval :: $body
  gat_obEndFrame
}

#############################################################################
#
# Use this for "transparent" actions.  Transparent actions are those
# for which a frame record is kept and placed in the stack, but which
# are not directly visible to the user.
#
proc transAction {name body} {
  gat_obBeginFrame $name 1
  namespace eval :: $body
  gat_obEndFrame
}

#############################################################################
#
# Use this action for operations for which undo processing will
# be suspended.
#
proc suspendAction {body} {
  set m [gat_obMode]
  gat_obMode 0
  namespace eval :: $body
  gat_obMode $m
}

#############################################################################
#
# Use this action for operations for which undo processing will
# be suspended, and all undo data cleared.
#
proc clearAction {body} {
  set m [gat_obMode]
  gat_obMode 0
  gat_obFlush
  namespace eval :: $body
  gat_obMode $m
}

#############################################################################
#
# Post box for undo/redo selection
#
proc tkg_postURBox {w} {
  cancel_help

  toplevel .undo -cursor arrow

  scan [winfo rootx $w] "%d" X
  scan [winfo pointery .] "%d" Y
  set Y [expr $Y - 10]
  wm geometry .undo +$X+$Y
  wm transient .undo .
  wm overrideredirect .undo 1

  listbox .undo.lb -bd 2 -relief sunken -yscrollcommand ".undo.vb set" -bg white -width 10 -height 5 -selectmode multiple
  scrollbar .undo.vb -command ".undo.lb yview"
  pack .undo.lb .undo.vb -side left -fill both
}

#############################################################################
#
# Called as a result of a mouse button release while the undo/redo listbox
# is active.  Causes the listbox to be deleted and the selected undo/redo
# actions to be executed.
#
proc tkg_finishURBox {} {
  global undo_type

  set n -1

  if { [.undo.lb size] > 0 } {
    set S [.undo.lb curselection]
    foreach j $S {
      if { $j > $n } {set n $j }
    }
  }

  destroy .undo
  if { $n >= 0 } {
    gat_$undo_type [expr $n + 1]
  }
}

#############################################################################
#
# Handles pointer motion within the undo/redo listbox.
#
proc tkg_motionURBox {y} {
  set n [.undo.lb size]
  set i [.undo.lb nearest $y]
  .undo.lb selection set 0 $i
  if { [expr $i + 1] < $n } {
    .undo.lb selection clear [expr $i + 1] end
  }
}

#############################################################################
#
# Waits for the undo/redo listbox to be deleted.
#
proc tkg_waitURBox {} {
  bind .undo <ButtonRelease-1> tkg_finishURBox
  bind .undo.lb <B1-Motion> { tkg_motionURBox %y }
  bind .undo.lb <Leave> {.undo.lb selection clear 0 end }

  tkwait visibility .undo
  .undo.lb selection clear 0 end
  .undo.lb selection set 0
  grab set .undo
  tkwait window .undo
  grab release .undo
}

#############################################################################
#
# Called as a result of hitting the "undo" button.  Causes the list of undo
# actions to be displayed in the listbox.
#
proc tkg_undoSelect {} {
  global undo_type

  set i 0
  set undo_type undo
  tkg_postURBox .tbar.du.back
  foreach u [gat_getUndoList] {
    .undo.lb insert end $u
  }
  tkg_waitURBox
}

#############################################################################
#
# Called as a result of hitting the "redo" button.  Causes the list of redo
# actions to be displayed in the listbox.
#
proc tkg_redoSelect {} {
  global undo_type

  set i 0
  set undo_type redo
  tkg_postURBox .tbar.du.forward
  foreach u [gat_getRedoList] {
    .undo.lb insert end $u
  }
  tkg_waitURBox
}

#############################################################################
#############################################################################
##
## Tkgate actions are defined below here.  All calls from the interface
## go through an action function.
##
#############################################################################
#############################################################################


#############################################################################
#
# Show debugging info about wires.
#
proc act_dumpWires {} {
  gat_dumpWires
}

#############################################################################
#
# Run the internal datat structure verifier.
#
proc act_verify {} {
  gat_verify
}

#############################################################################
#
# Undo last action
#
proc act_undo {} {
  gat_undo
}

#############################################################################
#
# Redo last undone action
#
proc act_redo {} {
  gat_redo
}

#############################################################################
#
# Add an input to the selected gate.
#
proc act_addIn {} {
  action AddIn { gat_addIn }
}

#############################################################################
#
# Add an output to the selected gate.
#
proc act_addOut {} {
  action AddOut { gat_addOut }
}

#############################################################################
#
# Add a tristate port to the selected gate.
#
proc act_addInOut {} {
  action AddInOut { gat_addInOut }
}

#############################################################################
#
# Change the type of a port
#
proc act_editPort {} {
  action EditPort { gat_changePinDir }
}


#############################################################################
#
# Set debugging mode
#
proc act_debugMode {} {
  action Mode { tkg_setMode 12 }
}

#############################################################################
#
# Put tkgate in standard edit mode
#
proc act_editMode {} {
  action Mode { tkg_setMode 1 }
}

#############################################################################
#
# Put tkgate in delete gate mode
#
proc act_deleteMode {} {
  action Mode {tkg_setMode 3 }
}

#############################################################################
#
# Put tkgate in cut wire mode
#
proc act_cutMode {} { 
  action Mode {tkg_setMode 0 }
}

#############################################################################
#
# Put tkgate in invert in/out mode
#
proc act_invertMode {} { 
  action Mode {tkg_setMode 2 }
}

#############################################################################
#
# Put tkgate in set wire size mode
#
proc act_sizeMode {} { 
  action Mode {tkg_setMode 9 }
}

#############################################################################
#
# Set gate rotation to 0 degrees
#
proc act_rot0 {} { 
  set rot 0
  action Rotation { gat_setrot $rot }
}

#############################################################################
#
# Set gate rotation to 90 degrees
#
proc act_rot90 {} { 
  set rot 1
  action Rotation { gat_setrot $rot }
}

#############################################################################
#
# Set gate rotation to 180 degrees
#
proc act_rot180 {} { 
  set rot 2
  action Rotation { gat_setrot $rot }
}

#############################################################################
#
# Set gate rotation to 270 degrees
#
proc act_rot270 {} { 
  set rot 3
  action Rotation { gat_setrot $rot }
}

#############################################################################
#
# Open the selected module.
#
proc act_openMod {} {
  action Open { if { "%W" != ".scope"} { gat_openBox } else { .scope.main.frame.canvas zoom 1 } }
}

#############################################################################
#
# Close the current module.
#
proc act_closeMod {} {
  action Close { if { "%W" != ".scope"} { gat_closeBox } else { .scope.main.frame.canvas zoom -1 } }
}

######################################################################
#
# Replicate a gate.
#
proc act_replicate {} {
  action Replicate { gat_replicate } 
}

######################################################################
#
# Delete the selected object.
#
proc act_delete {} {
  action Delete { gat_deleteSelected }
}

######################################################################
#
# Edit the properties of the selected object.
#
proc act_editProps {} {
  action EditProps { gat_editProps }
}

######################################################################
#
# Set/unset a probe on the selected wire.
#
proc act_toggleProbe {} {
  gat_toggleProbe
}

######################################################################
#
# Move to the previous error in the error box.
#
proc act_errBoxUp {} {
  action ErrUp { tkg_errBoxUp } 
}

######################################################################
#
# Move to the next error in the error box.
#
proc act_errBoxDown {} { 
  action ErrDown { tkg_errBoxDown }
}

######################################################################
#
# Align gates horizontally
#
proc act_hAlign {} {
  action HAlign { gat_align 1 }
}

######################################################################
#
# Align gates vertically
#
proc act_vAlign {} {
  action VAlign { gat_align 0 }
}

######################################################################
#
# Clear the current circuit and start editing a new file.
#
proc act_newFile {} {
  clearAction tkg_new
}

######################################################################
#
# Load the specified file
#
proc act_loadFile {} {
  clearAction tkg_load
}

######################################################################
#
# Load modules from a library
#
proc act_loadLibrary {} {
  clearAction tkg_loadLibrary
}

######################################################################
#
# Save the current circuit to the current file
#
proc act_saveFile {} {
  suspendAction tkg_saveCurrent
}

######################################################################
#
# Save the current circuit to a new file.
#
proc act_saveAsFile {} {
  clearAction tkg_save
}

######################################################################
#
# Print the current circuit
#
proc act_print {} {
  suspendAction tkg_printDlg
}

######################################################################
#
# Exit tkgate
#
proc act_exit {} {
  tkg_exit
}

######################################################################
#
# Edit tkgate options
#
proc act_editOptions {} {
  suspendAction tkg_editOptions
}

######################################################################
#
# Copy selected region to cut buffer
#
proc act_copyToBuf {} { 
  action Copy { gat_copyToBuf }
}

######################################################################
#
# Select all gates in the current module
#
proc act_selectAll {} { 
  action SelectAll { gat_selectAll }
}

######################################################################
#
# Edit circuit properties
#
proc act_editCProps {} {
  action EditCProps { tkg_editCircProp }
}

######################################################################
#
# Set an anchor on the selected gate
#
proc act_anchor {} { 
  action Anchor { gat_anchor 1 }
}

######################################################################
#
# Remove an anchor from the current gate.
#
proc act_unAnchor {} {
  action UnAnchor { gat_anchor 0 }
}

######################################################################
#
# Begin critical path analysis mode
#
proc act_cpathAnal {} {
  tkg_cpathAnal
}

######################################################################
#
# Cut the selected objects and place them in the cut buffer
#
proc act_cutToBuf {} {
  action Cut gat_cutToBuf
}

######################################################################
#
# Yank a copy of the cut buffer to the current module
#
proc act_yankFromBuf {} {
  action Yank gat_yankFromBuf
}

######################################################################
#
# Find an object
#
proc act_findObject {} {
  transAction Find { tkg_findObject }
}

######################################################################
#
# Start the simulator
#
proc act_startSimulator {} {
  gat_setMajorMode simulate
}

######################################################################
#
# End the simulator
#
proc act_endSimulator {} {
  gat_setMajorMode edit
}

######################################################################
#
# Put simulator in "run" mode
#
proc act_simRun {} {
  tkg_simRun
}

######################################################################
#
# Put simulator in "stop" mode.
#
proc act_simStop {} {
  tkg_simStop
}

######################################################################
#
# Edit simulator breakpoints
#
proc act_editBreakpoints {} {
  tkg_editBreakpoints
}

######################################################################
#
# Execute a simulation script
#
proc act_doSimScript {} {
  tkg_doSimScript
}

######################################################################
#
# Load a memory file
#
proc act_simLoadMem {} {
  tkg_simLoadMem
}

######################################################################
#
# Dump a memory file.
#
proc act_simDumpMem {} {
  tkg_simDumpMem
}

######################################################################
#
# Print a scope trace
#
proc act_scopePrint {} {
  tkg_scopePrintDlg
}

######################################################################
#
# Step the simulator one epoch
#
proc act_simStep {} {
  tkg_simStep
}

######################################################################
#
# Step the simulator one "clock cycle".
#
proc act_simCycle {} {
  tkg_simCycle
}

######################################################################
#
# Refresh the screen.
#
proc act_refreshScreen {} {
  gat_refreshScreen
}

######################################################################
#
# Create a new module
#
proc act_blockNew {} {
  action NewMod tkg_blockNew
}

######################################################################
#
# Delete a module
#
proc act_blockDelete {} {
  action DeleteMod tkg_blockDelete
}

######################################################################
# 
# Copy a module
#
proc act_blockCopy {} {
  action CopyMod tkg_blockCopy
}

######################################################################
#
# Rename a module
#
proc act_blockRename {} {
  action RenameMod tkg_blockRename
}

######################################################################
#
# Set the interface of the selected module
#
proc act_setBlockDesc {} {
  action SetInterf { gat_setBlockDesc }
}

######################################################################
#
# Edit the module interfaces
#
proc act_editBlockDesc {} {
  action EditInterf gat_editBlockDesc
}

######################################################################
#
# "Claim" a block (make it a non-library block)
#
proc act_blockClaim {} {
  action Claim tkg_blockClaim
}

######################################################################
#
# Zoom in
#
proc act_zoomIn {} {
  action ZoomIn { gat_zoom 1 }
}

######################################################################
#
# Zoom out
#
proc act_zoomOut {} {
  action ZoomOut { gat_zoom -1 }
}

######################################################################
#
# Nudge the selected objects left
#
proc act_nudgeLeft {} {
  action NudgeL { gat_moveGate -1 0 }
}

######################################################################
#
# Nudge the selected objects right
#
proc act_nudgeRight {} {
  action NudgeR { gat_moveGate 1 0 }
}

######################################################################
#
# Nudge the selected objects up
#
proc act_nudgeUp {} {
  action NudgeU { gat_moveGate 0 -1 }
}

######################################################################
#
# Nudge the selected objects down
#
proc act_nudgeDown {} {
  action NudgeD { gat_moveGate 0 1 }
}

######################################################################
#
# Show the about message
#
proc act_showAbout {} {
  showAbout
}

######################################################################
#
# Show the license message
#
proc act_showLicense {} {
  showLicense
}

######################################################################
#
# Show the documentation message
#
proc act_showDocumentation {} {
  showDocumentation
}

######################################################################
#
# Load example circuit 1
#
proc act_loadExample1 {} { global tkgate_exampledir; gat_load $tkgate_exampledir/ex1/combinational.v }

######################################################################
#
# Load example circuit 2
#
proc act_loadExample2 {} { global tkgate_exampledir; gat_load $tkgate_exampledir/ex2/flipflop.v }

######################################################################
#
# Load example circuit 3
#
proc act_loadExample3 {} { global tkgate_exampledir; gat_load $tkgate_exampledir/ex3/counter.v }

######################################################################
#
# Load example circuit 4
#
proc act_loadExample4 {} { global tkgate_exampledir; gat_load $tkgate_exampledir/ex4/trff.v }

######################################################################
#
# Load example circuit 5
#
proc act_loadExample5 {} { global tkgate_exampledir; gat_load $tkgate_exampledir/ex5/menagerie.v }

######################################################################
#
# Load tutorial circuit 1
#
proc act_loadTutorial1 {} { global tkgate_tutorialdir; gat_load $tkgate_tutorialdir/welcome_tut.v }

######################################################################
#
# Load tutorial circuit 2
#
proc act_loadTutorial2 {} { global tkgate_tutorialdir; gat_load $tkgate_tutorialdir/create_tut.v }

######################################################################
#
# Load tutorial circuit 3
#
proc act_loadTutorial3 {} { global tkgate_tutorialdir; gat_load $tkgate_tutorialdir/edit1_tut.v }

######################################################################
#
# Load tutorial circuit 4
#
proc act_loadTutorial4 {} { global tkgate_tutorialdir; gat_load $tkgate_tutorialdir/edit2_tut.v }

######################################################################
#
# Load tutorial circuit 5
#
proc act_loadTutorial5 {} { global tkgate_tutorialdir; gat_load $tkgate_tutorialdir/module_tut.v }

######################################################################
#
# Load tutorial circuit 6
#
proc act_loadTutorial6 {} { global tkgate_tutorialdir; gat_load $tkgate_tutorialdir/sim_tut.v }

######################################################################
#
# Edit wire properties through popup menu
#
proc act_popupWireProps {} { continueAction WireProp gat_popupWireProps }

######################################################################
#
# Add a stub to a wire
#
proc act_popupWireAddStub {} { action AddStub gat_popupWireAddStub }

######################################################################
#
# Change the anchor status of an object
#
proc act_popupAnchor {} { action Anchor { global pop_anchor; gat_anchor $pop_anchor } }

######################################################################
#
# Change the port type of a module port
#
proc act_popupBlockChgPort {} { action ChgPort { gat_popupBlockIo change } }

######################################################################
#
# Add an input port to a module
#
proc act_popupBlockAddIn {} { action AddInPort { gat_popupBlockIo in } }

######################################################################
#
# Add an output port to a module
#
proc act_popupBlockAddOut {} { action AddOutPort { gat_popupBlockIo out } }

######################################################################
#
# Add an in/out port to a module
#
proc act_popupBlockAddInOut {} { action AddTriPort { gat_popupBlockIo inout } }
