#include "tiger.h"
/* vim:ts=4:sw=4:noet
 * (tabspace=4)
 * 
 * Copyright (C) 2004, 2005 Walter Doekes, <walter@djcvt.net>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * The greater part of this source comes from the creators of the Tiger Hash
 * algorithm, Ross Anderson and Eli Biham.
 */

#include "endian.h"
#include <string.h>

/* Tiger S boxes */
static uint64_t tiger_sboxes[4 * 256];

/* The following macro denotes that an optimization    */
/* for Alpha is required. It is used only for          */
/* optimization of time. Otherwise it does nothing.    */
#ifdef __alpha
#	define OPTIMIZE_FOR_ALPHA
#endif

/* NOTE that this code is NOT FULLY OPTIMIZED for any  */
/* machine. Assembly code might be much faster on some */
/* machines, especially if the code is compiled with   */
/* gcc.                                                */

/* The number of passes of the hash function.          */
/* Three passes are recommended.                       */
/* Use four passes when you need extra security.       */
/* Must be at least three.                             */
#define PASSES 3


#define t1 (tiger_sboxes)
#define t2 (tiger_sboxes + 256)
#define t3 (tiger_sboxes + 256 * 2)
#define t4 (tiger_sboxes + 256 * 3)

#define save_abc \
	aa = a; \
	bb = b; \
	cc = c;

#ifdef OPTIMIZE_FOR_ALPHA
/* This is the official definition of round */
#define round(a,b,c,x,mul) \
	c ^= x; \
	a -= t1[((c)>>(0*8))&0xFF] ^ t2[((c)>>(2*8))&0xFF] ^ \
		t3[((c)>>(4*8))&0xFF] ^ t4[((c)>>(6*8))&0xFF]; \
	b += t4[((c)>>(1*8))&0xFF] ^ t3[((c)>>(3*8))&0xFF] ^ \
		t2[((c)>>(5*8))&0xFF] ^ t1[((c)>>(7*8))&0xFF]; \
	b *= mul;
#else
/* This code works faster when compiled on 32-bit machines */
/* (but works slower on Alpha) */
#define round(a,b,c,x,mul) \
	c ^= x; \
	a -= t1[(uint8_t)(c)] ^ \
		t2[(uint8_t)(((uint32_t)(c))>>(2*8))] ^ \
		t3[(uint8_t)((c)>>(4*8))] ^ \
		t4[(uint8_t)(((uint32_t)((c)>>(4*8)))>>(2*8))]; \
	b += t4[(uint8_t)(((uint32_t)(c))>>(1*8))] ^ \
		t3[(uint8_t)(((uint32_t)(c))>>(3*8))] ^ \
		t2[(uint8_t)(((uint32_t)((c)>>(4*8)))>>(1*8))] ^ \
		t1[(uint8_t)(((uint32_t)((c)>>(4*8)))>>(3*8))]; \
	b *= mul;
#endif

#define pass(a,b,c,mul) \
	round(a,b,c,x0,mul) \
	round(b,c,a,x1,mul) \
	round(c,a,b,x2,mul) \
	round(a,b,c,x3,mul) \
	round(b,c,a,x4,mul) \
	round(c,a,b,x5,mul) \
	round(a,b,c,x6,mul) \
	round(b,c,a,x7,mul)

#define key_schedule \
	x0 -= x7 ^ 0xA5A5A5A5A5A5A5A5ull; \
	x1 ^= x0; \
	x2 += x1; \
	x3 -= x2 ^ ((~x1)<<19); \
	x4 ^= x3; \
	x5 += x4; \
	x6 -= x5 ^ ((~x4)>>23); \
	x7 ^= x6; \
	x0 += x7; \
	x1 -= x0 ^ ((~x7)<<19); \
	x2 ^= x1; \
	x3 += x2; \
	x4 -= x3 ^ ((~x2)>>23); \
	x5 ^= x4; \
	x6 += x5; \
	x7 -= x6 ^ 0x0123456789ABCDEFull;

#define feedforward \
	a ^= aa; \
	b -= bb; \
	c += cc;

#ifdef OPTIMIZE_FOR_ALPHA
/* The loop is unrolled: works better on Alpha */
#define compress \
	save_abc \
	pass(a,b,c,5) \
	key_schedule \
	pass(c,a,b,7) \
	key_schedule \
	pass(b,c,a,9) \
	for(pass_no=3; pass_no<PASSES; pass_no++) { \
		key_schedule \
		pass(a,b,c,9) \
		tmpa = a; a = c; c = b; b = tmpa; \
	} \
	feedforward
#else
/* loop: works better on PC and Sun (smaller cache?) */
#define compress \
	save_abc \
	for(pass_no=0; pass_no<PASSES; pass_no++) { \
		if(pass_no != 0) {key_schedule} \
			pass(a,b,c,(pass_no==0?5:pass_no==1?7:9)); \
		tmpa = a; a = c; c = b; b = tmpa; \
	} \
	feedforward
#endif

#define tiger_compress_macro(str, state) { \
	register uint64_t a, b, c, tmpa; \
	uint64_t aa, bb, cc; \
	register uint64_t x0, x1, x2, x3, x4, x5, x6, x7; \
	int pass_no; \
	\
	a = state[0]; \
	b = state[1]; \
	c = state[2]; \
	\
	x0 = str[0]; x1 = str[1]; x2 = str[2]; x3 = str[3]; \
	x4 = str[4]; x5 = str[5]; x6 = str[6]; x7 = str[7]; \
	\
	compress \
	\
	state[0] = a; \
	state[1] = b; \
	state[2] = c; \
}

/* The compress function is a function. Requires smaller cache?    */
static void tiger_compress(const uint64_t* str, uint64_t* state) {
	tiger_compress_macro(((const uint64_t*)str), ((uint64_t*)state));
}

#ifdef OPTIMIZE_FOR_ALPHA
/* The compress function is inlined: works better on Alpha.        */
/* Still leaves the function above in the code, in case some other */
/* module calls it directly.                                       */
#define tiger_compress(str, state) \
	tiger_compress_macro(((const uint64_t*)str), ((uint64_t*)state))
#endif

void tiger(const uint64_t* str, uint64_t length, uint64_t* res) {
	register uint64_t i, j;
	uint8_t temp[64];

	res[0]=0x0123456789ABCDEFull;
	res[1]=0xFEDCBA9876543210ull;
	res[2]=0xF096A5B4C3B2E187ull;

	for(i = length; i >= 64; i -= 64) {
#ifdef BIG_ENDIAN
		for(j = 0; j < 64; j++)
			temp[j ^ 7] = ((const uint8_t*)str)[j];
		tiger_compress(((const uint64_t*)temp), res);
#else /* !BIG_ENDIAN */
		tiger_compress(str, res);
#endif /* !BIG_ENDIAN */
		str += 8;
	}

#ifdef BIG_ENDIAN
	for(j = 0; j < i; j++)
	    temp[j ^ 7] = ((const uint8_t*)str)[j];

	temp[j ^ 7] = 0x01;
	j++;
	for(; j & 7; j++)
		temp[j ^ 7] = 0;
#else /* !BIG_ENDIAN */
	for(j = 0; j < i; j++)
		temp[j] = ((const uint8_t*)str)[j];

	temp[j++] = 0x01;
	for(; j & 7; j++)
		temp[j] = 0;
#endif /* !BIG_ENDIAN */
	
	if(j > 56) {
		for(; j < 64; j++)
			temp[j] = 0;
		tiger_compress(((uint64_t*)temp), res);
		j = 0;
	}

	for(; j < 56; j++)
		temp[j] = 0;
	((uint64_t*)(&(temp[56])))[0] = ((uint64_t)length) << 3;
	tiger_compress(((uint64_t*)temp), res);
}

/* The tiger_bp function does the same as the tiger function,
 * except that it prepends the specified byte to the input.
 * The THEX algorithm requires this prepended byte.
 * This function is here to prevent useless copying of large
 * blocks of data, since one should be able to drop a read-only
 * block of a (let's say memory mapped) file here. */
void tiger_bp(uint8_t prepend, const uint64_t* str, uint64_t length,
		uint64_t* res) {
	register uint64_t i, j;
	uint8_t temp[64];
	uint8_t bpbuf[64];
	int bpdone = 0;

	res[0]=0x0123456789ABCDEFull;
	res[1]=0xFEDCBA9876543210ull;
	res[2]=0xF096A5B4C3B2E187ull;

	++length;
	bpbuf[0] = prepend;

	i = length;
	if(i >= 64) {
		bpdone = 1;
		memcpy(bpbuf + 1, (const uint8_t*)str, 63);
#ifdef BIG_ENDIAN
		for(j = 0; j < 64; j++)
			temp[j ^ 7] = ((uint8_t*)bpbuf)[j];
		tiger_compress(((uint64_t*)temp), res);
#else /* !BIG_ENDIAN */
		tiger_compress((uint64_t*)bpbuf, res);
#endif /* !BIG_ENDIAN */
		str = (const uint64_t*)((const uint8_t*)str + (7 * 8 + 7));
		i -= 64;
	}

	for(; i >= 64; i -= 64) {
#ifdef BIG_ENDIAN
		for(j = 0; j < 64; j++)
			temp[j ^ 7] = ((const uint8_t*)str)[j];
		tiger_compress(((uint64_t*)temp), res);
#else /* !BIG_ENDIAN */
		tiger_compress(str, res);
#endif /* !BIG_ENDIAN */
		str += 8;
	}

	if(!bpdone) {
		bpdone = 1;
		memcpy(bpbuf + 1, (const uint8_t*)str, length - 1);
#ifdef BIG_ENDIAN
		for(j = 0; j < i; j++)
		    temp[j ^ 7] = ((uint8_t*)bpbuf)[j];

		temp[j++ ^ 7] = 0x01;
		for(; j & 7; j++)
			temp[j ^ 7] = 0;
#else /* !BIG_ENDIAN */
		for(j = 0; j < i; j++)
			temp[j] = ((uint8_t*)bpbuf)[j];

		temp[j++] = 0x01;
		for(; j & 7; j++)
			temp[j] = 0;
#endif /* !BIG_ENDIAN */
	} else {
#ifdef BIG_ENDIAN
		for(j = 0; j < i; j++)
		    temp[j ^ 7] = ((const uint8_t*)str)[j];

		temp[j++ ^ 7] = 0x01;
		for(; j & 7; j++)
			temp[j ^ 7] = 0;
#else /* !BIG_ENDIAN */
		for(j = 0; j < i; j++)
			temp[j] = ((const uint8_t*)str)[j];

		temp[j++] = 0x01;
		for(; j & 7; j++)
			temp[j] = 0;
#endif /* !BIG_ENDIAN */
	}
	
	if(j > 56) {
		for(; j < 64; j++)
			temp[j] = 0;
		tiger_compress(((uint64_t*)temp), res);
		j = 0;
	}

	for(; j < 56; j++)
		temp[j] = 0;
	((uint64_t*)(&(temp[56])))[0] = ((uint64_t)length) << 3;
	tiger_compress(((uint64_t*)temp), res);
}

static uint64_t tiger_sboxes[4 * 256] = {
	0x02AAB17CF7E90C5Eull  /*    0 */,	0xAC424B03E243A8ECull  /*    1 */,
	0x72CD5BE30DD5FCD3ull  /*    2 */,	0x6D019B93F6F97F3Aull  /*    3 */,
	0xCD9978FFD21F9193ull  /*    4 */,	0x7573A1C9708029E2ull  /*    5 */,
	0xB164326B922A83C3ull  /*    6 */,	0x46883EEE04915870ull  /*    7 */,
	0xEAACE3057103ECE6ull  /*    8 */,	0xC54169B808A3535Cull  /*    9 */,
	0x4CE754918DDEC47Cull  /*   10 */,	0x0AA2F4DFDC0DF40Cull  /*   11 */,
	0x10B76F18A74DBEFAull  /*   12 */,	0xC6CCB6235AD1AB6Aull  /*   13 */,
	0x13726121572FE2FFull  /*   14 */,	0x1A488C6F199D921Eull  /*   15 */,
	0x4BC9F9F4DA0007CAull  /*   16 */,	0x26F5E6F6E85241C7ull  /*   17 */,
	0x859079DBEA5947B6ull  /*   18 */,	0x4F1885C5C99E8C92ull  /*   19 */,
	0xD78E761EA96F864Bull  /*   20 */,	0x8E36428C52B5C17Dull  /*   21 */,
	0x69CF6827373063C1ull  /*   22 */,	0xB607C93D9BB4C56Eull  /*   23 */,
	0x7D820E760E76B5EAull  /*   24 */,	0x645C9CC6F07FDC42ull  /*   25 */,
	0xBF38A078243342E0ull  /*   26 */,	0x5F6B343C9D2E7D04ull  /*   27 */,
	0xF2C28AEB600B0EC6ull  /*   28 */,	0x6C0ED85F7254BCACull  /*   29 */,
	0x71592281A4DB4FE5ull  /*   30 */,	0x1967FA69CE0FED9Full  /*   31 */,
	0xFD5293F8B96545DBull  /*   32 */,	0xC879E9D7F2A7600Bull  /*   33 */,
	0x860248920193194Eull  /*   34 */,	0xA4F9533B2D9CC0B3ull  /*   35 */,
	0x9053836C15957613ull  /*   36 */,	0xDB6DCF8AFC357BF1ull  /*   37 */,
	0x18BEEA7A7A370F57ull  /*   38 */,	0x037117CA50B99066ull  /*   39 */,
	0x6AB30A9774424A35ull  /*   40 */,	0xF4E92F02E325249Bull  /*   41 */,
	0x7739DB07061CCAE1ull  /*   42 */,	0xD8F3B49CECA42A05ull  /*   43 */,
	0xBD56BE3F51382F73ull  /*   44 */,	0x45FAED5843B0BB28ull  /*   45 */,
	0x1C813D5C11BF1F83ull  /*   46 */,	0x8AF0E4B6D75FA169ull  /*   47 */,
	0x33EE18A487AD9999ull  /*   48 */,	0x3C26E8EAB1C94410ull  /*   49 */,
	0xB510102BC0A822F9ull  /*   50 */,	0x141EEF310CE6123Bull  /*   51 */,
	0xFC65B90059DDB154ull  /*   52 */,	0xE0158640C5E0E607ull  /*   53 */,
	0x884E079826C3A3CFull  /*   54 */,	0x930D0D9523C535FDull  /*   55 */,
	0x35638D754E9A2B00ull  /*   56 */,	0x4085FCCF40469DD5ull  /*   57 */,
	0xC4B17AD28BE23A4Cull  /*   58 */,	0xCAB2F0FC6A3E6A2Eull  /*   59 */,
	0x2860971A6B943FCDull  /*   60 */,	0x3DDE6EE212E30446ull  /*   61 */,
	0x6222F32AE01765AEull  /*   62 */,	0x5D550BB5478308FEull  /*   63 */,
	0xA9EFA98DA0EDA22Aull  /*   64 */,	0xC351A71686C40DA7ull  /*   65 */,
	0x1105586D9C867C84ull  /*   66 */,	0xDCFFEE85FDA22853ull  /*   67 */,
	0xCCFBD0262C5EEF76ull  /*   68 */,	0xBAF294CB8990D201ull  /*   69 */,
	0xE69464F52AFAD975ull  /*   70 */,	0x94B013AFDF133E14ull  /*   71 */,
	0x06A7D1A32823C958ull  /*   72 */,	0x6F95FE5130F61119ull  /*   73 */,
	0xD92AB34E462C06C0ull  /*   74 */,	0xED7BDE33887C71D2ull  /*   75 */,
	0x79746D6E6518393Eull  /*   76 */,	0x5BA419385D713329ull  /*   77 */,
	0x7C1BA6B948A97564ull  /*   78 */,	0x31987C197BFDAC67ull  /*   79 */,
	0xDE6C23C44B053D02ull  /*   80 */,	0x581C49FED002D64Dull  /*   81 */,
	0xDD474D6338261571ull  /*   82 */,	0xAA4546C3E473D062ull  /*   83 */,
	0x928FCE349455F860ull  /*   84 */,	0x48161BBACAAB94D9ull  /*   85 */,
	0x63912430770E6F68ull  /*   86 */,	0x6EC8A5E602C6641Cull  /*   87 */,
	0x87282515337DDD2Bull  /*   88 */,	0x2CDA6B42034B701Bull  /*   89 */,
	0xB03D37C181CB096Dull  /*   90 */,	0xE108438266C71C6Full  /*   91 */,
	0x2B3180C7EB51B255ull  /*   92 */,	0xDF92B82F96C08BBCull  /*   93 */,
	0x5C68C8C0A632F3BAull  /*   94 */,	0x5504CC861C3D0556ull  /*   95 */,
	0xABBFA4E55FB26B8Full  /*   96 */,	0x41848B0AB3BACEB4ull  /*   97 */,
	0xB334A273AA445D32ull  /*   98 */,	0xBCA696F0A85AD881ull  /*   99 */,
	0x24F6EC65B528D56Cull  /*  100 */,	0x0CE1512E90F4524Aull  /*  101 */,
	0x4E9DD79D5506D35Aull  /*  102 */,	0x258905FAC6CE9779ull  /*  103 */,
	0x2019295B3E109B33ull  /*  104 */,	0xF8A9478B73A054CCull  /*  105 */,
	0x2924F2F934417EB0ull  /*  106 */,	0x3993357D536D1BC4ull  /*  107 */,
	0x38A81AC21DB6FF8Bull  /*  108 */,	0x47C4FBF17D6016BFull  /*  109 */,
	0x1E0FAADD7667E3F5ull  /*  110 */,	0x7ABCFF62938BEB96ull  /*  111 */,
	0xA78DAD948FC179C9ull  /*  112 */,	0x8F1F98B72911E50Dull  /*  113 */,
	0x61E48EAE27121A91ull  /*  114 */,	0x4D62F7AD31859808ull  /*  115 */,
	0xECEBA345EF5CEAEBull  /*  116 */,	0xF5CEB25EBC9684CEull  /*  117 */,
	0xF633E20CB7F76221ull  /*  118 */,	0xA32CDF06AB8293E4ull  /*  119 */,
	0x985A202CA5EE2CA4ull  /*  120 */,	0xCF0B8447CC8A8FB1ull  /*  121 */,
	0x9F765244979859A3ull  /*  122 */,	0xA8D516B1A1240017ull  /*  123 */,
	0x0BD7BA3EBB5DC726ull  /*  124 */,	0xE54BCA55B86ADB39ull  /*  125 */,
	0x1D7A3AFD6C478063ull  /*  126 */,	0x519EC608E7669EDDull  /*  127 */,
	0x0E5715A2D149AA23ull  /*  128 */,	0x177D4571848FF194ull  /*  129 */,
	0xEEB55F3241014C22ull  /*  130 */,	0x0F5E5CA13A6E2EC2ull  /*  131 */,
	0x8029927B75F5C361ull  /*  132 */,	0xAD139FABC3D6E436ull  /*  133 */,
	0x0D5DF1A94CCF402Full  /*  134 */,	0x3E8BD948BEA5DFC8ull  /*  135 */,
	0xA5A0D357BD3FF77Eull  /*  136 */,	0xA2D12E251F74F645ull  /*  137 */,
	0x66FD9E525E81A082ull  /*  138 */,	0x2E0C90CE7F687A49ull  /*  139 */,
	0xC2E8BCBEBA973BC5ull  /*  140 */,	0x000001BCE509745Full  /*  141 */,
	0x423777BBE6DAB3D6ull  /*  142 */,	0xD1661C7EAEF06EB5ull  /*  143 */,
	0xA1781F354DAACFD8ull  /*  144 */,	0x2D11284A2B16AFFCull  /*  145 */,
	0xF1FC4F67FA891D1Full  /*  146 */,	0x73ECC25DCB920ADAull  /*  147 */,
	0xAE610C22C2A12651ull  /*  148 */,	0x96E0A810D356B78Aull  /*  149 */,
	0x5A9A381F2FE7870Full  /*  150 */,	0xD5AD62EDE94E5530ull  /*  151 */,
	0xD225E5E8368D1427ull  /*  152 */,	0x65977B70C7AF4631ull  /*  153 */,
	0x99F889B2DE39D74Full  /*  154 */,	0x233F30BF54E1D143ull  /*  155 */,
	0x9A9675D3D9A63C97ull  /*  156 */,	0x5470554FF334F9A8ull  /*  157 */,
	0x166ACB744A4F5688ull  /*  158 */,	0x70C74CAAB2E4AEADull  /*  159 */,
	0xF0D091646F294D12ull  /*  160 */,	0x57B82A89684031D1ull  /*  161 */,
	0xEFD95A5A61BE0B6Bull  /*  162 */,	0x2FBD12E969F2F29Aull  /*  163 */,
	0x9BD37013FEFF9FE8ull  /*  164 */,	0x3F9B0404D6085A06ull  /*  165 */,
	0x4940C1F3166CFE15ull  /*  166 */,	0x09542C4DCDF3DEFBull  /*  167 */,
	0xB4C5218385CD5CE3ull  /*  168 */,	0xC935B7DC4462A641ull  /*  169 */,
	0x3417F8A68ED3B63Full  /*  170 */,	0xB80959295B215B40ull  /*  171 */,
	0xF99CDAEF3B8C8572ull  /*  172 */,	0x018C0614F8FCB95Dull  /*  173 */,
	0x1B14ACCD1A3ACDF3ull  /*  174 */,	0x84D471F200BB732Dull  /*  175 */,
	0xC1A3110E95E8DA16ull  /*  176 */,	0x430A7220BF1A82B8ull  /*  177 */,
	0xB77E090D39DF210Eull  /*  178 */,	0x5EF4BD9F3CD05E9Dull  /*  179 */,
	0x9D4FF6DA7E57A444ull  /*  180 */,	0xDA1D60E183D4A5F8ull  /*  181 */,
	0xB287C38417998E47ull  /*  182 */,	0xFE3EDC121BB31886ull  /*  183 */,
	0xC7FE3CCC980CCBEFull  /*  184 */,	0xE46FB590189BFD03ull  /*  185 */,
	0x3732FD469A4C57DCull  /*  186 */,	0x7EF700A07CF1AD65ull  /*  187 */,
	0x59C64468A31D8859ull  /*  188 */,	0x762FB0B4D45B61F6ull  /*  189 */,
	0x155BAED099047718ull  /*  190 */,	0x68755E4C3D50BAA6ull  /*  191 */,
	0xE9214E7F22D8B4DFull  /*  192 */,	0x2ADDBF532EAC95F4ull  /*  193 */,
	0x32AE3909B4BD0109ull  /*  194 */,	0x834DF537B08E3450ull  /*  195 */,
	0xFA209DA84220728Dull  /*  196 */,	0x9E691D9B9EFE23F7ull  /*  197 */,
	0x0446D288C4AE8D7Full  /*  198 */,	0x7B4CC524E169785Bull  /*  199 */,
	0x21D87F0135CA1385ull  /*  200 */,	0xCEBB400F137B8AA5ull  /*  201 */,
	0x272E2B66580796BEull  /*  202 */,	0x3612264125C2B0DEull  /*  203 */,
	0x057702BDAD1EFBB2ull  /*  204 */,	0xD4BABB8EACF84BE9ull  /*  205 */,
	0x91583139641BC67Bull  /*  206 */,	0x8BDC2DE08036E024ull  /*  207 */,
	0x603C8156F49F68EDull  /*  208 */,	0xF7D236F7DBEF5111ull  /*  209 */,
	0x9727C4598AD21E80ull  /*  210 */,	0xA08A0896670A5FD7ull  /*  211 */,
	0xCB4A8F4309EBA9CBull  /*  212 */,	0x81AF564B0F7036A1ull  /*  213 */,
	0xC0B99AA778199ABDull  /*  214 */,	0x959F1EC83FC8E952ull  /*  215 */,
	0x8C505077794A81B9ull  /*  216 */,	0x3ACAAF8F056338F0ull  /*  217 */,
	0x07B43F50627A6778ull  /*  218 */,	0x4A44AB49F5ECCC77ull  /*  219 */,
	0x3BC3D6E4B679EE98ull  /*  220 */,	0x9CC0D4D1CF14108Cull  /*  221 */,
	0x4406C00B206BC8A0ull  /*  222 */,	0x82A18854C8D72D89ull  /*  223 */,
	0x67E366B35C3C432Cull  /*  224 */,	0xB923DD61102B37F2ull  /*  225 */,
	0x56AB2779D884271Dull  /*  226 */,	0xBE83E1B0FF1525AFull  /*  227 */,
	0xFB7C65D4217E49A9ull  /*  228 */,	0x6BDBE0E76D48E7D4ull  /*  229 */,
	0x08DF828745D9179Eull  /*  230 */,	0x22EA6A9ADD53BD34ull  /*  231 */,
	0xE36E141C5622200Aull  /*  232 */,	0x7F805D1B8CB750EEull  /*  233 */,
	0xAFE5C7A59F58E837ull  /*  234 */,	0xE27F996A4FB1C23Cull  /*  235 */,
	0xD3867DFB0775F0D0ull  /*  236 */,	0xD0E673DE6E88891Aull  /*  237 */,
	0x123AEB9EAFB86C25ull  /*  238 */,	0x30F1D5D5C145B895ull  /*  239 */,
	0xBB434A2DEE7269E7ull  /*  240 */,	0x78CB67ECF931FA38ull  /*  241 */,
	0xF33B0372323BBF9Cull  /*  242 */,	0x52D66336FB279C74ull  /*  243 */,
	0x505F33AC0AFB4EAAull  /*  244 */,	0xE8A5CD99A2CCE187ull  /*  245 */,
	0x534974801E2D30BBull  /*  246 */,	0x8D2D5711D5876D90ull  /*  247 */,
	0x1F1A412891BC038Eull  /*  248 */,	0xD6E2E71D82E56648ull  /*  249 */,
	0x74036C3A497732B7ull  /*  250 */,	0x89B67ED96361F5ABull  /*  251 */,
	0xFFED95D8F1EA02A2ull  /*  252 */,	0xE72B3BD61464D43Dull  /*  253 */,
	0xA6300F170BDC4820ull  /*  254 */,	0xEBC18760ED78A77Aull  /*  255 */,
	0xE6A6BE5A05A12138ull  /*  256 */,	0xB5A122A5B4F87C98ull  /*  257 */,
	0x563C6089140B6990ull  /*  258 */,	0x4C46CB2E391F5DD5ull  /*  259 */,
	0xD932ADDBC9B79434ull  /*  260 */,	0x08EA70E42015AFF5ull  /*  261 */,
	0xD765A6673E478CF1ull  /*  262 */,	0xC4FB757EAB278D99ull  /*  263 */,
	0xDF11C6862D6E0692ull  /*  264 */,	0xDDEB84F10D7F3B16ull  /*  265 */,
	0x6F2EF604A665EA04ull  /*  266 */,	0x4A8E0F0FF0E0DFB3ull  /*  267 */,
	0xA5EDEEF83DBCBA51ull  /*  268 */,	0xFC4F0A2A0EA4371Eull  /*  269 */,
	0xE83E1DA85CB38429ull  /*  270 */,	0xDC8FF882BA1B1CE2ull  /*  271 */,
	0xCD45505E8353E80Dull  /*  272 */,	0x18D19A00D4DB0717ull  /*  273 */,
	0x34A0CFEDA5F38101ull  /*  274 */,	0x0BE77E518887CAF2ull  /*  275 */,
	0x1E341438B3C45136ull  /*  276 */,	0xE05797F49089CCF9ull  /*  277 */,
	0xFFD23F9DF2591D14ull  /*  278 */,	0x543DDA228595C5CDull  /*  279 */,
	0x661F81FD99052A33ull  /*  280 */,	0x8736E641DB0F7B76ull  /*  281 */,
	0x15227725418E5307ull  /*  282 */,	0xE25F7F46162EB2FAull  /*  283 */,
	0x48A8B2126C13D9FEull  /*  284 */,	0xAFDC541792E76EEAull  /*  285 */,
	0x03D912BFC6D1898Full  /*  286 */,	0x31B1AAFA1B83F51Bull  /*  287 */,
	0xF1AC2796E42AB7D9ull  /*  288 */,	0x40A3A7D7FCD2EBACull  /*  289 */,
	0x1056136D0AFBBCC5ull  /*  290 */,	0x7889E1DD9A6D0C85ull  /*  291 */,
	0xD33525782A7974AAull  /*  292 */,	0xA7E25D09078AC09Bull  /*  293 */,
	0xBD4138B3EAC6EDD0ull  /*  294 */,	0x920ABFBE71EB9E70ull  /*  295 */,
	0xA2A5D0F54FC2625Cull  /*  296 */,	0xC054E36B0B1290A3ull  /*  297 */,
	0xF6DD59FF62FE932Bull  /*  298 */,	0x3537354511A8AC7Dull  /*  299 */,
	0xCA845E9172FADCD4ull  /*  300 */,	0x84F82B60329D20DCull  /*  301 */,
	0x79C62CE1CD672F18ull  /*  302 */,	0x8B09A2ADD124642Cull  /*  303 */,
	0xD0C1E96A19D9E726ull  /*  304 */,	0x5A786A9B4BA9500Cull  /*  305 */,
	0x0E020336634C43F3ull  /*  306 */,	0xC17B474AEB66D822ull  /*  307 */,
	0x6A731AE3EC9BAAC2ull  /*  308 */,	0x8226667AE0840258ull  /*  309 */,
	0x67D4567691CAECA5ull  /*  310 */,	0x1D94155C4875ADB5ull  /*  311 */,
	0x6D00FD985B813FDFull  /*  312 */,	0x51286EFCB774CD06ull  /*  313 */,
	0x5E8834471FA744AFull  /*  314 */,	0xF72CA0AEE761AE2Eull  /*  315 */,
	0xBE40E4CDAEE8E09Aull  /*  316 */,	0xE9970BBB5118F665ull  /*  317 */,
	0x726E4BEB33DF1964ull  /*  318 */,	0x703B000729199762ull  /*  319 */,
	0x4631D816F5EF30A7ull  /*  320 */,	0xB880B5B51504A6BEull  /*  321 */,
	0x641793C37ED84B6Cull  /*  322 */,	0x7B21ED77F6E97D96ull  /*  323 */,
	0x776306312EF96B73ull  /*  324 */,	0xAE528948E86FF3F4ull  /*  325 */,
	0x53DBD7F286A3F8F8ull  /*  326 */,	0x16CADCE74CFC1063ull  /*  327 */,
	0x005C19BDFA52C6DDull  /*  328 */,	0x68868F5D64D46AD3ull  /*  329 */,
	0x3A9D512CCF1E186Aull  /*  330 */,	0x367E62C2385660AEull  /*  331 */,
	0xE359E7EA77DCB1D7ull  /*  332 */,	0x526C0773749ABE6Eull  /*  333 */,
	0x735AE5F9D09F734Bull  /*  334 */,	0x493FC7CC8A558BA8ull  /*  335 */,
	0xB0B9C1533041AB45ull  /*  336 */,	0x321958BA470A59BDull  /*  337 */,
	0x852DB00B5F46C393ull  /*  338 */,	0x91209B2BD336B0E5ull  /*  339 */,
	0x6E604F7D659EF19Full  /*  340 */,	0xB99A8AE2782CCB24ull  /*  341 */,
	0xCCF52AB6C814C4C7ull  /*  342 */,	0x4727D9AFBE11727Bull  /*  343 */,
	0x7E950D0C0121B34Dull  /*  344 */,	0x756F435670AD471Full  /*  345 */,
	0xF5ADD442615A6849ull  /*  346 */,	0x4E87E09980B9957Aull  /*  347 */,
	0x2ACFA1DF50AEE355ull  /*  348 */,	0xD898263AFD2FD556ull  /*  349 */,
	0xC8F4924DD80C8FD6ull  /*  350 */,	0xCF99CA3D754A173Aull  /*  351 */,
	0xFE477BACAF91BF3Cull  /*  352 */,	0xED5371F6D690C12Dull  /*  353 */,
	0x831A5C285E687094ull  /*  354 */,	0xC5D3C90A3708A0A4ull  /*  355 */,
	0x0F7F903717D06580ull  /*  356 */,	0x19F9BB13B8FDF27Full  /*  357 */,
	0xB1BD6F1B4D502843ull  /*  358 */,	0x1C761BA38FFF4012ull  /*  359 */,
	0x0D1530C4E2E21F3Bull  /*  360 */,	0x8943CE69A7372C8Aull  /*  361 */,
	0xE5184E11FEB5CE66ull  /*  362 */,	0x618BDB80BD736621ull  /*  363 */,
	0x7D29BAD68B574D0Bull  /*  364 */,	0x81BB613E25E6FE5Bull  /*  365 */,
	0x071C9C10BC07913Full  /*  366 */,	0xC7BEEB7909AC2D97ull  /*  367 */,
	0xC3E58D353BC5D757ull  /*  368 */,	0xEB017892F38F61E8ull  /*  369 */,
	0xD4EFFB9C9B1CC21Aull  /*  370 */,	0x99727D26F494F7ABull  /*  371 */,
	0xA3E063A2956B3E03ull  /*  372 */,	0x9D4A8B9A4AA09C30ull  /*  373 */,
	0x3F6AB7D500090FB4ull  /*  374 */,	0x9CC0F2A057268AC0ull  /*  375 */,
	0x3DEE9D2DEDBF42D1ull  /*  376 */,	0x330F49C87960A972ull  /*  377 */,
	0xC6B2720287421B41ull  /*  378 */,	0x0AC59EC07C00369Cull  /*  379 */,
	0xEF4EAC49CB353425ull  /*  380 */,	0xF450244EEF0129D8ull  /*  381 */,
	0x8ACC46E5CAF4DEB6ull  /*  382 */,	0x2FFEAB63989263F7ull  /*  383 */,
	0x8F7CB9FE5D7A4578ull  /*  384 */,	0x5BD8F7644E634635ull  /*  385 */,
	0x427A7315BF2DC900ull  /*  386 */,	0x17D0C4AA2125261Cull  /*  387 */,
	0x3992486C93518E50ull  /*  388 */,	0xB4CBFEE0A2D7D4C3ull  /*  389 */,
	0x7C75D6202C5DDD8Dull  /*  390 */,	0xDBC295D8E35B6C61ull  /*  391 */,
	0x60B369D302032B19ull  /*  392 */,	0xCE42685FDCE44132ull  /*  393 */,
	0x06F3DDB9DDF65610ull  /*  394 */,	0x8EA4D21DB5E148F0ull  /*  395 */,
	0x20B0FCE62FCD496Full  /*  396 */,	0x2C1B912358B0EE31ull  /*  397 */,
	0xB28317B818F5A308ull  /*  398 */,	0xA89C1E189CA6D2CFull  /*  399 */,
	0x0C6B18576AAADBC8ull  /*  400 */,	0xB65DEAA91299FAE3ull  /*  401 */,
	0xFB2B794B7F1027E7ull  /*  402 */,	0x04E4317F443B5BEBull  /*  403 */,
	0x4B852D325939D0A6ull  /*  404 */,	0xD5AE6BEEFB207FFCull  /*  405 */,
	0x309682B281C7D374ull  /*  406 */,	0xBAE309A194C3B475ull  /*  407 */,
	0x8CC3F97B13B49F05ull  /*  408 */,	0x98A9422FF8293967ull  /*  409 */,
	0x244B16B01076FF7Cull  /*  410 */,	0xF8BF571C663D67EEull  /*  411 */,
	0x1F0D6758EEE30DA1ull  /*  412 */,	0xC9B611D97ADEB9B7ull  /*  413 */,
	0xB7AFD5887B6C57A2ull  /*  414 */,	0x6290AE846B984FE1ull  /*  415 */,
	0x94DF4CDEACC1A5FDull  /*  416 */,	0x058A5BD1C5483AFFull  /*  417 */,
	0x63166CC142BA3C37ull  /*  418 */,	0x8DB8526EB2F76F40ull  /*  419 */,
	0xE10880036F0D6D4Eull  /*  420 */,	0x9E0523C9971D311Dull  /*  421 */,
	0x45EC2824CC7CD691ull  /*  422 */,	0x575B8359E62382C9ull  /*  423 */,
	0xFA9E400DC4889995ull  /*  424 */,	0xD1823ECB45721568ull  /*  425 */,
	0xDAFD983B8206082Full  /*  426 */,	0xAA7D29082386A8CBull  /*  427 */,
	0x269FCD4403B87588ull  /*  428 */,	0x1B91F5F728BDD1E0ull  /*  429 */,
	0xE4669F39040201F6ull  /*  430 */,	0x7A1D7C218CF04ADEull  /*  431 */,
	0x65623C29D79CE5CEull  /*  432 */,	0x2368449096C00BB1ull  /*  433 */,
	0xAB9BF1879DA503BAull  /*  434 */,	0xBC23ECB1A458058Eull  /*  435 */,
	0x9A58DF01BB401ECCull  /*  436 */,	0xA070E868A85F143Dull  /*  437 */,
	0x4FF188307DF2239Eull  /*  438 */,	0x14D565B41A641183ull  /*  439 */,
	0xEE13337452701602ull  /*  440 */,	0x950E3DCF3F285E09ull  /*  441 */,
	0x59930254B9C80953ull  /*  442 */,	0x3BF299408930DA6Dull  /*  443 */,
	0xA955943F53691387ull  /*  444 */,	0xA15EDECAA9CB8784ull  /*  445 */,
	0x29142127352BE9A0ull  /*  446 */,	0x76F0371FFF4E7AFBull  /*  447 */,
	0x0239F450274F2228ull  /*  448 */,	0xBB073AF01D5E868Bull  /*  449 */,
	0xBFC80571C10E96C1ull  /*  450 */,	0xD267088568222E23ull  /*  451 */,
	0x9671A3D48E80B5B0ull  /*  452 */,	0x55B5D38AE193BB81ull  /*  453 */,
	0x693AE2D0A18B04B8ull  /*  454 */,	0x5C48B4ECADD5335Full  /*  455 */,
	0xFD743B194916A1CAull  /*  456 */,	0x2577018134BE98C4ull  /*  457 */,
	0xE77987E83C54A4ADull  /*  458 */,	0x28E11014DA33E1B9ull  /*  459 */,
	0x270CC59E226AA213ull  /*  460 */,	0x71495F756D1A5F60ull  /*  461 */,
	0x9BE853FB60AFEF77ull  /*  462 */,	0xADC786A7F7443DBFull  /*  463 */,
	0x0904456173B29A82ull  /*  464 */,	0x58BC7A66C232BD5Eull  /*  465 */,
	0xF306558C673AC8B2ull  /*  466 */,	0x41F639C6B6C9772Aull  /*  467 */,
	0x216DEFE99FDA35DAull  /*  468 */,	0x11640CC71C7BE615ull  /*  469 */,
	0x93C43694565C5527ull  /*  470 */,	0xEA038E6246777839ull  /*  471 */,
	0xF9ABF3CE5A3E2469ull  /*  472 */,	0x741E768D0FD312D2ull  /*  473 */,
	0x0144B883CED652C6ull  /*  474 */,	0xC20B5A5BA33F8552ull  /*  475 */,
	0x1AE69633C3435A9Dull  /*  476 */,	0x97A28CA4088CFDECull  /*  477 */,
	0x8824A43C1E96F420ull  /*  478 */,	0x37612FA66EEEA746ull  /*  479 */,
	0x6B4CB165F9CF0E5Aull  /*  480 */,	0x43AA1C06A0ABFB4Aull  /*  481 */,
	0x7F4DC26FF162796Bull  /*  482 */,	0x6CBACC8E54ED9B0Full  /*  483 */,
	0xA6B7FFEFD2BB253Eull  /*  484 */,	0x2E25BC95B0A29D4Full  /*  485 */,
	0x86D6A58BDEF1388Cull  /*  486 */,	0xDED74AC576B6F054ull  /*  487 */,
	0x8030BDBC2B45805Dull  /*  488 */,	0x3C81AF70E94D9289ull  /*  489 */,
	0x3EFF6DDA9E3100DBull  /*  490 */,	0xB38DC39FDFCC8847ull  /*  491 */,
	0x123885528D17B87Eull  /*  492 */,	0xF2DA0ED240B1B642ull  /*  493 */,
	0x44CEFADCD54BF9A9ull  /*  494 */,	0x1312200E433C7EE6ull  /*  495 */,
	0x9FFCC84F3A78C748ull  /*  496 */,	0xF0CD1F72248576BBull  /*  497 */,
	0xEC6974053638CFE4ull  /*  498 */,	0x2BA7B67C0CEC4E4Cull  /*  499 */,
	0xAC2F4DF3E5CE32EDull  /*  500 */,	0xCB33D14326EA4C11ull  /*  501 */,
	0xA4E9044CC77E58BCull  /*  502 */,	0x5F513293D934FCEFull  /*  503 */,
	0x5DC9645506E55444ull  /*  504 */,	0x50DE418F317DE40Aull  /*  505 */,
	0x388CB31A69DDE259ull  /*  506 */,	0x2DB4A83455820A86ull  /*  507 */,
	0x9010A91E84711AE9ull  /*  508 */,	0x4DF7F0B7B1498371ull  /*  509 */,
	0xD62A2EABC0977179ull  /*  510 */,	0x22FAC097AA8D5C0Eull  /*  511 */,
	0xF49FCC2FF1DAF39Bull  /*  512 */,	0x487FD5C66FF29281ull  /*  513 */,
	0xE8A30667FCDCA83Full  /*  514 */,	0x2C9B4BE3D2FCCE63ull  /*  515 */,
	0xDA3FF74B93FBBBC2ull  /*  516 */,	0x2FA165D2FE70BA66ull  /*  517 */,
	0xA103E279970E93D4ull  /*  518 */,	0xBECDEC77B0E45E71ull  /*  519 */,
	0xCFB41E723985E497ull  /*  520 */,	0xB70AAA025EF75017ull  /*  521 */,
	0xD42309F03840B8E0ull  /*  522 */,	0x8EFC1AD035898579ull  /*  523 */,
	0x96C6920BE2B2ABC5ull  /*  524 */,	0x66AF4163375A9172ull  /*  525 */,
	0x2174ABDCCA7127FBull  /*  526 */,	0xB33CCEA64A72FF41ull  /*  527 */,
	0xF04A4933083066A5ull  /*  528 */,	0x8D970ACDD7289AF5ull  /*  529 */,
	0x8F96E8E031C8C25Eull  /*  530 */,	0xF3FEC02276875D47ull  /*  531 */,
	0xEC7BF310056190DDull  /*  532 */,	0xF5ADB0AEBB0F1491ull  /*  533 */,
	0x9B50F8850FD58892ull  /*  534 */,	0x4975488358B74DE8ull  /*  535 */,
	0xA3354FF691531C61ull  /*  536 */,	0x0702BBE481D2C6EEull  /*  537 */,
	0x89FB24057DEDED98ull  /*  538 */,	0xAC3075138596E902ull  /*  539 */,
	0x1D2D3580172772EDull  /*  540 */,	0xEB738FC28E6BC30Dull  /*  541 */,
	0x5854EF8F63044326ull  /*  542 */,	0x9E5C52325ADD3BBEull  /*  543 */,
	0x90AA53CF325C4623ull  /*  544 */,	0xC1D24D51349DD067ull  /*  545 */,
	0x2051CFEEA69EA624ull  /*  546 */,	0x13220F0A862E7E4Full  /*  547 */,
	0xCE39399404E04864ull  /*  548 */,	0xD9C42CA47086FCB7ull  /*  549 */,
	0x685AD2238A03E7CCull  /*  550 */,	0x066484B2AB2FF1DBull  /*  551 */,
	0xFE9D5D70EFBF79ECull  /*  552 */,	0x5B13B9DD9C481854ull  /*  553 */,
	0x15F0D475ED1509ADull  /*  554 */,	0x0BEBCD060EC79851ull  /*  555 */,
	0xD58C6791183AB7F8ull  /*  556 */,	0xD1187C5052F3EEE4ull  /*  557 */,
	0xC95D1192E54E82FFull  /*  558 */,	0x86EEA14CB9AC6CA2ull  /*  559 */,
	0x3485BEB153677D5Dull  /*  560 */,	0xDD191D781F8C492Aull  /*  561 */,
	0xF60866BAA784EBF9ull  /*  562 */,	0x518F643BA2D08C74ull  /*  563 */,
	0x8852E956E1087C22ull  /*  564 */,	0xA768CB8DC410AE8Dull  /*  565 */,
	0x38047726BFEC8E1Aull  /*  566 */,	0xA67738B4CD3B45AAull  /*  567 */,
	0xAD16691CEC0DDE19ull  /*  568 */,	0xC6D4319380462E07ull  /*  569 */,
	0xC5A5876D0BA61938ull  /*  570 */,	0x16B9FA1FA58FD840ull  /*  571 */,
	0x188AB1173CA74F18ull  /*  572 */,	0xABDA2F98C99C021Full  /*  573 */,
	0x3E0580AB134AE816ull  /*  574 */,	0x5F3B05B773645ABBull  /*  575 */,
	0x2501A2BE5575F2F6ull  /*  576 */,	0x1B2F74004E7E8BA9ull  /*  577 */,
	0x1CD7580371E8D953ull  /*  578 */,	0x7F6ED89562764E30ull  /*  579 */,
	0xB15926FF596F003Dull  /*  580 */,	0x9F65293DA8C5D6B9ull  /*  581 */,
	0x6ECEF04DD690F84Cull  /*  582 */,	0x4782275FFF33AF88ull  /*  583 */,
	0xE41433083F820801ull  /*  584 */,	0xFD0DFE409A1AF9B5ull  /*  585 */,
	0x4325A3342CDB396Bull  /*  586 */,	0x8AE77E62B301B252ull  /*  587 */,
	0xC36F9E9F6655615Aull  /*  588 */,	0x85455A2D92D32C09ull  /*  589 */,
	0xF2C7DEA949477485ull  /*  590 */,	0x63CFB4C133A39EBAull  /*  591 */,
	0x83B040CC6EBC5462ull  /*  592 */,	0x3B9454C8FDB326B0ull  /*  593 */,
	0x56F56A9E87FFD78Cull  /*  594 */,	0x2DC2940D99F42BC6ull  /*  595 */,
	0x98F7DF096B096E2Dull  /*  596 */,	0x19A6E01E3AD852BFull  /*  597 */,
	0x42A99CCBDBD4B40Bull  /*  598 */,	0xA59998AF45E9C559ull  /*  599 */,
	0x366295E807D93186ull  /*  600 */,	0x6B48181BFAA1F773ull  /*  601 */,
	0x1FEC57E2157A0A1Dull  /*  602 */,	0x4667446AF6201AD5ull  /*  603 */,
	0xE615EBCACFB0F075ull  /*  604 */,	0xB8F31F4F68290778ull  /*  605 */,
	0x22713ED6CE22D11Eull  /*  606 */,	0x3057C1A72EC3C93Bull  /*  607 */,
	0xCB46ACC37C3F1F2Full  /*  608 */,	0xDBB893FD02AAF50Eull  /*  609 */,
	0x331FD92E600B9FCFull  /*  610 */,	0xA498F96148EA3AD6ull  /*  611 */,
	0xA8D8426E8B6A83EAull  /*  612 */,	0xA089B274B7735CDCull  /*  613 */,
	0x87F6B3731E524A11ull  /*  614 */,	0x118808E5CBC96749ull  /*  615 */,
	0x9906E4C7B19BD394ull  /*  616 */,	0xAFED7F7E9B24A20Cull  /*  617 */,
	0x6509EADEEB3644A7ull  /*  618 */,	0x6C1EF1D3E8EF0EDEull  /*  619 */,
	0xB9C97D43E9798FB4ull  /*  620 */,	0xA2F2D784740C28A3ull  /*  621 */,
	0x7B8496476197566Full  /*  622 */,	0x7A5BE3E6B65F069Dull  /*  623 */,
	0xF96330ED78BE6F10ull  /*  624 */,	0xEEE60DE77A076A15ull  /*  625 */,
	0x2B4BEE4AA08B9BD0ull  /*  626 */,	0x6A56A63EC7B8894Eull  /*  627 */,
	0x02121359BA34FEF4ull  /*  628 */,	0x4CBF99F8283703FCull  /*  629 */,
	0x398071350CAF30C8ull  /*  630 */,	0xD0A77A89F017687Aull  /*  631 */,
	0xF1C1A9EB9E423569ull  /*  632 */,	0x8C7976282DEE8199ull  /*  633 */,
	0x5D1737A5DD1F7ABDull  /*  634 */,	0x4F53433C09A9FA80ull  /*  635 */,
	0xFA8B0C53DF7CA1D9ull  /*  636 */,	0x3FD9DCBC886CCB77ull  /*  637 */,
	0xC040917CA91B4720ull  /*  638 */,	0x7DD00142F9D1DCDFull  /*  639 */,
	0x8476FC1D4F387B58ull  /*  640 */,	0x23F8E7C5F3316503ull  /*  641 */,
	0x032A2244E7E37339ull  /*  642 */,	0x5C87A5D750F5A74Bull  /*  643 */,
	0x082B4CC43698992Eull  /*  644 */,	0xDF917BECB858F63Cull  /*  645 */,
	0x3270B8FC5BF86DDAull  /*  646 */,	0x10AE72BB29B5DD76ull  /*  647 */,
	0x576AC94E7700362Bull  /*  648 */,	0x1AD112DAC61EFB8Full  /*  649 */,
	0x691BC30EC5FAA427ull  /*  650 */,	0xFF246311CC327143ull  /*  651 */,
	0x3142368E30E53206ull  /*  652 */,	0x71380E31E02CA396ull  /*  653 */,
	0x958D5C960AAD76F1ull  /*  654 */,	0xF8D6F430C16DA536ull  /*  655 */,
	0xC8FFD13F1BE7E1D2ull  /*  656 */,	0x7578AE66004DDBE1ull  /*  657 */,
	0x05833F01067BE646ull  /*  658 */,	0xBB34B5AD3BFE586Dull  /*  659 */,
	0x095F34C9A12B97F0ull  /*  660 */,	0x247AB64525D60CA8ull  /*  661 */,
	0xDCDBC6F3017477D1ull  /*  662 */,	0x4A2E14D4DECAD24Dull  /*  663 */,
	0xBDB5E6D9BE0A1EEBull  /*  664 */,	0x2A7E70F7794301ABull  /*  665 */,
	0xDEF42D8A270540FDull  /*  666 */,	0x01078EC0A34C22C1ull  /*  667 */,
	0xE5DE511AF4C16387ull  /*  668 */,	0x7EBB3A52BD9A330Aull  /*  669 */,
	0x77697857AA7D6435ull  /*  670 */,	0x004E831603AE4C32ull  /*  671 */,
	0xE7A21020AD78E312ull  /*  672 */,	0x9D41A70C6AB420F2ull  /*  673 */,
	0x28E06C18EA1141E6ull  /*  674 */,	0xD2B28CBD984F6B28ull  /*  675 */,
	0x26B75F6C446E9D83ull  /*  676 */,	0xBA47568C4D418D7Full  /*  677 */,
	0xD80BADBFE6183D8Eull  /*  678 */,	0x0E206D7F5F166044ull  /*  679 */,
	0xE258A43911CBCA3Eull  /*  680 */,	0x723A1746B21DC0BCull  /*  681 */,
	0xC7CAA854F5D7CDD3ull  /*  682 */,	0x7CAC32883D261D9Cull  /*  683 */,
	0x7690C26423BA942Cull  /*  684 */,	0x17E55524478042B8ull  /*  685 */,
	0xE0BE477656A2389Full  /*  686 */,	0x4D289B5E67AB2DA0ull  /*  687 */,
	0x44862B9C8FBBFD31ull  /*  688 */,	0xB47CC8049D141365ull  /*  689 */,
	0x822C1B362B91C793ull  /*  690 */,	0x4EB14655FB13DFD8ull  /*  691 */,
	0x1ECBBA0714E2A97Bull  /*  692 */,	0x6143459D5CDE5F14ull  /*  693 */,
	0x53A8FBF1D5F0AC89ull  /*  694 */,	0x97EA04D81C5E5B00ull  /*  695 */,
	0x622181A8D4FDB3F3ull  /*  696 */,	0xE9BCD341572A1208ull  /*  697 */,
	0x1411258643CCE58Aull  /*  698 */,	0x9144C5FEA4C6E0A4ull  /*  699 */,
	0x0D33D06565CF620Full  /*  700 */,	0x54A48D489F219CA1ull  /*  701 */,
	0xC43E5EAC6D63C821ull  /*  702 */,	0xA9728B3A72770DAFull  /*  703 */,
	0xD7934E7B20DF87EFull  /*  704 */,	0xE35503B61A3E86E5ull  /*  705 */,
	0xCAE321FBC819D504ull  /*  706 */,	0x129A50B3AC60BFA6ull  /*  707 */,
	0xCD5E68EA7E9FB6C3ull  /*  708 */,	0xB01C90199483B1C7ull  /*  709 */,
	0x3DE93CD5C295376Cull  /*  710 */,	0xAED52EDF2AB9AD13ull  /*  711 */,
	0x2E60F512C0A07884ull  /*  712 */,	0xBC3D86A3E36210C9ull  /*  713 */,
	0x35269D9B163951CEull  /*  714 */,	0x0C7D6E2AD0CDB5FAull  /*  715 */,
	0x59E86297D87F5733ull  /*  716 */,	0x298EF221898DB0E7ull  /*  717 */,
	0x55000029D1A5AA7Eull  /*  718 */,	0x8BC08AE1B5061B45ull  /*  719 */,
	0xC2C31C2B6C92703Aull  /*  720 */,	0x94CC596BAF25EF42ull  /*  721 */,
	0x0A1D73DB22540456ull  /*  722 */,	0x04B6A0F9D9C4179Aull  /*  723 */,
	0xEFFDAFA2AE3D3C60ull  /*  724 */,	0xF7C8075BB49496C4ull  /*  725 */,
	0x9CC5C7141D1CD4E3ull  /*  726 */,	0x78BD1638218E5534ull  /*  727 */,
	0xB2F11568F850246Aull  /*  728 */,	0xEDFABCFA9502BC29ull  /*  729 */,
	0x796CE5F2DA23051Bull  /*  730 */,	0xAAE128B0DC93537Cull  /*  731 */,
	0x3A493DA0EE4B29AEull  /*  732 */,	0xB5DF6B2C416895D7ull  /*  733 */,
	0xFCABBD25122D7F37ull  /*  734 */,	0x70810B58105DC4B1ull  /*  735 */,
	0xE10FDD37F7882A90ull  /*  736 */,	0x524DCAB5518A3F5Cull  /*  737 */,
	0x3C9E85878451255Bull  /*  738 */,	0x4029828119BD34E2ull  /*  739 */,
	0x74A05B6F5D3CECCBull  /*  740 */,	0xB610021542E13ECAull  /*  741 */,
	0x0FF979D12F59E2ACull  /*  742 */,	0x6037DA27E4F9CC50ull  /*  743 */,
	0x5E92975A0DF1847Dull  /*  744 */,	0xD66DE190D3E623FEull  /*  745 */,
	0x5032D6B87B568048ull  /*  746 */,	0x9A36B7CE8235216Eull  /*  747 */,
	0x80272A7A24F64B4Aull  /*  748 */,	0x93EFED8B8C6916F7ull  /*  749 */,
	0x37DDBFF44CCE1555ull  /*  750 */,	0x4B95DB5D4B99BD25ull  /*  751 */,
	0x92D3FDA169812FC0ull  /*  752 */,	0xFB1A4A9A90660BB6ull  /*  753 */,
	0x730C196946A4B9B2ull  /*  754 */,	0x81E289AA7F49DA68ull  /*  755 */,
	0x64669A0F83B1A05Full  /*  756 */,	0x27B3FF7D9644F48Bull  /*  757 */,
	0xCC6B615C8DB675B3ull  /*  758 */,	0x674F20B9BCEBBE95ull  /*  759 */,
	0x6F31238275655982ull  /*  760 */,	0x5AE488713E45CF05ull  /*  761 */,
	0xBF619F9954C21157ull  /*  762 */,	0xEABAC46040A8EAE9ull  /*  763 */,
	0x454C6FE9F2C0C1CDull  /*  764 */,	0x419CF6496412691Cull  /*  765 */,
	0xD3DC3BEF265B0F70ull  /*  766 */,	0x6D0E60F5C3578A9Eull  /*  767 */,
	0x5B0E608526323C55ull  /*  768 */,	0x1A46C1A9FA1B59F5ull  /*  769 */,
	0xA9E245A17C4C8FFAull  /*  770 */,	0x65CA5159DB2955D7ull  /*  771 */,
	0x05DB0A76CE35AFC2ull  /*  772 */,	0x81EAC77EA9113D45ull  /*  773 */,
	0x528EF88AB6AC0A0Dull  /*  774 */,	0xA09EA253597BE3FFull  /*  775 */,
	0x430DDFB3AC48CD56ull  /*  776 */,	0xC4B3A67AF45CE46Full  /*  777 */,
	0x4ECECFD8FBE2D05Eull  /*  778 */,	0x3EF56F10B39935F0ull  /*  779 */,
	0x0B22D6829CD619C6ull  /*  780 */,	0x17FD460A74DF2069ull  /*  781 */,
	0x6CF8CC8E8510ED40ull  /*  782 */,	0xD6C824BF3A6ECAA7ull  /*  783 */,
	0x61243D581A817049ull  /*  784 */,	0x048BACB6BBC163A2ull  /*  785 */,
	0xD9A38AC27D44CC32ull  /*  786 */,	0x7FDDFF5BAAF410ABull  /*  787 */,
	0xAD6D495AA804824Bull  /*  788 */,	0xE1A6A74F2D8C9F94ull  /*  789 */,
	0xD4F7851235DEE8E3ull  /*  790 */,	0xFD4B7F886540D893ull  /*  791 */,
	0x247C20042AA4BFDAull  /*  792 */,	0x096EA1C517D1327Cull  /*  793 */,
	0xD56966B4361A6685ull  /*  794 */,	0x277DA5C31221057Dull  /*  795 */,
	0x94D59893A43ACFF7ull  /*  796 */,	0x64F0C51CCDC02281ull  /*  797 */,
	0x3D33BCC4FF6189DBull  /*  798 */,	0xE005CB184CE66AF1ull  /*  799 */,
	0xFF5CCD1D1DB99BEAull  /*  800 */,	0xB0B854A7FE42980Full  /*  801 */,
	0x7BD46A6A718D4B9Full  /*  802 */,	0xD10FA8CC22A5FD8Cull  /*  803 */,
	0xD31484952BE4BD31ull  /*  804 */,	0xC7FA975FCB243847ull  /*  805 */,
	0x4886ED1E5846C407ull  /*  806 */,	0x28CDDB791EB70B04ull  /*  807 */,
	0xC2B00BE2F573417Full  /*  808 */,	0x5C9590452180F877ull  /*  809 */,
	0x7A6BDDFFF370EB00ull  /*  810 */,	0xCE509E38D6D9D6A4ull  /*  811 */,
	0xEBEB0F00647FA702ull  /*  812 */,	0x1DCC06CF76606F06ull  /*  813 */,
	0xE4D9F28BA286FF0Aull  /*  814 */,	0xD85A305DC918C262ull  /*  815 */,
	0x475B1D8732225F54ull  /*  816 */,	0x2D4FB51668CCB5FEull  /*  817 */,
	0xA679B9D9D72BBA20ull  /*  818 */,	0x53841C0D912D43A5ull  /*  819 */,
	0x3B7EAA48BF12A4E8ull  /*  820 */,	0x781E0E47F22F1DDFull  /*  821 */,
	0xEFF20CE60AB50973ull  /*  822 */,	0x20D261D19DFFB742ull  /*  823 */,
	0x16A12B03062A2E39ull  /*  824 */,	0x1960EB2239650495ull  /*  825 */,
	0x251C16FED50EB8B8ull  /*  826 */,	0x9AC0C330F826016Eull  /*  827 */,
	0xED152665953E7671ull  /*  828 */,	0x02D63194A6369570ull  /*  829 */,
	0x5074F08394B1C987ull  /*  830 */,	0x70BA598C90B25CE1ull  /*  831 */,
	0x794A15810B9742F6ull  /*  832 */,	0x0D5925E9FCAF8C6Cull  /*  833 */,
	0x3067716CD868744Eull  /*  834 */,	0x910AB077E8D7731Bull  /*  835 */,
	0x6A61BBDB5AC42F61ull  /*  836 */,	0x93513EFBF0851567ull  /*  837 */,
	0xF494724B9E83E9D5ull  /*  838 */,	0xE887E1985C09648Dull  /*  839 */,
	0x34B1D3C675370CFDull  /*  840 */,	0xDC35E433BC0D255Dull  /*  841 */,
	0xD0AAB84234131BE0ull  /*  842 */,	0x08042A50B48B7EAFull  /*  843 */,
	0x9997C4EE44A3AB35ull  /*  844 */,	0x829A7B49201799D0ull  /*  845 */,
	0x263B8307B7C54441ull  /*  846 */,	0x752F95F4FD6A6CA6ull  /*  847 */,
	0x927217402C08C6E5ull  /*  848 */,	0x2A8AB754A795D9EEull  /*  849 */,
	0xA442F7552F72943Dull  /*  850 */,	0x2C31334E19781208ull  /*  851 */,
	0x4FA98D7CEAEE6291ull  /*  852 */,	0x55C3862F665DB309ull  /*  853 */,
	0xBD0610175D53B1F3ull  /*  854 */,	0x46FE6CB840413F27ull  /*  855 */,
	0x3FE03792DF0CFA59ull  /*  856 */,	0xCFE700372EB85E8Full  /*  857 */,
	0xA7BE29E7ADBCE118ull  /*  858 */,	0xE544EE5CDE8431DDull  /*  859 */,
	0x8A781B1B41F1873Eull  /*  860 */,	0xA5C94C78A0D2F0E7ull  /*  861 */,
	0x39412E2877B60728ull  /*  862 */,	0xA1265EF3AFC9A62Cull  /*  863 */,
	0xBCC2770C6A2506C5ull  /*  864 */,	0x3AB66DD5DCE1CE12ull  /*  865 */,
	0xE65499D04A675B37ull  /*  866 */,	0x7D8F523481BFD216ull  /*  867 */,
	0x0F6F64FCEC15F389ull  /*  868 */,	0x74EFBE618B5B13C8ull  /*  869 */,
	0xACDC82B714273E1Dull  /*  870 */,	0xDD40BFE003199D17ull  /*  871 */,
	0x37E99257E7E061F8ull  /*  872 */,	0xFA52626904775AAAull  /*  873 */,
	0x8BBBF63A463D56F9ull  /*  874 */,	0xF0013F1543A26E64ull  /*  875 */,
	0xA8307E9F879EC898ull  /*  876 */,	0xCC4C27A4150177CCull  /*  877 */,
	0x1B432F2CCA1D3348ull  /*  878 */,	0xDE1D1F8F9F6FA013ull  /*  879 */,
	0x606602A047A7DDD6ull  /*  880 */,	0xD237AB64CC1CB2C7ull  /*  881 */,
	0x9B938E7225FCD1D3ull  /*  882 */,	0xEC4E03708E0FF476ull  /*  883 */,
	0xFEB2FBDA3D03C12Dull  /*  884 */,	0xAE0BCED2EE43889Aull  /*  885 */,
	0x22CB8923EBFB4F43ull  /*  886 */,	0x69360D013CF7396Dull  /*  887 */,
	0x855E3602D2D4E022ull  /*  888 */,	0x073805BAD01F784Cull  /*  889 */,
	0x33E17A133852F546ull  /*  890 */,	0xDF4874058AC7B638ull  /*  891 */,
	0xBA92B29C678AA14Aull  /*  892 */,	0x0CE89FC76CFAADCDull  /*  893 */,
	0x5F9D4E0908339E34ull  /*  894 */,	0xF1AFE9291F5923B9ull  /*  895 */,
	0x6E3480F60F4A265Full  /*  896 */,	0xEEBF3A2AB29B841Cull  /*  897 */,
	0xE21938A88F91B4ADull  /*  898 */,	0x57DFEFF845C6D3C3ull  /*  899 */,
	0x2F006B0BF62CAAF2ull  /*  900 */,	0x62F479EF6F75EE78ull  /*  901 */,
	0x11A55AD41C8916A9ull  /*  902 */,	0xF229D29084FED453ull  /*  903 */,
	0x42F1C27B16B000E6ull  /*  904 */,	0x2B1F76749823C074ull  /*  905 */,
	0x4B76ECA3C2745360ull  /*  906 */,	0x8C98F463B91691BDull  /*  907 */,
	0x14BCC93CF1ADE66Aull  /*  908 */,	0x8885213E6D458397ull  /*  909 */,
	0x8E177DF0274D4711ull  /*  910 */,	0xB49B73B5503F2951ull  /*  911 */,
	0x10168168C3F96B6Bull  /*  912 */,	0x0E3D963B63CAB0AEull  /*  913 */,
	0x8DFC4B5655A1DB14ull  /*  914 */,	0xF789F1356E14DE5Cull  /*  915 */,
	0x683E68AF4E51DAC1ull  /*  916 */,	0xC9A84F9D8D4B0FD9ull  /*  917 */,
	0x3691E03F52A0F9D1ull  /*  918 */,	0x5ED86E46E1878E80ull  /*  919 */,
	0x3C711A0E99D07150ull  /*  920 */,	0x5A0865B20C4E9310ull  /*  921 */,
	0x56FBFC1FE4F0682Eull  /*  922 */,	0xEA8D5DE3105EDF9Bull  /*  923 */,
	0x71ABFDB12379187Aull  /*  924 */,	0x2EB99DE1BEE77B9Cull  /*  925 */,
	0x21ECC0EA33CF4523ull  /*  926 */,	0x59A4D7521805C7A1ull  /*  927 */,
	0x3896F5EB56AE7C72ull  /*  928 */,	0xAA638F3DB18F75DCull  /*  929 */,
	0x9F39358DABE9808Eull  /*  930 */,	0xB7DEFA91C00B72ACull  /*  931 */,
	0x6B5541FD62492D92ull  /*  932 */,	0x6DC6DEE8F92E4D5Bull  /*  933 */,
	0x353F57ABC4BEEA7Eull  /*  934 */,	0x735769D6DA5690CEull  /*  935 */,
	0x0A234AA642391484ull  /*  936 */,	0xF6F9508028F80D9Dull  /*  937 */,
	0xB8E319A27AB3F215ull  /*  938 */,	0x31AD9C1151341A4Dull  /*  939 */,
	0x773C22A57BEF5805ull  /*  940 */,	0x45C7561A07968633ull  /*  941 */,
	0xF913DA9E249DBE36ull  /*  942 */,	0xDA652D9B78A64C68ull  /*  943 */,
	0x4C27A97F3BC334EFull  /*  944 */,	0x76621220E66B17F4ull  /*  945 */,
	0x967743899ACD7D0Bull  /*  946 */,	0xF3EE5BCAE0ED6782ull  /*  947 */,
	0x409F753600C879FCull  /*  948 */,	0x06D09A39B5926DB6ull  /*  949 */,
	0x6F83AEB0317AC588ull  /*  950 */,	0x01E6CA4A86381F21ull  /*  951 */,
	0x66FF3462D19F3025ull  /*  952 */,	0x72207C24DDFD3BFBull  /*  953 */,
	0x4AF6B6D3E2ECE2EBull  /*  954 */,	0x9C994DBEC7EA08DEull  /*  955 */,
	0x49ACE597B09A8BC4ull  /*  956 */,	0xB38C4766CF0797BAull  /*  957 */,
	0x131B9373C57C2A75ull  /*  958 */,	0xB1822CCE61931E58ull  /*  959 */,
	0x9D7555B909BA1C0Cull  /*  960 */,	0x127FAFDD937D11D2ull  /*  961 */,
	0x29DA3BADC66D92E4ull  /*  962 */,	0xA2C1D57154C2ECBCull  /*  963 */,
	0x58C5134D82F6FE24ull  /*  964 */,	0x1C3AE3515B62274Full  /*  965 */,
	0xE907C82E01CB8126ull  /*  966 */,	0xF8ED091913E37FCBull  /*  967 */,
	0x3249D8F9C80046C9ull  /*  968 */,	0x80CF9BEDE388FB63ull  /*  969 */,
	0x1881539A116CF19Eull  /*  970 */,	0x5103F3F76BD52457ull  /*  971 */,
	0x15B7E6F5AE47F7A8ull  /*  972 */,	0xDBD7C6DED47E9CCFull  /*  973 */,
	0x44E55C410228BB1Aull  /*  974 */,	0xB647D4255EDB4E99ull  /*  975 */,
	0x5D11882BB8AAFC30ull  /*  976 */,	0xF5098BBB29D3212Aull  /*  977 */,
	0x8FB5EA14E90296B3ull  /*  978 */,	0x677B942157DD025Aull  /*  979 */,
	0xFB58E7C0A390ACB5ull  /*  980 */,	0x89D3674C83BD4A01ull  /*  981 */,
	0x9E2DA4DF4BF3B93Bull  /*  982 */,	0xFCC41E328CAB4829ull  /*  983 */,
	0x03F38C96BA582C52ull  /*  984 */,	0xCAD1BDBD7FD85DB2ull  /*  985 */,
	0xBBB442C16082AE83ull  /*  986 */,	0xB95FE86BA5DA9AB0ull  /*  987 */,
	0xB22E04673771A93Full  /*  988 */,	0x845358C9493152D8ull  /*  989 */,
	0xBE2A488697B4541Eull  /*  990 */,	0x95A2DC2DD38E6966ull  /*  991 */,
	0xC02C11AC923C852Bull  /*  992 */,	0x2388B1990DF2A87Bull  /*  993 */,
	0x7C8008FA1B4F37BEull  /*  994 */,	0x1F70D0C84D54E503ull  /*  995 */,
	0x5490ADEC7ECE57D4ull  /*  996 */,	0x002B3C27D9063A3Aull  /*  997 */,
	0x7EAEA3848030A2BFull  /*  998 */,	0xC602326DED2003C0ull  /*  999 */,
	0x83A7287D69A94086ull  /* 1000 */,	0xC57A5FCB30F57A8Aull  /* 1001 */,
	0xB56844E479EBE779ull  /* 1002 */,	0xA373B40F05DCBCE9ull  /* 1003 */,
	0xD71A786E88570EE2ull  /* 1004 */,	0x879CBACDBDE8F6A0ull  /* 1005 */,
	0x976AD1BCC164A32Full  /* 1006 */,	0xAB21E25E9666D78Bull  /* 1007 */,
	0x901063AAE5E5C33Cull  /* 1008 */,	0x9818B34448698D90ull  /* 1009 */,
	0xE36487AE3E1E8ABBull  /* 1010 */,	0xAFBDF931893BDCB4ull  /* 1011 */,
	0x6345A0DC5FBBD519ull  /* 1012 */,	0x8628FE269B9465CAull  /* 1013 */,
	0x1E5D01603F9C51ECull  /* 1014 */,	0x4DE44006A15049B7ull  /* 1015 */,
	0xBF6C70E5F776CBB1ull  /* 1016 */,	0x411218F2EF552BEDull  /* 1017 */,
	0xCB0C0708705A36A3ull  /* 1018 */,	0xE74D14754F986044ull  /* 1019 */,
	0xCD56D9430EA8280Eull  /* 1020 */,	0xC12591D7535F5065ull  /* 1021 */,
	0xC83223F1720AEF96ull  /* 1022 */,	0xC3A0396F7363A51Full  /* 1023 */
};
