/*
    Tucnak - VHF contest log
    Copyright (C) 2002-2006  Ladislav Vaiz <ok1zia@nagano.cz>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

 
#include "header.h"

#define RC_MAIN "/etc/tucnakrc"
#define RC_USER "/tucnak/tucnakrc"
#define RC_LOCAL "/tucnak/tucnakrc.local"


struct config *cfg;
    
enum {
    SYMBOL_OBSOLETE     = G_TOKEN_LAST + 10000,
    SYMBOL_MY_FIRST     = G_TOKEN_LAST + 1,
    
    SYMBOL_PCALL        = G_TOKEN_LAST + 1,
    SYMBOL_PWWLO        = G_TOKEN_LAST + 2,
    SYMBOL_PEXCH        = G_TOKEN_LAST + 3,
    SYMBOL_PADR1        = G_TOKEN_LAST + 4,
    SYMBOL_PADR2        = G_TOKEN_LAST + 5,
    SYMBOL_PCLUB        = G_TOKEN_LAST + 6,
    SYMBOL_RNAME        = G_TOKEN_LAST + 7,
    SYMBOL_RCALL        = G_TOKEN_LAST + 8,
    SYMBOL_RADR1        = G_TOKEN_LAST + 9,
    SYMBOL_RADR2        = G_TOKEN_LAST + 10,
    SYMBOL_RPOCO        = G_TOKEN_LAST + 11,
    SYMBOL_RCITY        = G_TOKEN_LAST + 12,
    SYMBOL_RCOUN        = G_TOKEN_LAST + 13,
    SYMBOL_RPHON        = G_TOKEN_LAST + 14,
    SYMBOL_RHBBS        = G_TOKEN_LAST + 15,
    SYMBOL_RS           = G_TOKEN_LAST + 16,
    SYMBOL_RST          = G_TOKEN_LAST + 17,
    SYMBOL_QSOP_M       = G_TOKEN_LAST + 18,
    SYMBOL_TOTAL_M      = G_TOKEN_LAST + 19,
    
    SYMBOL_CQ           = G_TOKEN_LAST + 20,
    SYMBOL_CW_STR       = G_TOKEN_LAST + 21,
    SYMBOL_CW_SPEED     = G_TOKEN_LAST + 22,
    SYMBOL_CW_REPEAT    = G_TOKEN_LAST + 23,
    SYMBOL_CW_TS        = G_TOKEN_LAST + 24,
    SYMBOL_CW_ALLOWUND  = G_TOKEN_LAST + 25,
    SYMBOL_SSB_FILE     = G_TOKEN_LAST + 26,
    SYMBOL_SSB_TS       = G_TOKEN_LAST + 27,
    SYMBOL_SSB_REPEAT   = G_TOKEN_LAST + 28,
    
    SYMBOL_STARTBAND    = G_TOKEN_LAST + 30,
    SYMBOL_PBAND        = G_TOKEN_LAST + 31,
    SYMBOL_BANDCHAR     = G_TOKEN_LAST + 32,
    SYMBOL_QRV          = G_TOKEN_LAST + 33,
    SYMBOL_READONLY     = G_TOKEN_LAST + 34,
    SYMBOL_PSECT        = G_TOKEN_LAST + 35,
    SYMBOL_STXEQ        = G_TOKEN_LAST + 36,
    SYMBOL_SPOWE        = G_TOKEN_LAST + 37,
    SYMBOL_SRXEQ        = G_TOKEN_LAST + 38,
    SYMBOL_SANTE        = G_TOKEN_LAST + 39,
    SYMBOL_SANTH        = G_TOKEN_LAST + 40,
    SYMBOL_MOPE1        = G_TOKEN_LAST + 41,
    SYMBOL_MOPE2        = G_TOKEN_LAST + 42,
    SYMBOL_REMARKS      = G_TOKEN_LAST + 43,
    
    SYMBOL_OK_SECTION_SINGLE    = G_TOKEN_LAST + 44,
    SYMBOL_OK_SECTION_MULTI     = G_TOKEN_LAST + 45,
    SYMBOL_ADIFBAND             = G_TOKEN_LAST + 46,
    SYMBOL_QRG_MIN      = G_TOKEN_LAST + 47,
    SYMBOL_QRG_MAX      = G_TOKEN_LAST + 48,
    SYMBOL_BANDMULTI    = G_TOKEN_LAST + 49,
    SYMBOL_SKEDQRG      = G_TOKEN_LAST + 50,
    
    SYMBOL_AS_DAQ       = G_TOKEN_LAST + 60,
    SYMBOL_AS_DAM       = G_TOKEN_LAST + 61,
    SYMBOL_AS_DFS       = G_TOKEN_LAST + 62,
    SYMBOL_AS_FAQ       = G_TOKEN_LAST + 63,
    SYMBOL_AS_FAM       = G_TOKEN_LAST + 64,
    SYMBOL_AS_FFS       = G_TOKEN_LAST + 65,
    SYMBOL_AS_FMO       = G_TOKEN_LAST + 66,
    SYMBOL_AS_FPA       = G_TOKEN_LAST + 67,
    SYMBOL_AS_FMC       = G_TOKEN_LAST + 68,
    
    SYMBOL_IF_IGNORE    = G_TOKEN_LAST + 70,
    SYMBOL_IP_IGNORE    = G_TOKEN_LAST + 71,
    SYMBOL_IP_ANNOUNCE  = G_TOKEN_LAST + 72,
    SYMBOL_TRACE_BCAST  = G_TOKEN_LAST + 73,
    SYMBOL_TRACE_SOCK   = G_TOKEN_LAST + 74,
    SYMBOL_TRACE_RECV   = G_TOKEN_LAST + 75,
    SYMBOL_TRACE_SEND   = G_TOKEN_LAST + 76,
    SYMBOL_TRACE_QSOS   = G_TOKEN_LAST + 77,
    SYMBOL_LANGUAGE     = G_TOKEN_LAST + 78,

    SYMBOL_CWDA_HOST     = G_TOKEN_LAST + 80,
    SYMBOL_CWDA_UDP      = G_TOKEN_LAST + 81,
    SYMBOL_CWDA_DEVICE_S = G_TOKEN_LAST + 82,
    SYMBOL_CWDA_SPEED    = G_TOKEN_LAST + 83,
    SYMBOL_CWDA_WEIGHT   = G_TOKEN_LAST + 84,
    SYMBOL_CWDA_SPKDEV   = G_TOKEN_LAST + 85,
    SYMBOL_CWDA_SPKVOL   = G_TOKEN_LAST + 86,
    SYMBOL_CWDA_PTTDELAY = G_TOKEN_LAST + 87,
#if 0    
    SYMBOL_CWDA_DEVICE   = G_TOKEN_LAST + 88,/* obsolete */
    SYMBOL_CWDA_SPEAKER  = G_TOKEN_LAST + 89,/* obsolete */
    SYMBOL_CWDA_PORT     = G_TOKEN_LAST + 90,/* obsolete */
#endif

	SYMBOL_SSBD_HOST       = G_TOKEN_LAST + 100,
	SYMBOL_SSBD_UDP        = G_TOKEN_LAST + 101,
	SYMBOL_SSBD_DSP        = G_TOKEN_LAST + 102,
	SYMBOL_SSBD_RECORD     = G_TOKEN_LAST + 103,
	SYMBOL_SSBD_FORMAT     = G_TOKEN_LAST + 104,
	SYMBOL_SSBD_CHANNELS   = G_TOKEN_LAST + 105,
	SYMBOL_SSBD_SAMPLERATE = G_TOKEN_LAST + 106,
	SYMBOL_SSBD_PIPE       = G_TOKEN_LAST + 107,
	SYMBOL_SSBD_COMMAND    = G_TOKEN_LAST + 108,
	SYMBOL_SSBD_TEMPLATE   = G_TOKEN_LAST + 109,
	SYMBOL_SSBD_MIXER      = G_TOKEN_LAST + 110,
	SYMBOL_SSBD_RECSRC     = G_TOKEN_LAST + 111,
    
	SYMBOL_TERM 			 = G_TOKEN_LAST + 120,
	SYMBOL_TERM_MODE 		 = G_TOKEN_LAST + 121,
	SYMBOL_TERM_M11_HACK	 = G_TOKEN_LAST + 122,
	SYMBOL_TERM_RESTRICT_852 = G_TOKEN_LAST + 123,
	SYMBOL_TERM_BLOCK_CURSOR = G_TOKEN_LAST + 124,
	SYMBOL_TERM_COL			 = G_TOKEN_LAST + 125,
	SYMBOL_TERM_UTF_8_IO     = G_TOKEN_LAST + 126,
	SYMBOL_TERM_CHARSET		 = G_TOKEN_LAST + 127,

    SYMBOL_ROTAR            = G_TOKEN_LAST + 130,
	SYMBOL_ROT_TYPE	        = G_TOKEN_LAST + 131,
	SYMBOL_ROT_PORT         = G_TOKEN_LAST + 132,
	SYMBOL_ROT_SADDR        = G_TOKEN_LAST + 133,
	SYMBOL_ROT_TIMEOUT_MS   = G_TOKEN_LAST + 134,
	SYMBOL_ROT_FILENAME     = G_TOKEN_LAST + 135,
	SYMBOL_ROT_HOSTNAME     = G_TOKEN_LAST + 136,
	SYMBOL_ROT_DESC         = G_TOKEN_LAST + 137,
	SYMBOL_ROT_BEAMWIDTH    = G_TOKEN_LAST + 138,
	
	
    SYMBOL_SW                = G_TOKEN_LAST + 150,
    SYMBOL_SW_TYPE           = G_TOKEN_LAST + 151,
    SYMBOL_SW_COMMAND        = G_TOKEN_LAST + 152,
    SYMBOL_SW_AUTORUN        = G_TOKEN_LAST + 153,
    SYMBOL_SW_RESPAWN_TIME   = G_TOKEN_LAST + 154,

    SYMBOL_LOGLINES          = G_TOKEN_LAST + 160,
    SYMBOL_SKEDSHIFT         = G_TOKEN_LAST + 161,
    SYMBOL_TAKEOFF           = G_TOKEN_LAST + 162,
    
    SYMBOL_MY_LAST     		 = G_TOKEN_LAST + 170
};

void add_symbols(GScanner *scanner){
    g_scanner_add_symbol(scanner,"pcall",GINT_TO_POINTER(SYMBOL_PCALL)); 
    g_scanner_add_symbol(scanner,"pwwlo",GINT_TO_POINTER(SYMBOL_PWWLO)); 
    g_scanner_add_symbol(scanner,"pexch",GINT_TO_POINTER(SYMBOL_PEXCH)); 
    
    g_scanner_add_symbol(scanner,"padr1",GINT_TO_POINTER(SYMBOL_PADR1)); 
    g_scanner_add_symbol(scanner,"padr2",GINT_TO_POINTER(SYMBOL_PADR2)); 
    g_scanner_add_symbol(scanner,"pclub",GINT_TO_POINTER(SYMBOL_PCLUB)); 
    
    g_scanner_add_symbol(scanner,"rname",GINT_TO_POINTER(SYMBOL_RNAME)); 
    g_scanner_add_symbol(scanner,"rcall",GINT_TO_POINTER(SYMBOL_RCALL)); 
    g_scanner_add_symbol(scanner,"radr1",GINT_TO_POINTER(SYMBOL_RADR1)); 
    g_scanner_add_symbol(scanner,"radr2",GINT_TO_POINTER(SYMBOL_RADR2)); 
    g_scanner_add_symbol(scanner,"rpoco",GINT_TO_POINTER(SYMBOL_RPOCO)); 
    g_scanner_add_symbol(scanner,"rcity",GINT_TO_POINTER(SYMBOL_RCITY)); 
    g_scanner_add_symbol(scanner,"rcoun",GINT_TO_POINTER(SYMBOL_RCOUN)); 
    g_scanner_add_symbol(scanner,"rphon",GINT_TO_POINTER(SYMBOL_RPHON)); 
    g_scanner_add_symbol(scanner,"rhbbs",GINT_TO_POINTER(SYMBOL_RHBBS)); 
    
    g_scanner_add_symbol(scanner,"default_rs",  GINT_TO_POINTER(SYMBOL_RS)); 
    g_scanner_add_symbol(scanner,"default_rst", GINT_TO_POINTER(SYMBOL_RST)); 
    g_scanner_add_symbol(scanner,"qsop_method", GINT_TO_POINTER(SYMBOL_QSOP_M)); 
    g_scanner_add_symbol(scanner,"total_method",GINT_TO_POINTER(SYMBOL_TOTAL_M)); 
    
    g_scanner_add_symbol(scanner,"cq",         GINT_TO_POINTER(SYMBOL_CQ)); 
    g_scanner_add_symbol(scanner,"cw_str",     GINT_TO_POINTER(SYMBOL_CW_STR)); 
    g_scanner_add_symbol(scanner,"cw_speed",   GINT_TO_POINTER(SYMBOL_CW_SPEED)); 
    g_scanner_add_symbol(scanner,"cw_repeat",  GINT_TO_POINTER(SYMBOL_CW_REPEAT)); 
    g_scanner_add_symbol(scanner,"cw_ts",      GINT_TO_POINTER(SYMBOL_CW_TS)); 
    g_scanner_add_symbol(scanner,"cw_allowifundef", GINT_TO_POINTER(SYMBOL_CW_ALLOWUND)); 
    g_scanner_add_symbol(scanner,"ssb_file",   GINT_TO_POINTER(SYMBOL_SSB_FILE)); 
    g_scanner_add_symbol(scanner,"ssb_ts",     GINT_TO_POINTER(SYMBOL_SSB_TS)); 
    g_scanner_add_symbol(scanner,"ssb_repeat", GINT_TO_POINTER(SYMBOL_SSB_REPEAT)); 


    g_scanner_add_symbol(scanner,"startband",   GINT_TO_POINTER(SYMBOL_STARTBAND)); 
    g_scanner_add_symbol(scanner,"pband",   GINT_TO_POINTER(SYMBOL_PBAND)); 
    g_scanner_add_symbol(scanner,"bandchar",GINT_TO_POINTER(SYMBOL_BANDCHAR)); 
    g_scanner_add_symbol(scanner,"qrv",     GINT_TO_POINTER(SYMBOL_QRV)); 
    g_scanner_add_symbol(scanner,"readonly",GINT_TO_POINTER(SYMBOL_READONLY)); 
    g_scanner_add_symbol(scanner,"psect",GINT_TO_POINTER(SYMBOL_PSECT)); 
    g_scanner_add_symbol(scanner,"stxeq",GINT_TO_POINTER(SYMBOL_STXEQ)); 
    g_scanner_add_symbol(scanner,"spowe",GINT_TO_POINTER(SYMBOL_SPOWE)); 
    g_scanner_add_symbol(scanner,"srxeq",GINT_TO_POINTER(SYMBOL_SRXEQ)); 
    g_scanner_add_symbol(scanner,"sante",GINT_TO_POINTER(SYMBOL_SANTE)); 
    g_scanner_add_symbol(scanner,"santh",GINT_TO_POINTER(SYMBOL_SANTH)); 
    g_scanner_add_symbol(scanner,"mope1",GINT_TO_POINTER(SYMBOL_MOPE1)); 
    g_scanner_add_symbol(scanner,"mope2",GINT_TO_POINTER(SYMBOL_MOPE2)); 
    g_scanner_add_symbol(scanner,"remarks",GINT_TO_POINTER(SYMBOL_REMARKS)); 
    g_scanner_add_symbol(scanner,"ok_section_single",GINT_TO_POINTER(SYMBOL_OK_SECTION_SINGLE)); 
    g_scanner_add_symbol(scanner,"ok_section_multi", GINT_TO_POINTER(SYMBOL_OK_SECTION_MULTI)); 
    g_scanner_add_symbol(scanner,"adifband", GINT_TO_POINTER(SYMBOL_ADIFBAND)); 
    g_scanner_add_symbol(scanner,"qrg_min", GINT_TO_POINTER(SYMBOL_QRG_MIN)); 
    g_scanner_add_symbol(scanner,"qrg_max", GINT_TO_POINTER(SYMBOL_QRG_MAX)); 
    g_scanner_add_symbol(scanner,"bandmulti", GINT_TO_POINTER(SYMBOL_BANDMULTI)); 
    g_scanner_add_symbol(scanner,"skedqrg", GINT_TO_POINTER(SYMBOL_SKEDQRG)); 
    
    g_scanner_add_symbol(scanner,"as_disk_qso",     GINT_TO_POINTER(SYMBOL_AS_DAQ)); 
    g_scanner_add_symbol(scanner,"as_disk_min",     GINT_TO_POINTER(SYMBOL_AS_DAM)); 
    g_scanner_add_symbol(scanner,"as_disk_fsync",   GINT_TO_POINTER(SYMBOL_AS_DFS)); 
    g_scanner_add_symbol(scanner,"as_floppy_qso",   GINT_TO_POINTER(SYMBOL_AS_FAQ)); 
    g_scanner_add_symbol(scanner,"as_floppy_min",   GINT_TO_POINTER(SYMBOL_AS_FAM)); 
    g_scanner_add_symbol(scanner,"as_floppy_fsync", GINT_TO_POINTER(SYMBOL_AS_FFS)); 
    g_scanner_add_symbol(scanner,"as_mount_floppy", GINT_TO_POINTER(SYMBOL_AS_FMO)); 
    g_scanner_add_symbol(scanner,"as_floppy_path",  GINT_TO_POINTER(SYMBOL_AS_FPA)); 
    g_scanner_add_symbol(scanner,"as_mount_cmd",    GINT_TO_POINTER(SYMBOL_AS_FMC)); 
    
    g_scanner_add_symbol(scanner,"net_if_ignore",    GINT_TO_POINTER(SYMBOL_IF_IGNORE)); 
    g_scanner_add_symbol(scanner,"net_ip_ignore",    GINT_TO_POINTER(SYMBOL_IP_IGNORE)); 
    g_scanner_add_symbol(scanner,"net_ip_announce",  GINT_TO_POINTER(SYMBOL_IP_ANNOUNCE)); 
    g_scanner_add_symbol(scanner,"trace_bcast",      GINT_TO_POINTER(SYMBOL_TRACE_BCAST)); 
    g_scanner_add_symbol(scanner,"trace_sock",       GINT_TO_POINTER(SYMBOL_TRACE_SOCK)); 
    g_scanner_add_symbol(scanner,"trace_recv",       GINT_TO_POINTER(SYMBOL_TRACE_RECV)); 
    g_scanner_add_symbol(scanner,"trace_send",       GINT_TO_POINTER(SYMBOL_TRACE_SEND)); 
    g_scanner_add_symbol(scanner,"trace_qsos",       GINT_TO_POINTER(SYMBOL_TRACE_QSOS)); 
    g_scanner_add_symbol(scanner,"language",         GINT_TO_POINTER(SYMBOL_LANGUAGE)); 
    
    g_scanner_add_symbol(scanner,"cwda_hostname",  GINT_TO_POINTER(SYMBOL_CWDA_HOST)); 
    g_scanner_add_symbol(scanner,"cwda_udp_port",  GINT_TO_POINTER(SYMBOL_CWDA_UDP)); 
    g_scanner_add_symbol(scanner,"cwda_device_s",  GINT_TO_POINTER(SYMBOL_CWDA_DEVICE_S)); 
    g_scanner_add_symbol(scanner,"cwda_speed",     GINT_TO_POINTER(SYMBOL_CWDA_SPEED)); 
    g_scanner_add_symbol(scanner,"cwda_weight",     GINT_TO_POINTER(SYMBOL_CWDA_WEIGHT)); 
    g_scanner_add_symbol(scanner,"cwda_spkdev",     GINT_TO_POINTER(SYMBOL_CWDA_SPKDEV)); 
    g_scanner_add_symbol(scanner,"cwda_spkvol",     GINT_TO_POINTER(SYMBOL_CWDA_SPKVOL)); 
    g_scanner_add_symbol(scanner,"cwda_pttdelay",     GINT_TO_POINTER(SYMBOL_CWDA_PTTDELAY)); 
    
	g_scanner_add_symbol(scanner,"ssbd_hostname",   GINT_TO_POINTER(SYMBOL_SSBD_HOST)); 
	g_scanner_add_symbol(scanner,"ssbd_udp_port",   GINT_TO_POINTER(SYMBOL_SSBD_UDP)); 
	g_scanner_add_symbol(scanner,"ssbd_dsp",        GINT_TO_POINTER(SYMBOL_SSBD_DSP)); 
	g_scanner_add_symbol(scanner,"ssbd_record",     GINT_TO_POINTER(SYMBOL_SSBD_RECORD)); 
	g_scanner_add_symbol(scanner,"ssbd_format",     GINT_TO_POINTER(SYMBOL_SSBD_FORMAT)); 
	g_scanner_add_symbol(scanner,"ssbd_channels",   GINT_TO_POINTER(SYMBOL_SSBD_CHANNELS)); 
	g_scanner_add_symbol(scanner,"ssbd_samplerate", GINT_TO_POINTER(SYMBOL_SSBD_SAMPLERATE)); 
	g_scanner_add_symbol(scanner,"ssbd_pipe",       GINT_TO_POINTER(SYMBOL_SSBD_PIPE)); 
	g_scanner_add_symbol(scanner,"ssbd_command",    GINT_TO_POINTER(SYMBOL_SSBD_COMMAND)); 
	g_scanner_add_symbol(scanner,"ssbd_template",   GINT_TO_POINTER(SYMBOL_SSBD_TEMPLATE)); 
	g_scanner_add_symbol(scanner,"ssbd_mixer",      GINT_TO_POINTER(SYMBOL_SSBD_MIXER)); 
	g_scanner_add_symbol(scanner,"ssbd_recsrc",     GINT_TO_POINTER(SYMBOL_SSBD_RECSRC)); 
	
    g_scanner_add_symbol(scanner,"term",   		 		GINT_TO_POINTER(SYMBOL_TERM)); 
    g_scanner_add_symbol(scanner,"term_mode",			GINT_TO_POINTER(SYMBOL_TERM_MODE)); 
    g_scanner_add_symbol(scanner,"term_m11_hack",   	GINT_TO_POINTER(SYMBOL_TERM_M11_HACK)); 
    g_scanner_add_symbol(scanner,"term_restrict_852",   GINT_TO_POINTER(SYMBOL_TERM_RESTRICT_852)); 
    g_scanner_add_symbol(scanner,"term_block_cursor",   GINT_TO_POINTER(SYMBOL_TERM_BLOCK_CURSOR)); 
    g_scanner_add_symbol(scanner,"term_col",   			GINT_TO_POINTER(SYMBOL_TERM_COL));
    g_scanner_add_symbol(scanner,"term_utf_8_io",       GINT_TO_POINTER(SYMBOL_TERM_UTF_8_IO));	
    g_scanner_add_symbol(scanner,"term_charset",   		GINT_TO_POINTER(SYMBOL_TERM_CHARSET)); 
    
	g_scanner_add_symbol(scanner,"rotar",   	        GINT_TO_POINTER(SYMBOL_ROTAR)); 
	g_scanner_add_symbol(scanner,"rot_type",   		    GINT_TO_POINTER(SYMBOL_ROT_TYPE)); 
	g_scanner_add_symbol(scanner,"rot_port",   		    GINT_TO_POINTER(SYMBOL_ROT_PORT)); 
	g_scanner_add_symbol(scanner,"rot_saddr",   		GINT_TO_POINTER(SYMBOL_ROT_SADDR)); 
	g_scanner_add_symbol(scanner,"rot_timeout_ms",   	GINT_TO_POINTER(SYMBOL_ROT_TIMEOUT_MS)); 
	g_scanner_add_symbol(scanner,"rot_filename",   	    GINT_TO_POINTER(SYMBOL_ROT_FILENAME)); 
	g_scanner_add_symbol(scanner,"rot_hostname",   	    GINT_TO_POINTER(SYMBOL_ROT_HOSTNAME)); 
	g_scanner_add_symbol(scanner,"rot_desc",            GINT_TO_POINTER(SYMBOL_ROT_DESC)); 
	g_scanner_add_symbol(scanner,"rot_beamwidth",       GINT_TO_POINTER(SYMBOL_ROT_BEAMWIDTH)); 
    
	g_scanner_add_symbol(scanner,"subwin",   	        GINT_TO_POINTER(SYMBOL_SW)); 
	g_scanner_add_symbol(scanner,"sw_type",   	        GINT_TO_POINTER(SYMBOL_SW_TYPE)); 
	g_scanner_add_symbol(scanner,"sw_command",   	    GINT_TO_POINTER(SYMBOL_SW_COMMAND)); 
	g_scanner_add_symbol(scanner,"sw_autorun",   	    GINT_TO_POINTER(SYMBOL_SW_AUTORUN)); 
	g_scanner_add_symbol(scanner,"sw_respawn_time",     GINT_TO_POINTER(SYMBOL_SW_RESPAWN_TIME)); 
    
	g_scanner_add_symbol(scanner,"loglines",         GINT_TO_POINTER(SYMBOL_LOGLINES)); 
	g_scanner_add_symbol(scanner,"skedshift",        GINT_TO_POINTER(SYMBOL_SKEDSHIFT)); 
	g_scanner_add_symbol(scanner,"takeoff",          GINT_TO_POINTER(SYMBOL_TAKEOFF)); 
    
	/* obsolete items */
    g_scanner_add_symbol(scanner,"cwda_device",    GINT_TO_POINTER(SYMBOL_OBSOLETE));
    g_scanner_add_symbol(scanner,"cwda_port",      GINT_TO_POINTER(SYMBOL_OBSOLETE));
    g_scanner_add_symbol(scanner,"cwda_speaker",   GINT_TO_POINTER(SYMBOL_OBSOLETE));
    g_scanner_add_symbol(scanner,"rota_type",   		    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
	g_scanner_add_symbol(scanner,"rota_port",   		    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
	g_scanner_add_symbol(scanner,"rota_saddr",   		    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
	g_scanner_add_symbol(scanner,"rota_timeout_ms",   	    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
	g_scanner_add_symbol(scanner,"rota_filename",   	    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
	g_scanner_add_symbol(scanner,"rota_hostname",   	    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
    g_scanner_add_symbol(scanner,"rotb_type",   		    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
	g_scanner_add_symbol(scanner,"rotb_port",   		    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
	g_scanner_add_symbol(scanner,"rotb_saddr",   		    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
	g_scanner_add_symbol(scanner,"rotb_timeout_ms",   	    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
	g_scanner_add_symbol(scanner,"rotb_filename",   	    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
	g_scanner_add_symbol(scanner,"rotb_hostname",   	    GINT_TO_POINTER(SYMBOL_OBSOLETE)); 
}
    


/**************************************************************************/

gint init_rc(void){
    cfg = g_new0(struct config, 1);
    
    cfg->bands      = g_ptr_array_new();
    cfg->cqs        = g_ptr_array_new();
    cfg->crotars    = g_ptr_array_new();
    cfg->sws        = g_ptr_array_new();
    cfg->takeoff    = g_ptr_array_new();
    return 0;
}

#define FREE_IF_EXISTS(base, item) if (base->item) g_free(base->item)

gint free_rc(void){
    int i;
    struct config_band *band;
    struct config_rotar *rot;

    
    FREE_IF_EXISTS(cfg,pcall);
    FREE_IF_EXISTS(cfg,pwwlo);
    FREE_IF_EXISTS(cfg,pexch);
    FREE_IF_EXISTS(cfg,padr1);
    FREE_IF_EXISTS(cfg,padr2);
    FREE_IF_EXISTS(cfg,pclub);
    FREE_IF_EXISTS(cfg,rname);
    FREE_IF_EXISTS(cfg,rcall);
    FREE_IF_EXISTS(cfg,radr1);
    FREE_IF_EXISTS(cfg,radr2);
    FREE_IF_EXISTS(cfg,rpoco);
    FREE_IF_EXISTS(cfg,rcity);
    FREE_IF_EXISTS(cfg,rcoun);
    FREE_IF_EXISTS(cfg,rphon);
    FREE_IF_EXISTS(cfg,rhbbs);
    FREE_IF_EXISTS(cfg,default_rs);
    FREE_IF_EXISTS(cfg,default_rst);
    
    FREE_IF_EXISTS(cfg,as_floppy_path);
    FREE_IF_EXISTS(cfg,as_mount_cmd);
    
    FREE_IF_EXISTS(cfg,net_if_ignore);
    FREE_IF_EXISTS(cfg,net_ip_ignore);
    FREE_IF_EXISTS(cfg,net_ip_announce);
    /* trace_xx (5x) */
    /* language */

    
    /* cwda-> in cwdaemon.c */

    for (i=0;i<cfg->bands->len; i++){
        band = g_ptr_array_index(cfg->bands, i);
        FREE_IF_EXISTS(band,pband);
        FREE_IF_EXISTS(band,stxeq);
        FREE_IF_EXISTS(band,spowe);
        FREE_IF_EXISTS(band,srxeq);
        FREE_IF_EXISTS(band,sante);
        FREE_IF_EXISTS(band,santh);
        FREE_IF_EXISTS(band,mope1);
        FREE_IF_EXISTS(band,mope2);
        FREE_IF_EXISTS(band,remarks);
        FREE_IF_EXISTS(band,ok_section_single);
        FREE_IF_EXISTS(band,ok_section_multi);
        FREE_IF_EXISTS(band,adifband);
        FREE_IF_EXISTS(band,skedqrg);
    }
    
    g_ptr_array_free_all(cfg->bands);
    
    for (i=0;i<cfg->cqs->len; i++){
        free_cq(g_ptr_array_index(cfg->cqs,i));
    }
    g_ptr_array_free(cfg->cqs, TRUE); /* YES, TRUE */

	/*TODO ostatni gchar *cfg->  ???*/
    
    for (i=0;i<cfg->crotars->len; i++){
        rot = g_ptr_array_index(cfg->crotars, i);
	    FREE_IF_EXISTS(rot,rot_filename);
	    FREE_IF_EXISTS(rot,rot_hostname);
	    FREE_IF_EXISTS(rot,rot_desc);
    }
    g_ptr_array_free_all(cfg->bands);
    
	
    for (i=0;i<cfg->sws->len;i++){
        struct config_subwin *csw;

        csw=(struct config_subwin *)g_ptr_array_index(cfg->sws, i);
        FREE_IF_EXISTS(csw, command);
        FREE_IF_EXISTS(csw, autorun);
        g_free(csw);
    }
    g_ptr_array_free(cfg->sws, TRUE); /* YES, TRUE */

    for (i=0;i<cfg->takeoff->len;i++){
        struct takeoff *toff;
        toff=(struct takeoff*)g_ptr_array_index(cfg->takeoff, i);
        g_free(toff);
    }
    g_ptr_array_free(cfg->takeoff, TRUE);
    g_free(cfg);
    return 0;
}


#define STORE_RC_STRING(base, SYM,item) case SYM: \
            if (base){\
                if (base->item) g_free(base->item); \
                base->item = g_strdup(scanner->value.v_string);\
            }\
            break

#define STORE_RC_STRING_UC(base, SYM,item) case SYM: \
            if (base){\
                if (base->item) g_free(base->item); \
                base->item = g_strdup(scanner->value.v_string);\
                uc(base->item);\
            }\
            break

#define STORE_RC_STRING_FS(base, SYM,item) case SYM: \
            if (base){\
                if (base->item) g_free(base->item); \
                base->item = g_strdup(scanner->value.v_string);\
                fixsemi(base->item);\
            }\
            break

#define STORE_RC_STRING_FS_UC(base, SYM,item) case SYM: \
            if (base){\
                if (base->item) g_free(base->item); \
                base->item = g_strdup(scanner->value.v_string);\
                fixsemi(base->item);\
                uc(base->item);\
            }\
            break

#define STORE_RC_INT(base, SYM,item) case SYM: \
            if (base){\
                base->item = scanner->value.v_int;\
            }\
            break

/* known items are identifiers */
/* all other are symbols */            
            
            
guint read_one_token(GScanner *scanner){
    guint symbol;

/*    dbg("read_one_token\n");*/
    g_scanner_get_next_token(scanner);
    symbol = scanner->token;

    /* symbol (abc) is unresolved, probably valid identifier from newer version */
    if (symbol == G_TOKEN_IDENTIFIER) return G_TOKEN_SYMBOL;
    
    
    /* config record must begin with known symbol */
    if ((symbol < SYMBOL_MY_FIRST) || (symbol > SYMBOL_MY_LAST)){
        return G_TOKEN_NONE;
    }
    
    /* its one of my symbols */
    g_scanner_get_next_token(scanner); /* expecting '=' */
    if (scanner->token != '='){
        return '=';
    }

    g_scanner_get_next_token(scanner);
/*    dbg("scanner->token=%d  G_TOKEN_STRING=%d \n",scanner->token, G_TOKEN_STRING);*/
    if (scanner->token==G_TOKEN_ERROR){
        return G_TOKEN_ERROR;
    }

/*    if ((scanner->token) != (G_TOKEN_STRING)){
        dbg("not equal\n");
        return G_TOKEN_STRING;
    }else{
        dbg("equal\n");
    }*/
    
    
    switch(symbol){

        STORE_RC_STRING_FS_UC(cfg, SYMBOL_PCALL, pcall);
        STORE_RC_STRING_FS_UC(cfg, SYMBOL_PWWLO, pwwlo);
        STORE_RC_STRING_FS_UC(cfg, SYMBOL_PEXCH, pexch);
        STORE_RC_STRING   (cfg, SYMBOL_PADR1, padr1);
        STORE_RC_STRING   (cfg, SYMBOL_PADR2, padr2);
        STORE_RC_STRING_FS_UC(cfg, SYMBOL_PCLUB, pclub);
        STORE_RC_STRING   (cfg, SYMBOL_RNAME, rname);
        STORE_RC_STRING_UC(cfg, SYMBOL_RCALL, rcall);
        STORE_RC_STRING   (cfg, SYMBOL_RADR1, radr1);
        STORE_RC_STRING   (cfg, SYMBOL_RADR2, radr2);
        STORE_RC_STRING   (cfg, SYMBOL_RPOCO, rpoco);
        STORE_RC_STRING   (cfg, SYMBOL_RCITY, rcity);
        STORE_RC_STRING   (cfg, SYMBOL_RCOUN, rcoun);
        STORE_RC_STRING   (cfg, SYMBOL_RPHON, rphon);
        STORE_RC_STRING   (cfg, SYMBOL_RHBBS, rhbbs);
        STORE_RC_STRING_UC(cfg, SYMBOL_RS,    default_rs);
        STORE_RC_STRING_UC(cfg, SYMBOL_RST,   default_rst);
        STORE_RC_INT      (cfg, SYMBOL_QSOP_M,qsop_method);
        STORE_RC_INT      (cfg, SYMBOL_TOTAL_M,total_method);
        
        STORE_RC_INT(cfg, SYMBOL_AS_DAQ, as_disk_aq);
        STORE_RC_INT(cfg, SYMBOL_AS_DAM, as_disk_am);
        STORE_RC_INT(cfg, SYMBOL_AS_DFS, as_disk_fsync);
        STORE_RC_INT(cfg, SYMBOL_AS_FAQ, as_floppy_aq);
        STORE_RC_INT(cfg, SYMBOL_AS_FAM, as_floppy_am);
        STORE_RC_INT(cfg, SYMBOL_AS_FFS, as_floppy_fsync);
        STORE_RC_INT(cfg, SYMBOL_AS_FMO, as_mount);
        STORE_RC_STRING(cfg, SYMBOL_AS_FPA, as_floppy_path);
        STORE_RC_STRING(cfg, SYMBOL_AS_FMC, as_mount_cmd);
        
        STORE_RC_STRING(cfg, SYMBOL_IF_IGNORE, net_if_ignore);
        STORE_RC_STRING(cfg, SYMBOL_IP_IGNORE, net_ip_ignore);
        STORE_RC_STRING(cfg, SYMBOL_IP_ANNOUNCE, net_ip_announce);
        STORE_RC_INT   (cfg, SYMBOL_TRACE_BCAST, trace_bcast);
        STORE_RC_INT   (cfg, SYMBOL_TRACE_SOCK, trace_sock);
        STORE_RC_INT   (cfg, SYMBOL_TRACE_RECV, trace_recv);
        STORE_RC_INT   (cfg, SYMBOL_TRACE_SEND, trace_send);
        STORE_RC_INT   (cfg, SYMBOL_TRACE_QSOS, trace_qsos);
        
        case SYMBOL_LANGUAGE:
            current_language = scanner->value.v_int;\
        break;

        case SYMBOL_CQ:
            cfg->cq = get_cq_by_number(cfg->cqs, scanner->value.v_int);
            if (!cfg->cq){
                cfg->cq = g_new0(struct cq, 1);
                g_ptr_array_add(cfg->cqs, cfg->cq);
                cfg->cq->nr = scanner->value.v_int;
            }  
            break;
            
        STORE_RC_STRING(cfg->cq, SYMBOL_CW_STR,      cw_str);
        STORE_RC_INT   (cfg->cq, SYMBOL_CW_SPEED,    cw_speed);
        STORE_RC_INT   (cfg->cq, SYMBOL_CW_REPEAT,   cw_repeat);
        STORE_RC_INT   (cfg->cq, SYMBOL_CW_TS,       cw_ts);
        STORE_RC_INT   (cfg->cq, SYMBOL_CW_ALLOWUND, cw_allowifundef);
        STORE_RC_STRING(cfg->cq, SYMBOL_SSB_FILE,    ssb_file);
        STORE_RC_INT   (cfg->cq, SYMBOL_SSB_REPEAT,  ssb_repeat);
        STORE_RC_INT   (cfg->cq, SYMBOL_SSB_TS,      ssb_ts);
        
        
		case SYMBOL_TERM:
			cfg->ts = new_term_spec(scanner->value.v_string);
			break;
		
		STORE_RC_INT(cfg->ts, SYMBOL_TERM_MODE, 		mode);	
		STORE_RC_INT(cfg->ts, SYMBOL_TERM_M11_HACK, 	m11_hack);	
		STORE_RC_INT(cfg->ts, SYMBOL_TERM_RESTRICT_852, restrict_852);	
		STORE_RC_INT(cfg->ts, SYMBOL_TERM_BLOCK_CURSOR, block_cursor);	
		STORE_RC_INT(cfg->ts, SYMBOL_TERM_COL, 			col);	
		STORE_RC_INT(cfg->ts, SYMBOL_TERM_UTF_8_IO,     utf_8_io);
		STORE_RC_INT(cfg->ts, SYMBOL_TERM_CHARSET, 		charset);	
			
        STORE_RC_STRING(cfg, SYMBOL_STARTBAND,    startband);
		
        case SYMBOL_PBAND:
            cfg->band = get_config_band_by_pband(fixsemi(scanner->value.v_string));
            if (!cfg->band){
                cfg->band = g_new0(struct config_band, 1);
                g_ptr_array_add(cfg->bands, cfg->band); 
                cfg->band->pband = fixsemi(g_strdup(scanner->value.v_string));
                cfg->band->skedqrg=g_strdup("");
            }
            /*dbg("new band: '%s' cfg->band='%p' \n", scanner->value.v_string,cfg->band);*/
            
            break;
        case SYMBOL_BANDCHAR: 
            if (cfg->band){
                cfg->band->bandchar = *(scanner->value.v_string);
            }
            break;
        case SYMBOL_QRV: 
            if (cfg->band){
                cfg->band->qrv = 
                cfg->band->qrvnow = scanner->value.v_int;
            }
            break;
        case SYMBOL_READONLY: 
            if (cfg->band){
                cfg->band->readonly = scanner->value.v_int;
                /*dbg("readonly=%d\n", cfg->band->readonly);*/
            }
            break;
/*        STORE_RC_INT      (cfg->band, SYMBOL_QRV,   qrv);    
        STORE_RC_INT      (cfg->band, SYMBOL_QRVNOW,qrvnow);    */
        STORE_RC_INT      (cfg->band, SYMBOL_PSECT, psect);    
        STORE_RC_STRING   (cfg->band, SYMBOL_STXEQ, stxeq);
        STORE_RC_STRING   (cfg->band, SYMBOL_SPOWE, spowe);
        STORE_RC_STRING   (cfg->band, SYMBOL_SRXEQ, srxeq);
        STORE_RC_STRING   (cfg->band, SYMBOL_SANTE, sante);
        STORE_RC_STRING   (cfg->band, SYMBOL_SANTH, santh);
        STORE_RC_STRING_UC(cfg->band, SYMBOL_MOPE1, mope1);
        STORE_RC_STRING_UC(cfg->band, SYMBOL_MOPE2, mope2);
        STORE_RC_STRING   (cfg->band, SYMBOL_REMARKS,remarks);
        STORE_RC_STRING   (cfg->band, SYMBOL_OK_SECTION_SINGLE,ok_section_single);
        STORE_RC_STRING   (cfg->band, SYMBOL_OK_SECTION_MULTI, ok_section_multi);
        STORE_RC_STRING   (cfg->band, SYMBOL_ADIFBAND, adifband);
        STORE_RC_INT      (cfg->band, SYMBOL_QRG_MIN, qrg_min);
        STORE_RC_INT      (cfg->band, SYMBOL_QRG_MAX, qrg_max);
        STORE_RC_STRING   (cfg->band, SYMBOL_SKEDQRG, skedqrg);

        STORE_RC_STRING   (cfg, SYMBOL_CWDA_HOST,    cwda_hostname);
        STORE_RC_INT      (cfg, SYMBOL_CWDA_UDP,     cwda_udp_port);
        STORE_RC_STRING   (cfg, SYMBOL_CWDA_DEVICE_S,  cwda_device_s);
        STORE_RC_INT      (cfg, SYMBOL_CWDA_SPEED,   cwda_speed);
        STORE_RC_INT      (cfg, SYMBOL_CWDA_WEIGHT,  cwda_weight);
        STORE_RC_INT      (cfg, SYMBOL_CWDA_SPKDEV,  cwda_spkdev);
        STORE_RC_INT      (cfg, SYMBOL_CWDA_SPKVOL,  cwda_spkvol);
        STORE_RC_INT      (cfg, SYMBOL_CWDA_PTTDELAY,cwda_pttdelay);
#if 0        
        STORE_RC_INT      (cfg, SYMBOL_CWDA_DEVICE,  cwda_device);  /* obsolete */
        STORE_RC_INT      (cfg, SYMBOL_CWDA_SPEAKER, cwda_speaker); /* obsolete */
        STORE_RC_INT      (cfg, SYMBOL_CWDA_PORT,    cwda_port);    /* obsolete */
#endif        
		STORE_RC_STRING   (cfg, SYMBOL_SSBD_HOST,       ssbd_hostname);
        STORE_RC_INT      (cfg, SYMBOL_SSBD_UDP,        ssbd_udp_port);
		STORE_RC_STRING   (cfg, SYMBOL_SSBD_DSP,        ssbd_dsp);
        STORE_RC_INT      (cfg, SYMBOL_SSBD_RECORD,     ssbd_record);
        STORE_RC_INT      (cfg, SYMBOL_SSBD_FORMAT,     ssbd_format);
        STORE_RC_INT      (cfg, SYMBOL_SSBD_CHANNELS,   ssbd_channels);
        STORE_RC_INT      (cfg, SYMBOL_SSBD_SAMPLERATE, ssbd_samplerate);
/*		STORE_RC_INT      (cfg, SYMBOL_SSBD_PIPE, 		ssbd_pipe);
		STORE_RC_STRING   (cfg, SYMBOL_SSBD_COMMAND, 	ssbd_command);*/
        STORE_RC_STRING   (cfg, SYMBOL_SSBD_TEMPLATE, 	ssbd_template);
        STORE_RC_STRING   (cfg, SYMBOL_SSBD_MIXER, 	    ssbd_mixer);
        STORE_RC_INT      (cfg, SYMBOL_SSBD_RECSRC, 	ssbd_recsrc);

        case SYMBOL_ROTAR:
            cfg->crotar = get_config_rotar_by_number(cfg->crotars, scanner->value.v_int);
            if (!cfg->crotar){
                cfg->crotar=g_new0(struct config_rotar, 1);
                g_ptr_array_add(cfg->crotars, cfg->crotar);
                cfg->crotar->nr = scanner->value.v_int;
            }
            break;
                
        STORE_RC_INT      (cfg->crotar, SYMBOL_ROT_TYPE, 		rot_type);
        STORE_RC_INT      (cfg->crotar, SYMBOL_ROT_PORT, 		rot_port);
        STORE_RC_INT      (cfg->crotar, SYMBOL_ROT_SADDR, 		rot_saddr);
        STORE_RC_INT      (cfg->crotar, SYMBOL_ROT_TIMEOUT_MS,   rot_timeout_ms);
        STORE_RC_STRING   (cfg->crotar, SYMBOL_ROT_FILENAME, 	rot_filename);
        STORE_RC_STRING   (cfg->crotar, SYMBOL_ROT_HOSTNAME, 	rot_hostname);
        STORE_RC_STRING   (cfg->crotar, SYMBOL_ROT_DESC,     	rot_desc);
        STORE_RC_INT      (cfg->crotar, SYMBOL_ROT_BEAMWIDTH,  	rot_beamwidth);
        
        
        case SYMBOL_SW:
            cfg->sw = get_config_sw_by_number(cfg->sws, scanner->value.v_int);
            if (!cfg->sw){
                cfg->sw = g_new0(struct config_subwin, 1);
                g_ptr_array_add(cfg->sws, cfg->sw);
                cfg->sw->nr = scanner->value.v_int;
            }  
            break;
        STORE_RC_INT      (cfg->sw, SYMBOL_SW_TYPE, 		type);
        STORE_RC_STRING   (cfg->sw, SYMBOL_SW_COMMAND, 		command);
        STORE_RC_STRING   (cfg->sw, SYMBOL_SW_AUTORUN, 		autorun);
        STORE_RC_INT      (cfg->sw, SYMBOL_SW_RESPAWN_TIME, respawn_time);
        
        STORE_RC_INT   (cfg, SYMBOL_LOGLINES, loglines);
        STORE_RC_INT   (cfg, SYMBOL_SKEDSHIFT, skedshift);
        case SYMBOL_TAKEOFF:{
                gchar **items;
                struct takeoff *toff;
                
                if (!scanner->value.v_string) break;
                items=g_strsplit(scanner->value.v_string, ",", 3);
                if (!items[0] || !items[1] || !items[2]) {
                    g_strfreev(items);
                    break;  
                }
                toff=g_new0(struct takeoff, 1);
                toff->from=atoi(items[0]);
                toff->to=atoi(items[1]);
                toff->value=atoi(items[2]);
                g_ptr_array_add(cfg->takeoff, toff); 
            
            }
            break;
        

    }
    /* force logging */
#if 0
    cfg->trace_sock=1;
    cfg->trace_qsos=1;
#endif    
    
    return G_TOKEN_NONE;
}

gint read_rc_file(gchar *filename){
   FILE *f;
   GScanner *scanner;
   guint expected;
   gint errors;

/*  dbg("read_rc_file(%s)\n", filename);*/
   f=fopen(filename,"rt");
   if (!f) return -1;

   scanner = g_scanner_new(NULL);
   
   scanner->config->numbers_2_int   = TRUE;
   scanner->config->symbol_2_token  = TRUE;

   add_symbols(scanner);
   
   g_scanner_input_file(scanner, fileno(f));
   scanner->input_name = filename;
 
   errors=0;
   do{
       expected = read_one_token(scanner);
       if (expected == G_TOKEN_SYMBOL)  errors++;
       
       /*dbg("expected=%d\n",expected);*/
       g_scanner_peek_next_token(scanner);
       
   } while (scanner->next_token != G_TOKEN_EOF &&
            scanner->next_token != G_TOKEN_ERROR);


   g_scanner_destroy(scanner);    
   fclose(f); 
/*   dbg("call %s\n", cfg->call);*/
   if (errors){
       log_addf(TEXT(T_ERR_RC_DSS),errors, filename);
       dbg("Found %d error%s while reading %s\n",errors, errors>1?"s":"",filename);
       return errors;
   }
   return 0;
}

gint read_rc_line(gchar *str){
   GScanner *scanner;
   guint expected;
   gint errors;

   scanner = g_scanner_new(NULL);
   
   scanner->config->numbers_2_int   = TRUE;
   scanner->config->symbol_2_token  = TRUE;

   add_symbols(scanner);
   
   g_scanner_input_text(scanner, str, strlen(str));
   scanner->input_name = "network";
 
   errors=0;
   do{
       expected = read_one_token(scanner);
       if (expected == G_TOKEN_SYMBOL)  errors++;
       
       /*dbg("expected=%d\n",expected);*/
       g_scanner_peek_next_token(scanner);
       
   } while (expected == G_TOKEN_NONE &&
            scanner->next_token != G_TOKEN_EOF &&
            scanner->next_token != G_TOKEN_ERROR);

   if (expected != G_TOKEN_NONE){
       g_scanner_unexp_token(scanner, expected, NULL, "symbol", NULL, NULL, TRUE);
   };

   g_scanner_destroy(scanner);    
   if (errors){
       log_addf(TEXT(T_ERR_RC_DS),errors );
   }
   return 0;
}


void terminate_bh(void *xxx){
    terminate=1;
}

void show_rc_error(void *xxx){
    msg_box(term, NULL, 
            TEXT(T_ERROR), AL_CENTER, 
            TEXT(T_CANT_LOAD_CFG), NULL, 1, 
            TEXT(T_CANCEL), terminate_bh, B_ENTER | B_ESC);
}

void show_rc_warning(void *xxx){
    msg_box(term, NULL, 
            TEXT(T_WARNING), AL_CENTER, 
            TEXT(T_ERR_RC), NULL, 1, 
            TEXT(T_CANCEL), NULL, B_ENTER | B_ESC);
}

gint read_rc_files(void){
    gchar *user,*local;
    int fail1,fail2,fail3;
    
    cfg->loglines=3;
    cfg->skedshift=5;
    
    fail1=0;
    
    user = g_strconcat(getenv("HOME"), RC_USER, NULL);
    fail2=read_rc_file(user);
    g_free(user);      
    
    if (fail2==-1){  /* reads global config only if no user config exists */
      fail1=read_rc_file(RC_MAIN);
    }
    
    local = g_strconcat(getenv("HOME"), RC_LOCAL, NULL);
    fail3=read_rc_file(local);
    g_free(local);      
    
    
    
    if (!cfg->pcall || !cfg->pwwlo) 
        register_bottom_half(show_rc_error, NULL);
    else
        if (fail1>0 || fail2>0 || fail3>0) register_bottom_half(show_rc_warning, NULL);
        
        
    
    return 0;
}


#define S(item) safe_strncpy0(ss, item, 1024)
#define U(item) uc(S(item))

void save_rc_string(GString *gs){    
    char ss[1026];
	struct term_spec *t;
    int i;
    
    g_string_sprintfa(gs, "# This file is generated automatically. Edit only at your own risk!\n\n");
    g_string_sprintfa(gs, "pcall = \"%s\"\n", U(cfg->pcall));
    g_string_sprintfa(gs, "pwwlo = \"%s\"\n", U(cfg->pwwlo));
    g_string_sprintfa(gs, "pexch = \"%s\"\n", U(cfg->pexch));
    g_string_sprintfa(gs, "\n");
    g_string_sprintfa(gs, "padr1 = \"%s\"\n", S(cfg->padr1));
    g_string_sprintfa(gs, "padr2 = \"%s\"\n", S(cfg->padr2));
    g_string_sprintfa(gs, "pclub = \"%s\"\n", U(cfg->pclub));
    g_string_sprintfa(gs, "\n");
    g_string_sprintfa(gs, "rname = \"%s\"\n", S(cfg->rname));
    g_string_sprintfa(gs, "rcall = \"%s\"\n", U(cfg->rcall));
    g_string_sprintfa(gs, "radr1 = \"%s\"\n", S(cfg->radr1));
    g_string_sprintfa(gs, "radr2 = \"%s\"\n", S(cfg->radr2));
    g_string_sprintfa(gs, "rpoco = \"%s\"\n", S(cfg->rpoco));
    g_string_sprintfa(gs, "rcity = \"%s\"\n", S(cfg->rcity));
    g_string_sprintfa(gs, "rcoun = \"%s\"\n", S(cfg->rcoun));
    g_string_sprintfa(gs, "rphon = \"%s\"\n", S(cfg->rphon));
    g_string_sprintfa(gs, "rhbbs = \"%s\"\n", S(cfg->rhbbs));
    g_string_sprintfa(gs, "\n");
    g_string_sprintfa(gs, "default_rs   = \"%s\"\n", U(cfg->default_rs));
    g_string_sprintfa(gs, "default_rst  = \"%s\"\n", U(cfg->default_rst));
    g_string_sprintfa(gs, "qsop_method  = %d\n", cfg->qsop_method);
    g_string_sprintfa(gs, "total_method = %d\n", cfg->total_method);
    g_string_sprintfa(gs, "\n");
    g_string_sprintfa(gs, "as_disk_qso     = %d\n", cfg->as_disk_aq);
    g_string_sprintfa(gs, "as_disk_min     = %d\n", cfg->as_disk_am);
    g_string_sprintfa(gs, "as_disk_fsync   = %d\n", cfg->as_disk_fsync);
    g_string_sprintfa(gs, "as_floppy_qso   = %d\n", cfg->as_floppy_aq);
    g_string_sprintfa(gs, "as_floppy_min   = %d\n", cfg->as_floppy_am);
    g_string_sprintfa(gs, "as_floppy_fsync = %d\n", cfg->as_floppy_fsync);
    g_string_sprintfa(gs, "as_mount_floppy = %d\n", cfg->as_mount);
    g_string_sprintfa(gs, "as_floppy_path  = \"%s\"\n", S(cfg->as_floppy_path));
    g_string_sprintfa(gs, "as_mount_cmd    = \"%s\"\n", S(cfg->as_mount_cmd));
    g_string_sprintfa(gs, "\n");
    g_string_sprintfa(gs, "net_if_ignore   = \"%s\"\n", S(cfg->net_if_ignore));
    g_string_sprintfa(gs, "net_ip_ignore   = \"%s\"\n", S(cfg->net_ip_ignore));
    g_string_sprintfa(gs, "net_ip_announce = \"%s\"\n", S(cfg->net_ip_announce));
    g_string_sprintfa(gs, "trace_bcast = %d\n", cfg->trace_bcast);
    g_string_sprintfa(gs, "trace_sock  = %d\n", cfg->trace_send);
    g_string_sprintfa(gs, "trace_recv  = %d\n", cfg->trace_recv);
    g_string_sprintfa(gs, "trace_send  = %d\n", cfg->trace_send);
    g_string_sprintfa(gs, "trace_qsos  = %d\n", cfg->trace_qsos);
    g_string_sprintfa(gs, "language= %d\n", current_language);
    g_string_sprintfa(gs, "\n");
    g_string_sprintfa(gs, "cwda_hostname = \"%s\"\n", U(cfg->cwda_hostname));
    g_string_sprintfa(gs, "cwda_udp_port = %d\n", cfg->cwda_udp_port);
    g_string_sprintfa(gs, "cwda_device_s = \"%s\"\n", S(cfg->cwda_device_s));
    g_string_sprintfa(gs, "cwda_speed    = %d\n", cfg->cwda_speed);
    g_string_sprintfa(gs, "cwda_weight   = %d\n", cfg->cwda_weight);
    g_string_sprintfa(gs, "cwda_spkdev   = %d\n", cfg->cwda_spkdev);
    g_string_sprintfa(gs, "cwda_spkvol   = %d\n", cfg->cwda_spkvol);
    g_string_sprintfa(gs, "cwda_pttdelay = %d\n", cfg->cwda_pttdelay);
    /*g_string_sprintfa(gs, "cwda_device   = %d\n", cfg->cwda_device);*/
    /*g_string_sprintfa(gs, "cwda_speaker  = %d\n", cfg->cwda_speaker);*/
    /*g_string_sprintfa(gs, "cwda_port     = 0x%x\n", cfg->cwda_port);*/
    g_string_sprintfa(gs, "\n");
    g_string_sprintfa(gs, "ssbd_hostname   = \"%s\"\n", S(cfg->ssbd_hostname));
    g_string_sprintfa(gs, "ssbd_udp_port   = %d\n", cfg->ssbd_udp_port);
    g_string_sprintfa(gs, "ssbd_dsp        = \"%s\"\n", S(cfg->ssbd_dsp));
    g_string_sprintfa(gs, "ssbd_record     = %d\n", cfg->ssbd_record);
    g_string_sprintfa(gs, "ssbd_format     = 0x%x\n", cfg->ssbd_format);
    g_string_sprintfa(gs, "ssbd_channels   = %d\n", cfg->ssbd_channels);
    g_string_sprintfa(gs, "ssbd_samplerate = %d\n", cfg->ssbd_samplerate);
/*    g_string_sprintfa(gs, "ssbd_pipe       = %d\n", cfg->ssbd_pipe);
    g_string_sprintfa(gs, "ssbd_command    = \"%s\"\n", cfg->ssbd_command);*/
    g_string_sprintfa(gs, "ssbd_template   = \"%s\"\n", S(cfg->ssbd_template));
    g_string_sprintfa(gs, "ssbd_mixer      = \"%s\"\n", S(cfg->ssbd_mixer));
    g_string_sprintfa(gs, "ssbd_recsrc     = 0x%x\n", cfg->ssbd_recsrc);
    g_string_sprintfa(gs, "\n\n");
    
    for (i=0; i<cfg->crotars->len; i++){
        struct config_rotar *crot;

        crot = (struct config_rotar *)g_ptr_array_index(cfg->crotars, i);
        g_string_sprintfa(gs, "ROTAR = %d\n", crot->nr);
        g_string_sprintfa(gs, "    rot_type       = %d\n", crot->rot_type);
        g_string_sprintfa(gs, "    rot_port       = %d\n", crot->rot_port);
        g_string_sprintfa(gs, "    rot_saddr      = %d\n", crot->rot_saddr);
        g_string_sprintfa(gs, "    rot_timeout_ms = %d\n", crot->rot_timeout_ms);
        g_string_sprintfa(gs, "    rot_filename   = \"%s\"\n", S(crot->rot_filename));
        g_string_sprintfa(gs, "    rot_hostname   = \"%s\"\n", S(crot->rot_hostname));
        g_string_sprintfa(gs, "    rot_desc       = \"%s\"\n", S(crot->rot_desc));
        g_string_sprintfa(gs, "    rot_beamwidth  = %d\n", crot->rot_beamwidth);
        g_string_sprintfa(gs, "\n");
    }
    g_string_sprintfa(gs, "\n");
	g_string_sprintfa(gs, "loglines        = %d\n", cfg->loglines);
	g_string_sprintfa(gs, "skedshift       = %d\n", cfg->skedshift);
    g_string_sprintfa(gs, "startband       = \"%s\"\n", S(cfg->startband));
    for (i=0; i<cfg->takeoff->len;i++){
        struct takeoff *toff;
        toff=(struct takeoff*)g_ptr_array_index(cfg->takeoff, i);
        g_string_sprintfa(gs, "takeoff         = \"%3d,%3d,%2d\"\n", toff->from, toff->to, toff->value);

    }
    g_string_sprintfa(gs, "\n\n");

    for (i=0; i<cfg->cqs->len; i++){
        struct cq *cq;

        cq = (struct cq *) g_ptr_array_index(cfg->cqs, i);
        g_string_sprintfa(gs, "CQ = %d\n", i);
        g_string_sprintfa(gs, "    cw_str          = \"%s\"\n", S(cq->cw_str));
        g_string_sprintfa(gs, "    cw_speed        = %d\n", cq->cw_speed);
        g_string_sprintfa(gs, "    cw_repeat       = %d\n", cq->cw_repeat);
        g_string_sprintfa(gs, "    cw_ts           = %d\n", cq->cw_ts);
        g_string_sprintfa(gs, "    cw_allowifundef = %d\n", cq->cw_allowifundef);
        g_string_sprintfa(gs, "    ssb_file        = \"%s\"\n", S(cq->ssb_file));
        g_string_sprintfa(gs, "    ssb_ts          = %d\n", cq->ssb_ts);
        g_string_sprintfa(gs, "    ssb_repeat      = %d\n", cq->ssb_repeat);
        g_string_sprintfa(gs, "\n");
    }
    g_string_sprintfa(gs, "\n");
    
    foreach(t, term_specs){
		g_string_sprintfa(gs, "TERM = \"%s\"\n", S(t->term));
		g_string_sprintfa(gs, "    term_mode         = %d\n", t->mode);
		g_string_sprintfa(gs, "    term_m11_hack     = %d\n", t->m11_hack);
		g_string_sprintfa(gs, "    term_restrict_852 = %d\n", t->restrict_852);
		g_string_sprintfa(gs, "    term_col          = %d\n", t->col);
		g_string_sprintfa(gs, "    term_utf_8_io     = %d\n", t->utf_8_io);
		g_string_sprintfa(gs, "    term_charset      = %d\n", t->charset);
    	g_string_sprintfa(gs, "\n");
	}	
    g_string_sprintfa(gs, "\n");
    
    for (i=0; i<gses->subwins->len; i++){
        struct subwin *sw;

        sw = (struct subwin *) g_ptr_array_index(gses->subwins, i);
        g_string_sprintfa(gs, "SUBWIN = %d\n", i);
        g_string_sprintfa(gs, "    sw_type         = %d\n", sw->type);
        g_string_sprintfa(gs, "    sw_command      = \"\"\n");
        g_string_sprintfa(gs, "    sw_autorun      = \"\"\n");
        g_string_sprintfa(gs, "    sw_respawn_time = 5\n");
        g_string_sprintfa(gs, "\n");
    }	
    g_string_sprintfa(gs, "\n");
    
    for (i=0; i<cfg->bands->len; i++){
        struct config_band *b;

        b = (struct config_band *) g_ptr_array_index(cfg->bands, i);
        g_string_sprintfa(gs, "PBAND = \"%s\"\n", S(b->pband));
        g_string_sprintfa(gs, "    bandchar = \"%c\"\n", b->bandchar);
        g_string_sprintfa(gs, "    qrv = %d\n", b->qrv);
        g_string_sprintfa(gs, "    psect = %d\n", b->psect);
        g_string_sprintfa(gs, "    readonly = %d\n", b->readonly);
        g_string_sprintfa(gs, "\n");
        g_string_sprintfa(gs, "    stxeq = \"%s\"\n", S(b->stxeq));
        g_string_sprintfa(gs, "    spowe = \"%s\"\n", S(b->spowe));
        g_string_sprintfa(gs, "    srxeq = \"%s\"\n", S(b->srxeq));
        g_string_sprintfa(gs, "    sante = \"%s\"\n", S(b->sante));
        g_string_sprintfa(gs, "    santh = \"%s\"\n", S(b->santh));
        g_string_sprintfa(gs, "    mope1 = \"%s\"\n", S(b->mope1));
        g_string_sprintfa(gs, "    mope2 = \"%s\"\n", S(b->mope2));
        g_string_sprintfa(gs, "    remarks = \"%s\"\n", S(b->remarks));
        g_string_sprintfa(gs, "\n");
        g_string_sprintfa(gs, "    ok_section_single = \"%s\"\n", S(b->ok_section_single));
        g_string_sprintfa(gs, "    ok_section_multi  = \"%s\"\n", S(b->ok_section_multi));
        g_string_sprintfa(gs, "    qrg_min   = %d\n", b->qrg_min);
        g_string_sprintfa(gs, "    qrg_max   = %d\n", b->qrg_max);
        g_string_sprintfa(gs, "    adifband  = \"%s\"\n", S(b->adifband));
        g_string_sprintfa(gs, "    skedqrg   = \"%s\"\n", S(b->skedqrg));
        g_string_sprintfa(gs, "\n");
    }
}

gint save_rc_file(gchar *filename){
    FILE *f;
    GString *gs;
    int ret;
    
    gs = g_string_sized_new(10000);
    
    f=fopen(filename,"wt");
    if (!f) {
/*        c = g_strconcat("Can't open file '", filename, "'", NULL);
        errbox(c,0);
        g_free(c);*/
        return errno;
    }

    save_rc_string(gs);
    ret = fprintf(f, "%s", gs->str) != gs->len ? errno:0;
    fclose(f);
    g_string_free(gs, TRUE);

    return ret;
}

int term_spec_init(void){
    struct term_spec *ts;
    
    if (!(ts = new_term_spec("linux"))) return -1;
    ts->mode = 1;
    ts->m11_hack = 1;
    ts->restrict_852 = 0;
    ts->col = 1;
    ts->utf_8_io = 0;
    ts->charset = 2;
    
    if (!(ts = new_term_spec("xterm"))) return -1;
    ts->mode = 1;
    ts->m11_hack = 1;
    ts->restrict_852 = 0;
    ts->col = 1;
    ts->utf_8_io = 0;
    ts->charset = 2;
    
    if (!(ts = new_term_spec("screen"))) return -1;
    ts->mode = 1;
    ts->m11_hack = 1;
    ts->restrict_852 = 0;
    ts->col = 1;
    ts->utf_8_io = 0;
    ts->charset = 2;
    
    if (!(ts = new_term_spec("cygwin"))) return -1;
    ts->mode = 2;
    ts->m11_hack = 1;
    ts->restrict_852 = 0;
    ts->col = 1;
    ts->utf_8_io = 0;
    ts->charset = 2;
    
    return 0;
}


char *parse_options(int argc, char **argv){
    int c;
    char *s;

    s=getenv("TUCNAK_DEBUG");
    if (s){
        if (strlen(s)!=0){
            debug_type=1;
            debug_filename=s;
        }else{
            debug_type=2;
        }
    }
#ifdef HAVE_GETOPT_H    
    
    while (1){
        
        int option_index = 0;
        static struct option long_options[] = {
            {"debug", 2, 0, 'd'},
            {"help", 0, 0, 'h'},
            {"version", 0, 0, 'v'},
            {0, 0, 0, 0}
        };

        c = getopt_long (argc, argv, "dhv?", long_options, &option_index);
        if (c == -1) break;

        switch (c){
            case 0:
                printf ("option %s", long_options[option_index].name);
                if (optarg) printf (" with arg %s", optarg);
                printf ("\n");
                break;

           case 'd':
                if (optarg) {
                    debug_type=1;
                    debug_filename=optarg;
                }else{
                    debug_type=2;
                }
                break;

           case 'h':     
           case '?':
                printf ("option h\n");
                break;
           
           case 'v':
                printf ("option v\n");
                break;


           default:
                printf ("?? getopt returned character code 0%o ??\n", c);
        }
    }

    if (optind < argc){
        printf ("non-option ARGV-elements: ");
        while (optind < argc) printf ("%s ", argv[optind++]);
        printf ("\n");
    }

#endif    
    return NULL;
}

