/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "issuebatch.h"

#include <QFile>

#include "commands/command.h"
#include "datamanager.h"

using namespace WebIssues;

IssueBatch::IssueBatch( int issueId ) : AbstractBatch( 0 ),
    m_issueId( issueId ),
    m_update( false ),
    m_saveFile( NULL ),
    m_fileError( QFile::NoError )
{
}

IssueBatch::IssueBatch( int folderId, const QString& name ) : AbstractBatch( 0 ),
    m_issueId( 0 ),
    m_update( false ),
    m_saveFile( NULL ),
    m_fileError( QFile::NoError )
{
    Job job( &IssueBatch::addIssueJob );
    job.addArg( folderId );
    job.addArg( name );
    m_queue.addJob( job );
}

IssueBatch::~IssueBatch()
{
    if ( m_saveFile && m_saveFile->isOpen() )
        m_saveFile->remove();

    delete m_saveFile;
}

void IssueBatch::renameIssue( const QString& newName )
{
    Job job( &IssueBatch::renameIssueJob );
    job.addArg( newName );
    m_queue.addJob( job );
}

void IssueBatch::setValue( int attributeId, const QString& newValue )
{
    Job job( &IssueBatch::setValueJob );
    job.addArg( attributeId );
    job.addArg( newValue );
    m_queue.addJob( job );
}

void IssueBatch::addComment( const QString& text )
{
    Job job( &IssueBatch::addCommentJob );
    job.addArg( text );
    m_queue.addJob( job );
}

void IssueBatch::addAttachment( const QString& name, const QString& description, const QString& path )
{
    Job job( &IssueBatch::addAttachmentJob );
    job.addArg( name );
    job.addArg( description );
    job.addArg( path );
    m_queue.addJob( job );
}

void IssueBatch::getAttachment( int attachmentId, const QString& path )
{
    Job job( &IssueBatch::getAttachmentJob );
    job.addArg( attachmentId );
    job.addArg( path );
    m_queue.addJob( job );
}

Command* IssueBatch::fetchNext()
{
    if ( m_fileError != QFile::NoError )
        return NULL;

    delete m_saveFile;
    m_saveFile = NULL;

    if ( m_queue.moreJobs() )
        return m_queue.callJob( this );

    if ( m_update ) {
        m_update = false;
        return dataManager->updateIssue( m_issueId );
    }

    return NULL;
}

Command* IssueBatch::addIssueJob( const Job& job )
{
    Command* command = new Command();

    command->setKeyword( "ADD ISSUE" );
    command->addArg( job.argInt( 0 ) );
    command->addArg( job.argString( 1 ) );

    command->addRule( "ID i", ReplyRule::One );

    connect( command, SIGNAL( commandReply( const Reply& ) ), this, SLOT( addIssueReply( const Reply& ) ) );

    return command;
}

Command* IssueBatch::renameIssueJob( const Job& job )
{
    Command* command = new Command();

    command->setKeyword( "RENAME ISSUE" );
    command->addArg( m_issueId );
    command->addArg( job.argString( 0 ) );

    command->setAcceptNullReply( true );
    command->addRule( "ID i", ReplyRule::One );

    connect( command, SIGNAL( commandReply( const Reply& ) ), this, SLOT( setUpdate() ) );

    return command;
}

Command* IssueBatch::setValueJob( const Job& job )
{
    Command* command = new Command();

    command->setKeyword( "SET VALUE" );
    command->addArg( m_issueId );
    command->addArg( job.argInt( 0 ) );
    command->addArg( job.argString( 1 ) );

    command->setAcceptNullReply( true );
    command->addRule( "ID i", ReplyRule::One );

    connect( command, SIGNAL( commandReply( const Reply& ) ), this, SLOT( setUpdate() ) );

    return command;
}

Command* IssueBatch::addCommentJob( const Job& job )
{
    Command* command = new Command();

    command->setKeyword( "ADD COMMENT" );
    command->addArg( m_issueId );
    command->addArg( job.argString( 0 ) );

    command->addRule( "ID i", ReplyRule::One );

    connect( command, SIGNAL( commandReply( const Reply& ) ), this, SLOT( setUpdate() ) );

    return command;
}

Command* IssueBatch::addAttachmentJob( const Job& job )
{
    QString path = job.argString( 2 );

    QFile file( path );

    if ( !file.open( QIODevice::ReadOnly | QIODevice::Unbuffered ) ) {
        m_fileError = file.error();
        return NULL;
    }

    QByteArray data = file.readAll();

    if ( file.error() != QFile::NoError ) {
        m_fileError = file.error();
        return NULL;
    }

    m_fileSize = data.size();

    Command* command = new Command();

    command->setKeyword( "ADD ATTACHMENT" );
    command->addArg( m_issueId );
    command->addArg( job.argString( 0 ) );
    command->addArg( job.argString( 1 ) );

    command->setAttachment( data );

    command->addRule( "ID i", ReplyRule::One );

    connect( command, SIGNAL( commandReply( const Reply& ) ), this, SLOT( setUpdate() ) );
    connect( command, SIGNAL( sendProgress( int, int ) ), this, SLOT( sendProgress( int, int ) ) );

    return command;
}

Command* IssueBatch::getAttachmentJob( const Job& job )
{
    QString path = job.argString( 1 );

    m_saveFile = new QFile( path );

    if ( !m_saveFile->open( QIODevice::WriteOnly | QIODevice::Unbuffered ) ) {
        m_fileError = m_saveFile->error();
        return NULL;
    }

    Command* command = new Command();

    command->setKeyword( "GET ATTACHMENT" );
    command->addArg( job.argInt( 0 ) );

    command->setAcceptBinaryResponse( true );

    connect( command, SIGNAL( binaryBlock( const char *, int ) ), this, SLOT( saveBlock( const char *, int ) ) );
    connect( command, SIGNAL( readProgress( int, int ) ), this, SLOT( readProgress( int, int ) ) );

    return command;
}

void IssueBatch::addIssueReply( const Reply& reply )
{
    ReplyLine line = reply.lines().at( 0 );
    m_issueId = line.argInt( 0 );

    m_update = true;
}

void IssueBatch::saveBlock( const char* data, int length )
{
    if ( m_fileError != QFile::NoError )
        return;

    if ( m_saveFile->write( data, length ) < 0 )
        m_fileError = m_saveFile->error();
}

void IssueBatch::sendProgress( int done, int total )
{
    int sent = done - ( total - m_fileSize );
    if ( sent > 0 )
        emit uploadProgress( sent );
}

void IssueBatch::readProgress( int done, int total )
{
    emit downloadProgress( done );
}

void IssueBatch::setUpdate()
{
    m_update = true;
}

#include "issuebatch.moc"
