/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "view.h"

#include "commands/commandmanager.h"
#include "data/datamanager.h"
#include "data/updatebatch.h"
#include "xmlui/builder.h"
#include "viewevent.h"

using namespace WebIssues;

View::View( QObject* parent ) : QObject( parent ),
    m_mainWidget( NULL ),
    m_id( 0 ),
    m_access( UnknownAccess ),
    m_enabled( false ),
    m_updating( false )
{
    dataManager->addObserver( this );
}

View::~View()
{
    if ( dataManager )
        dataManager->removeObserver( this );

    delete m_mainWidget;
}

void View::setMainWidget( QWidget* widget )
{
    m_mainWidget = widget;
}

void View::setViewerSizeHint( const QSize& size )
{
    m_sizeHint = size;
}

void View::setAccess( Access access, bool initial )
{
    if ( initial || m_access != access ) {
        m_access = access;

        if ( ( initial || !m_enabled ) && ( access == NormalAccess || access == AdminAccess ) ) {
            m_enabled = true;
            enableView();
            emit enabledChanged( true );
            emit showInfo( tr( "OK" ) );
        } else if ( ( initial || m_enabled ) && ( access == UnknownAccess || access == NoAccess ) ) {
            m_enabled = false;
            emit enabledChanged( false );
            emit showSummary( QPixmap(), QString() );
            disableView();
        }

        if ( access == NormalAccess || access == AdminAccess ) {
            updateAccess( access );
            if ( builder() )
                builder()->rebuildAll();
        }

        if ( access == UnknownAccess )
            emit showWarning( tr( "Please wait until the data is updated" ) );
        else if ( access == NoAccess )
            emit showWarning( tr( "You don't have permissions to access this view" ) );
    }
}

void View::enableView()
{
}

void View::disableView()
{
}

void View::updateAccess( Access access )
{
}

void View::updateEvent( UpdateEvent* e )
{
}

void View::viewEvent( ViewEvent* e )
{
}

bool View::queryClose()
{
    return true;
}

void View::setCaption( const QString& text )
{
    m_caption = text;
    emit captionChanged( text );
}

void View::customEvent( QEvent* e )
{
    if ( e->type() == UpdateEvent::Type )
        updateEvent( (UpdateEvent*)e );
    else if ( e->type() == ViewEvent::Type )
        viewEvent( (ViewEvent*)e );
}

void View::initialUpdate()
{
}

void View::executeUpdate( UpdateBatch* batch )
{
    showBusy( tr( "Updating the view..." ) );

    connect( batch, SIGNAL( completed( bool ) ), this, SLOT( updateCompleted( bool ) ) );

    commandManager->execute( batch );

    m_updating = true;
}

void View::updateCompleted( bool successful )
{
    if ( successful )
        showInfo( tr( "Update successful" ) );
    else
        showWarning( commandManager->errorMessage( tr( "Update failed" ) ) );

    m_updating = false;
}

#include "view.moc"
