/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#ifndef _WF_CONFIG_H
#define _WF_CONFIG_H

#include <iostream>
#include <string>
#include <map>

#include <getopt.h>

enum wf_option_types {
  WF_OPTION_TYPE_NONE,
  WF_OPTION_TYPE_INT,
  WF_OPTION_TYPE_STR,
  WF_OPTION_TYPE_BOOL
};

class wf_option {
 public:
  wf_option();
  wf_option(enum wf_option_types type);
  virtual ~wf_option();
  wf_option(const wf_option& option);
  wf_option& operator=(const wf_option& option);

  bool set(int integer);
  bool set(const char* str);
  bool set(bool boolean);

  wf_option& operator=(int integer);
  wf_option& operator=(const char* string);
  wf_option& operator=(bool boolean);

  int toint() const;
  string tostr() const;
  bool tobool() const;

  enum wf_option_types type() const;
  string type_tostr() const;

  ostream& print(ostream& os) const;

  void short_help(const string& short_help);
  void long_help(const string& long_help);
  string short_help() const;
  string long_help() const;


 private:
  enum wf_option_types _type;
  union {
    bool boolean;
    char* str;
    int integer;
  } _val;
  bool _defined;
  string _short_help;
  string _long_help;
};

struct compare_strings {
  bool operator()(const char* s1_, const char* s2_) const {
    return strcmp (s1_, s2_) < 0;
  }
};

enum wf_config_parse_ret {
  WF_CONFIG_PARSE_OK,
  WF_CONFIG_PARSE_UNKNOWN,
  WF_CONFIG_PARSE_ARG,
  WF_CONFIG_PARSE_OTHER
};

/**
 * 
 */
class wf_config {
 public:
  wf_config();
  virtual ~wf_config();
  // wf_config(const wf_config&); // what about the copy constructor? RV@@5

  bool option_add(const char* name, enum wf_option_types type,
		  const string& short_help, const string& long_help = "");

  bool option_set(const char* name, int a);
  bool option_set(const char* name, const char* str);
  bool option_set(const char* name, bool boolean);

  wf_option& option_get(const char* name);
  wf_option& operator[](const char* name);

  enum wf_config_parse_ret parse(int argc, char* const argv[]);

  template <class _Function>
  _Function for_each(_Function __f) const {
    return std::for_each(_names.begin(), _names.end(), __f);
  }

  ostream& debugprint(ostream& os) const;

 private:
  map<const char*, wf_option, compare_strings> _names;
  struct option* long_options;
};

#endif
