#include <errno.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/param.h>

#include "common.h"
#include "sigctl.h"
#include "config.h"
#include "logfile.h"


LogFile::LogFile()  { 
  h=-1; filename=0;
  registerSignalHandler(this,termFunc,signalTerm);
  registerSignalHandler(this,hupFunc,signalHup);
}

LogFile::~LogFile() { 
  unregisterSignalHandler(this,termFunc); 
  unregisterSignalHandler(this,hupFunc); 
  delete[] filename;
  if(h!=-1) close(h);
}

void LogFile::termFunc(void *l) {
  ((LogFile*)l)->log(status,"Server terminated because of received signal");
}

void LogFile::open() {
  h=::open(filename,O_WRONLY|O_APPEND);
  if(h==-1) {
    // Create the file:
    h=::open(filename,O_CREAT|O_WRONLY|O_APPEND,
      S_IRUSR|S_IWUSR|
      S_IRGRP|S_IWGRP|
      S_IROTH);
    
    if(h!=-1) fchown(h,own,grp);
  }
}

void LogFile::hupFunc(void *lo) {
  LogFile* l=(LogFile*)lo;
  
  // Reopen logfile:
  if(l->h!=-1) {
    l->log(status,"HUP signal received, closing logfile");
    close(l->h);
  }
  l->open();
  l->log(status,"HUP signal received, logfile reopened");
}

char* LogFile::init(Config* cnf, int ps) {
  prnstderr=ps;
  own=cnf->getuid();
  grp=cnf->getgid();
  filename=new char[strlen(cnf->logfile())+1];
  strcpy(filename,cnf->logfile());

  blockSignals(); // Don't allow a SIGHUP signal to reopen the logfile now
  open();
  if(h==-1) {
    snprintf(errmsg,sizeof(errmsg),"Could not create logfile %s: %s",cnf->logfile(),strerror(errno));
    unblockSignals();
    return errmsg;
  }

  logaddrchange=cnf->logaddrchange();
  logbadpackets=cnf->logbadpackets();
  logtablefull=cnf->logtablefull();
  
  unblockSignals();
  
  return 0;
}

void LogFile::log(logTypes t, const char* format,...) {
  blockSignals();
  if((h!=-1) &&
     (((t==addrchange) && logaddrchange) ||
      ((t==badpacket) && logbadpackets) ||
      ((t==tablefull) && logtablefull) ||
      (t==status) || (t==print))) {
     
    char buf[256];
    va_list ls;
    va_start(ls,format);
    vsnprintf(buf,sizeof(buf),format,ls);
    va_end(ls);
    
    char buf1[256];
    Time t; gettimeofday(&t.tv,0);
    sprintf(buf1,"%s: %s\n",t.tostr(),buf);
    write(h,buf1,strlen(buf1));

    if(prnstderr) {
      fprintf(stderr,buf1);
    }
  }
  unblockSignals();
}

void LogFile::closeAllOtherFiles() {
  blockSignals();
  for(int a=0; a<NOFILE; a++) if(a!=h) close(a);
  unblockSignals();
}
