/*
 *	Xenophilia GTK+ Theme Engine
 *
 *  xeno_patches.c:
 *		Run-time modifications to GTK+.
 *
 *	Copyright  1999-2002 Johan Hanson <misagon@bahnhof.se>.
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include <stdlib.h>		/* for alloca() */
#include <stddef.h>		/* for offsetof() */
#include <string.h>
#include <stdio.h>

#include "xeno_rc_style.h"
#include "xeno_style.h"
#include "xeno_patches.h"

#if !XENO_GTK2 /* patches not applied */

/* #include "xeno_wheel.h" */

/* Patch control */
static void xeno_patch_install (GtkObjectClass *klass, size_t offset,
								GtkFunction xeno_func, GtkFunction * old_func_ptr);
static void xeno_patch_restore (GtkObjectClass *klass, size_t offset,
								GtkFunction xeno_func, GtkFunction old_func);

/* Patches */
#define XENO_SPIN_BUTTON_ARROW_SIZE		11

static void xeno_vscrollbar_draw_trough (GtkRange *);
static void xeno_vscrollbar_size_allocate (GtkWidget *, GtkAllocation *);
static void xeno_vscrollbar_size_request (GtkWidget *, GtkRequisition *);
static void xeno_vscrollbar_realize (GtkWidget *);
static void xeno_vscrollbar_slider_update (GtkRange *);
static gint xeno_vscrollbar_trough_click (GtkRange *, gint, gint, gfloat *);
static void	xeno_vscrollbar_motion (GtkRange *, gint, gint);

static void xeno_hscrollbar_draw_trough (GtkRange *);
static void xeno_hscrollbar_size_request (GtkWidget *, GtkRequisition *);
static void xeno_hscrollbar_size_allocate (GtkWidget *, GtkAllocation *);
static void xeno_hscrollbar_realize (GtkWidget *);
static void xeno_hscrollbar_slider_update (GtkRange *);
static gint xeno_hscrollbar_trough_click (GtkRange *, gint, gint, gfloat *);
static void	xeno_hscrollbar_motion (GtkRange *, gint, gint);

static void xeno_vscale_draw_trough (GtkRange *);
static void xeno_hscale_draw_trough (GtkRange *);
static void xeno_vscale_draw_slider (GtkRange *);
static void xeno_hscale_draw_slider (GtkRange *);

static gint xeno_vscale_expose_event(GtkWidget *, GdkEventExpose *);
static void xeno_vscale_draw_focus (GtkWidget *);
static gint xeno_hscale_expose_event(GtkWidget *, GdkEventExpose *);
static void xeno_hscale_draw_focus (GtkWidget *);

static void xeno_vpaned_realize (GtkWidget *);
static void xeno_hpaned_realize (GtkWidget *);
static void xeno_vpaned_size_allocate(GtkWidget *, GtkAllocation *);
static void xeno_hpaned_size_allocate(GtkWidget *, GtkAllocation *);

static gint xeno_vpaned_enter_notify_event (GtkWidget *, GdkEventCrossing *);
static gint xeno_hpaned_enter_notify_event (GtkWidget *, GdkEventCrossing *);
static gint xeno_vpaned_leave_notify_event (GtkWidget *, GdkEventCrossing *);
static gint xeno_hpaned_leave_notify_event (GtkWidget *, GdkEventCrossing *);

static void xeno_combo_size_allocate(GtkWidget *, GtkAllocation *);
static void xeno_entry_realize (GtkWidget *);
static void xeno_entry_size_allocate(GtkWidget *, GtkAllocation *);

static void xeno_clist_realize (GtkWidget *);

static void xeno_spin_button_size_allocate (GtkWidget *, GtkAllocation *);
static void xeno_spin_button_changed (GtkEditable *);
static void xeno_spin_button_realize (GtkWidget *);
static gint xeno_spin_button_button_press_event (GtkWidget *, GdkEventButton *);

static void xeno_check_button_size_allocate (GtkWidget *, GtkAllocation *);
static void xeno_check_button_size_request (GtkWidget *, GtkRequisition *);
static void xeno_radio_button_size_allocate (GtkWidget *, GtkAllocation *);
static void xeno_radio_button_size_request (GtkWidget *, GtkRequisition *);


/*
 *	Original (or previous) functions and values
 */

static void (* old_vscrollbar_realize)(GtkWidget *);
static void (* old_vscrollbar_slider_update)(GtkRange *);
static void (* old_vscrollbar_size_request)(GtkWidget *, GtkRequisition *);
static void (* old_vscrollbar_size_allocate)(GtkWidget *, GtkAllocation *);
static gint (* old_vscrollbar_trough_click)(GtkRange *, gint, gint, gfloat *);
static void (* old_vscrollbar_draw_trough)(GtkRange *);
static void (* old_vscrollbar_motion)(GtkRange *, gint, gint);

static void (* old_hscrollbar_realize)(GtkWidget *);
static void (* old_hscrollbar_slider_update)(GtkRange *);
static void (* old_hscrollbar_size_request)(GtkWidget *, GtkRequisition *);
static void (* old_hscrollbar_size_allocate)(GtkWidget *, GtkAllocation *);
static gint (* old_hscrollbar_trough_click)(GtkRange *, gint, gint, gfloat *);
static void (* old_hscrollbar_draw_trough)(GtkRange *);
static void (* old_hscrollbar_motion)(GtkRange *, gint, gint);

static void (* old_vpaned_size_allocate)(GtkWidget *, GtkAllocation *);
static void (* old_hpaned_size_allocate)(GtkWidget *, GtkAllocation *);
static void (* old_vpaned_realize)(GtkWidget *);
static void (* old_hpaned_realize)(GtkWidget *);

static gint (* old_vpaned_enter_notify_event)(GtkWidget *, GdkEventCrossing *);
static gint (* old_hpaned_enter_notify_event)(GtkWidget *, GdkEventCrossing *);
static gint (* old_vpaned_leave_notify_event)(GtkWidget *, GdkEventCrossing *);
static gint (* old_hpaned_leave_notify_event)(GtkWidget *, GdkEventCrossing *);

static void (* old_hscale_draw_trough)(GtkRange *);
static void (* old_hscale_draw_slider)(GtkRange *);
static gint (* old_hscale_expose_event)(GtkWidget *, GdkEventExpose *);
static void (* old_hscale_draw_focus)(GtkWidget *);

static void (* old_vscale_draw_trough)(GtkRange *);
static void (* old_vscale_draw_slider)(GtkRange *);
static gint (* old_vscale_expose_event)(GtkWidget *, GdkEventExpose *);
static void (* old_vscale_draw_focus)(GtkWidget *);

static void (* old_entry_realize)(GtkWidget *);
static void (* old_entry_size_allocate)(GtkWidget *, GtkAllocation *);
static void (* old_combo_size_allocate)(GtkWidget *, GtkAllocation *);

static void (* old_clist_realize)(GtkWidget *);

static void (* old_spin_button_size_allocate)(GtkWidget *, GtkAllocation *);
static void (* old_spin_button_changed)(GtkEditable *);
static void (* old_spin_button_realize)(GtkWidget *);
static gint (* old_spin_button_button_press_event)(GtkWidget *, GdkEventButton *);

#if !XENO_STYLE_PROP
static void (* old_check_button_size_allocate)(GtkWidget *, GtkAllocation *);
static void (* old_check_button_size_request)(GtkWidget *, GtkRequisition *);
static void (* old_radio_button_size_allocate)(GtkWidget *, GtkAllocation *);
static void (* old_radio_button_size_request)(GtkWidget *, GtkRequisition *);
#endif

static GtkWidgetClass * xeno_spin_button_parent_class = NULL;

static gint old_scrollbar_spacing;

static gint xeno_patch_count = 0;

/*
	Install and Remove
*/
#define PATCH_INSTALL(klass, type, name, field) \
	xeno_patch_install ((GtkObjectClass *)klass, offsetof(type, field), \
						(GtkFunction)xeno_ ## name ## _ ## field, (GtkFunction *)&old_ ## name ## _ ## field)

#define PATCH_RESTORE(klass, type, name, field) \
	xeno_patch_restore ((GtkObjectClass *)klass, offsetof(type, field), \
						(GtkFunction)xeno_ ## name ## _ ## field, (GtkFunction)old_ ## name ## _ ## field)

/* remember what is patched */

guint16 xeno_patch_config	= XENO_PATCH_DEFAULT;

void xeno_patches_install () {
	GtkWidgetClass	*widget_class;
	const char *config, *token;
	int n, mod, invert;
	
	/* Test if we should install patches */
	if (xeno_patch_count++ != 0)
		return;
	
	config = getenv("XENO_THEME_PATCH");
	if (config) {
		/* scan */
		xeno_patch_config = 0;
		token = config;
		invert = 0;
		while(1) {
			while (*token == ' ')
				++token;
			
			if ((n = strcspn(token, " _\n")) == 0)
				break;
			
			if (strncasecmp(token, "NOT ", 4) == 0) {
				invert ^= 1;
			} else {
				mod = 0;
				
				if (strncasecmp(token, "GTK", 3) == 0)
					token += 3;
				
				if (strncasecmp(token, "SCROLL", 5) == 0) {
					mod = XENO_PATCH_SCROLLBAR;
				} else if (strncasecmp(token, "SCALE", 5) == 0) {
					mod = XENO_PATCH_SCALE;
				} else if (strncasecmp(token, "RADIO", 5) == 0) {
					mod = XENO_PATCH_RADIO_BUTTON;
				} else if (strncasecmp(token, "CHECK", 5) == 0) {
					mod = XENO_PATCH_CHECK_BUTTON;
				} else if (strncasecmp(token, "PANE", 4) == 0 || strncasecmp(token, "RESIZE", 6) == 0) {
					mod = XENO_PATCH_PANED;
				} else if (strncasecmp(token, "SPIN", 4) == 0 || strncasecmp(token, "NUM", 3) == 0) {
					mod = XENO_PATCH_SPIN_BUTTON;
				} else if (strncasecmp(token, "COMBO", 5) == 0) {
					mod = XENO_PATCH_COMBO;
				} else if (strncasecmp(token, "CLIST", 5) == 0) {
					mod = XENO_PATCH_CLIST;
				} else if (   strncasecmp(token, "ALL", 3) == 0
						   || strncasecmp(token, "YES", 3) == 0
						   || strncasecmp(token, "TRUE", 4) == 0)
				{
					mod = XENO_PATCH_ALL;
				} else if (strncasecmp(token, "DEFAULT", 3) == 0) {
					mod = XENO_PATCH_DEFAULT;
				} else if (   strncasecmp(token, "NO", 2) == 0
						   || strncasecmp(token, "FALSE", 6) == 0)
				{
					invert ^= 1;
					mod = XENO_PATCH_ALL;
				}
				
				if (invert) {
					xeno_patch_config &= ~mod;
					invert = 0;
				} else {
					xeno_patch_config |= mod;
				}
			}
			
			token += n;
		}
	}
	
	/* Scrollbars */
	if (xeno_patch_config & XENO_PATCH_SCROLLBAR) {
		GtkScrolledWindowClass *scwinclass;
		GtkRangeClass	*range_class;
		
		/* Make scrollbars wider */
		range_class = (GtkRangeClass *)gtk_type_class(gtk_scrollbar_get_type());
		
		/* Move the scrollbar buttons to the lower right corner */
		widget_class = (GtkWidgetClass *) gtk_type_class(gtk_vscrollbar_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, vscrollbar, size_request);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vscrollbar, size_allocate);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vscrollbar, realize);
		PATCH_INSTALL(widget_class, GtkRangeClass, vscrollbar, slider_update);
		PATCH_INSTALL(widget_class, GtkRangeClass, vscrollbar, trough_click);
		PATCH_INSTALL(widget_class, GtkRangeClass, vscrollbar, draw_trough);
		PATCH_INSTALL(widget_class, GtkRangeClass, vscrollbar, motion);
		
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hscrollbar_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, hscrollbar, size_request);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hscrollbar, size_allocate);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hscrollbar, realize);
		PATCH_INSTALL(widget_class, GtkRangeClass, hscrollbar, slider_update);
		PATCH_INSTALL(widget_class, GtkRangeClass, hscrollbar, trough_click);
		PATCH_INSTALL(widget_class, GtkRangeClass, hscrollbar, draw_trough);
		PATCH_INSTALL(widget_class, GtkRangeClass, hscrollbar, motion);
		
		/* Make ScrolledWindow spacing smaller */
		scwinclass = (GtkScrolledWindowClass *) gtk_type_class (gtk_scrolled_window_get_type ());
		old_scrollbar_spacing = scwinclass->scrollbar_spacing;
		scwinclass->scrollbar_spacing = 1;
	}
	
	/* Make radio and check buttons larger */
	if (xeno_patch_config & XENO_PATCH_RADIO_BUTTON) {
		widget_class = (GtkWidgetClass *) gtk_type_class (gtk_radio_button_get_type ());
		PATCH_INSTALL(widget_class, GtkWidgetClass, radio_button, size_request);
		PATCH_INSTALL(widget_class, GtkWidgetClass, radio_button, size_allocate);
	}
	
	if (xeno_patch_config & XENO_PATCH_CHECK_BUTTON) {
		widget_class = (GtkWidgetClass *) gtk_type_class (gtk_check_button_get_type ());
		PATCH_INSTALL(widget_class, GtkWidgetClass, check_button, size_request);
		PATCH_INSTALL(widget_class, GtkWidgetClass, check_button, size_allocate);
	}
	
	/* Make entire panes sensitive */
	if (xeno_patch_config & XENO_PATCH_PANED) {
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_vpaned_get_type());
		
		PATCH_INSTALL(widget_class, GtkWidgetClass, vpaned, size_allocate);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vpaned, realize);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vpaned, enter_notify_event);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vpaned, leave_notify_event);
		
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hpaned_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, hpaned, size_allocate);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hpaned, realize);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hpaned, enter_notify_event);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hpaned, leave_notify_event);
	}

	/* Draw sliders differently */
	if (xeno_patch_config & XENO_PATCH_SCALE) {
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hscale_get_type());
		PATCH_INSTALL(widget_class, GtkRangeClass, hscale, draw_trough);
		PATCH_INSTALL(widget_class, GtkRangeClass, hscale, draw_slider);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hscale, expose_event);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hscale, draw_focus);
		
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_vscale_get_type());
		PATCH_INSTALL(widget_class, GtkRangeClass, vscale, draw_trough);
		PATCH_INSTALL(widget_class, GtkRangeClass, vscale, draw_slider);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vscale, expose_event);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vscale, draw_focus);
	}
	
	/* Combo box / entry combination */
	if (xeno_patch_config & XENO_PATCH_COMBO) {
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_combo_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, combo, size_allocate);
		
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_entry_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, entry, realize);
		PATCH_INSTALL(widget_class, GtkWidgetClass, entry, size_allocate);
	}
	
	/* CList */
	if (xeno_patch_config & XENO_PATCH_CLIST) {
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_clist_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, clist, realize);
	}
	
	/* GtkSpinButton */
	if (xeno_patch_config & XENO_PATCH_SPIN_BUTTON) {
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_spin_button_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, spin_button, size_allocate);
		PATCH_INSTALL(widget_class, GtkWidgetClass, spin_button, realize);
		PATCH_INSTALL(widget_class, GtkEditableClass, spin_button, changed);
		PATCH_INSTALL(widget_class, GtkWidgetClass, spin_button, button_press_event);
		
	  #if XENO_GTK2
		xeno_spin_button_parent_class = GTK_WIDGET_CLASS(g_type_class_peek(g_type_parent(GTK_TYPE_SPIN_BUTTON)));
	  #else
		xeno_spin_button_parent_class = GTK_WIDGET_CLASS(gtk_type_parent_class(gtk_spin_button_get_type()));
	  #endif
	}
}


void xeno_patches_uninstall () {
	GtkWidgetClass	*widget_class;
	
	if (xeno_patch_count == 0)
		return;
	
	xeno_patch_count = 0;

	/* GtkSpinButton */
	if (xeno_patch_config & XENO_PATCH_SPIN_BUTTON) {
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_spin_button_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, spin_button, button_press_event);
	  #if XENO_GTK2
		PATCH_RESTORE(widget_class, GtkEntryClass, spin_button, changed);
	  #else
		PATCH_RESTORE(widget_class, GtkEditableClass, spin_button, changed);
	  #endif
		PATCH_RESTORE(widget_class, GtkWidgetClass, spin_button, realize);
		PATCH_RESTORE(widget_class, GtkWidgetClass, spin_button, size_allocate);
	}
	
	/* CList */
	if (xeno_patch_config & XENO_PATCH_CLIST) {
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_clist_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, clist, realize);
	}

	/* GtkCombo, GtkEntry */
	if (xeno_patch_config & XENO_PATCH_COMBO) {
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_entry_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, entry, size_allocate);
		PATCH_RESTORE(widget_class, GtkWidgetClass, entry, realize);
		
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_combo_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, combo, size_allocate);
	}

	/* GtkVScale, GtkHScale */
	if (xeno_patch_config & XENO_PATCH_SCALE) {
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_vscale_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, vscale, draw_focus);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vscale, expose_event);
		PATCH_RESTORE(widget_class, GtkRangeClass, vscale, draw_slider);
		PATCH_RESTORE(widget_class, GtkRangeClass, vscale, draw_trough);

		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hscale_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, hscale, draw_focus);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hscale, expose_event);
		PATCH_RESTORE(widget_class, GtkRangeClass, hscale, draw_slider);
		PATCH_RESTORE(widget_class, GtkRangeClass, hscale, draw_trough);
	}

	/* GtkVPaned, GtkHPaned.. they suck btw, use my new paned table class and be happy!
	   You can find it at <http://www.obsession.se/johan/gtk.html>.
	*/
	if (xeno_patch_config & XENO_PATCH_PANED) {
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hpaned_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, hpaned, leave_notify_event);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hpaned, enter_notify_event);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hpaned, realize);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hpaned, size_allocate);
	
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_vpaned_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, vpaned, leave_notify_event);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vpaned, enter_notify_event);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vpaned, realize);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vpaned, size_allocate);
	}
	
	/* GtkRadioButton and GtkCheckButton */
	if (xeno_patch_config & XENO_PATCH_CHECK_BUTTON) {
		widget_class = (GtkWidgetClass *) gtk_type_class (gtk_check_button_get_type ());
		PATCH_RESTORE(widget_class, GtkWidgetClass, check_button, size_allocate);
		PATCH_RESTORE(widget_class, GtkWidgetClass, check_button, size_request);
	}
	
	if (xeno_patch_config & XENO_PATCH_RADIO_BUTTON) {
		widget_class = (GtkWidgetClass *) gtk_type_class (gtk_radio_button_get_type ());
		PATCH_RESTORE(widget_class, GtkWidgetClass, radio_button, size_allocate);
		PATCH_RESTORE(widget_class, GtkWidgetClass, radio_button, size_request);
	}
	
	/* Scrollbars */
	if (xeno_patch_config & XENO_PATCH_SCROLLBAR) {
		GtkScrolledWindowClass *scwinclass;
		
		/* GtkScrolledWindow */
		scwinclass = (GtkScrolledWindowClass *) gtk_type_class (gtk_scrolled_window_get_type ());
		scwinclass->scrollbar_spacing = old_scrollbar_spacing;
		
		/* Scrollbars */
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hscrollbar_get_type());
		PATCH_INSTALL(widget_class, GtkRangeClass, hscrollbar, motion);
		PATCH_RESTORE(widget_class, GtkRangeClass, hscrollbar, draw_trough);
		PATCH_RESTORE(widget_class, GtkRangeClass, hscrollbar, trough_click);
		PATCH_RESTORE(widget_class, GtkRangeClass, hscrollbar, slider_update);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hscrollbar, realize);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hscrollbar, size_allocate);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hscrollbar, size_request);

		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_vscrollbar_get_type());
		PATCH_INSTALL(widget_class, GtkRangeClass, vscrollbar, motion);
		PATCH_RESTORE(widget_class, GtkRangeClass, vscrollbar, draw_trough);
		PATCH_RESTORE(widget_class, GtkRangeClass, vscrollbar, trough_click);
		PATCH_RESTORE(widget_class, GtkRangeClass, vscrollbar, slider_update);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vscrollbar, realize);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vscrollbar, size_allocate);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vscrollbar, size_request);
	}
}


/*
 *	Patch Control
 */
void xeno_patch_install (GtkObjectClass *klass, size_t offset,
						 GtkFunction xeno_func, GtkFunction * old_func_ptr)
{
  /*
	GList			*children, *child;
  */
	GtkFunction	*ptr;
	
	ptr = (GtkFunction *)((guchar *)klass + offset);
	
	*old_func_ptr = *ptr;
	if (*ptr == xeno_func) {
	  #if XENO_GTK2
		g_warning ("Redundant patch of %s::%d", G_OBJECT_CLASS_NAME(klass), offset);
	  #else
		g_warning ("Redundant patch of %s::%d", gtk_type_name(klass->type), offset);
	  #endif
	}
	
	*ptr = xeno_func;
}


void xeno_patch_restore (GtkObjectClass *klass, size_t offset,
						 GtkFunction xeno_func, GtkFunction old_func)
{
	GList		*child;
	GtkFunction	*ptr;

	ptr = (GtkFunction *)((guchar *)klass + offset);
	if (*ptr == xeno_func)
		*ptr = old_func;

  #if XENO_GTK2
	{
		GType	*children;
		guint	n_children, i;
		
		children = g_type_children (G_OBJECT_CLASS_TYPE(klass), &n_children);
		if (children && n_children) {
			for (i = 0; i < n_children; ++i)
				xeno_patch_restore (gtk_type_class(GPOINTER_TO_UINT(children[i])),
									offset, xeno_func, old_func);
			
			g_free(children);
		}
	}
  #else
	for (child = gtk_type_children_types(klass->type); child; child = child->next)
		xeno_patch_restore (gtk_type_class(GPOINTER_TO_UINT(child->data)),
							offset, xeno_func, old_func);
  #endif
}


/*
 * Patched functions
 */
#if XENO_GTK2
#define RANGE_CLASS(widget) \
	GTK_RANGE_GET_CLASS(widget)
#else
#define RANGE_CLASS(widget) \
	GTK_RANGE_CLASS(GTK_OBJECT(widget)->klass)
#endif

typedef struct {
	gint	start, breadth, length;		/* y, w, h	*/
	gint	bt, lt;						/* xthickness, ythickness */
	gint	min_length;
	
	gint	back_start, back_size;		/* sb, s	*/
	gint	forw_start, forw_size;		/* sf, s	*/
} XenoScrollbarDims;

static void
xeno_scrollbar_config  (GtkWidget			*widget,
						XenoScrollbarDims	*dims,
						GtkOrientation		orn)
{
	GtkStyle	*style;
	XenoRcData	*rc_data;
	gint	start, length, need, size, spacing, extra;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (dims != NULL);
	
	size	= RANGE_CLASS(widget)->stepper_size;
	spacing = RANGE_CLASS(widget)->stepper_slider_spacing;
	dims->min_length = RANGE_CLASS(widget)->min_slider_size;
	
	style = widget->style;
	if (orn == GTK_ORIENTATION_VERTICAL) {
		dims->bt = XENO_STYLE_XTHICKNESS(style);
		dims->lt = XENO_STYLE_YTHICKNESS(style);
		dims->length = widget->allocation.height;
	} else {
		dims->bt = XENO_STYLE_YTHICKNESS(style);
		dims->lt = XENO_STYLE_XTHICKNESS(style);
		dims->length = widget->allocation.width;
	}
	
	rc_data = NULL;
	extra = 0;
	if (XENO_STYLE_IS_XENO(style)) {
		rc_data = XENO_STYLE_RC_DATA(widget->style);
		spacing = 0;
		if (rc_data != NULL) {
			size = rc_data->scrollbar_width;
			if (size < 8)
				size = RANGE_CLASS(widget)->stepper_size;
			
			if (   rc_data->scrollbar_flush
				|| rc_data->shadow_in[GTK_STATE_NORMAL] == XENO_SHADOW_NONE)
			{
				dims->lt = 0;
				dims->bt = 0;
			} else if (rc_data->shadow_in[GTK_STATE_NORMAL] == XENO_SHADOW_BLACK_IN) {
				dims->lt = MIN(dims->lt, 1);
				dims->bt = MIN(dims->bt, 1);
			}
			if (rc_data->stepper_box && rc_data->shadow_out[GTK_STATE_NORMAL] == XENO_SHADOW_BLACK_OUT ) {
				spacing = -1;
				extra = -1;
			}
		}
		dims->min_length = size;
	}
	
	start	= dims->lt;
	length	= dims->length - start * 2;
	need	= (size + spacing) * 2;
	if (length - need < dims->min_length) {
		dims->back_start = dims->forw_start = length;
		dims->breadth = size;
		size = 0;
	} else {
		length -= need;
		dims->back_start = start;
		dims->breadth = size;
		if (rc_data && !rc_data->stepper_ends) {
			GtkWidget *parent = widget->parent;
			if (   parent != NULL
				&& GTK_IS_SCROLLED_WINDOW(parent)
				&& (GTK_SCROLLED_WINDOW(parent)->window_placement & ((orn == GTK_ORIENTATION_VERTICAL) ? 0x01 : 0x02)))
			{
				start += size * 2 + spacing;
			} else {
				dims->back_start = start + length + spacing;
			}
			dims->forw_start = dims->back_start + size + extra;
			/* length -= spacing; */
		} else {
			start += size + spacing;
			dims->forw_start = start + length + spacing;
		}
	}
	dims->start  = start;
	dims->length = length;
	dims->back_size = size;
	dims->forw_size	= size;
}

static void
xeno_scrollbar_update	(GtkRange			*range,
						 XenoScrollbarDims	*dims,
						 GtkOrientation		orientation)
{
	GtkAdjustment	*adjustment;
	gint	start, length, min_length, max_length;
	gfloat	adj_size, adj_page;
	
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_SCROLLBAR (range));
	
	adjustment = range->adjustment;
	if (adjustment == NULL)
		return; /* not an error */
	
	start  = 0;
	length = dims->length;
	adj_size = adjustment->upper - adjustment->lower;
	adj_page = adjustment->page_size;
	if (adj_page > 0.0F && adj_page < adj_size) {
		max_length = length;
		min_length = dims->min_length;
		length = ((gfloat)max_length * adj_page) / adj_size;
		if (length < min_length)
			length = min_length;
		
		start = ((max_length - length) * (adjustment->value - adjustment->lower)) / (adj_size - adj_page);
		min_length >>= 1;
		if (start < 0) {
			length += start;
			start = 0;
			if (length < min_length)
				length = min_length;
		} else if (start + length > max_length) {
			length = dims->length - start;
			if (length < min_length) {
				length = min_length;
				start = max_length - length;
			}
		}
	}
	start += dims->start;
	
	if (GTK_WIDGET_REALIZED(range)) {
		if (orientation == GTK_ORIENTATION_VERTICAL) {
			gdk_window_move_resize (range->slider, dims->bt, start, dims->breadth, length);
		} else {
			gdk_window_move_resize (range->slider, start, dims->bt, length, dims->breadth);
		}
	}
}

static void
xeno_scrollbar_adjust	(GtkRange	*range,
						 gint		pos,
						 gint		length)
{
	GtkAdjustment *adjustment;
	gdouble old_value;
	
	pos = CLAMP(pos, 0, length);
	adjustment = range->adjustment;
	old_value = adjustment->value;
	
	adjustment->value = (adjustment->upper - adjustment->lower - adjustment->page_size)
					  * (pos / (gfloat)length)
					  + adjustment->lower;
	
	if (range->digits >= 0) {
		char buffer[64];
		
		sprintf (buffer, "%0.*f", range->digits, adjustment->value);
		sscanf (buffer, "%f", &adjustment->value);
	}
	
	if (old_value != adjustment->value) {
		if (range->policy == GTK_UPDATE_CONTINUOUS) {
			gtk_signal_emit_by_name (GTK_OBJECT(adjustment), "value_changed");
		} else  {
			gtk_range_slider_update (range);
			gtk_range_clear_background (range);
			if (range->policy == GTK_UPDATE_DELAYED) {
				if (range->timer) {
					gtk_timeout_remove (range->timer);
					range->need_timer = FALSE;
				}
				range->timer = gtk_timeout_add (300, (GtkFunction)RANGE_CLASS(range)->timer,
												(gpointer)range);
			}
		}
	}
}

static void
xeno_vscrollbar_draw_trough (GtkRange *range)
{
	GtkWidget	*widget;
	GtkStyle	*style;
	XenoRcData	*rc_data;
	
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_SCROLLBAR(range));
	
	widget	= (GtkWidget *)range;
	style	= widget->style;
	rc_data = NULL;
	if (XENO_STYLE_IS_XENO(style))
		rc_data = XENO_STYLE_RC_DATA(style);
	
	gtk_paint_flat_box (widget->style, range->trough,
						(widget->state != GTK_STATE_INSENSITIVE) ? GTK_STATE_ACTIVE : GTK_STATE_INSENSITIVE,
						GTK_SHADOW_IN, NULL, widget, "trough", 0, 0, -1, -1);
	
	gtk_paint_shadow (widget->style, range->trough,
					  (widget->state != GTK_STATE_INSENSITIVE) ? GTK_STATE_NORMAL : GTK_STATE_INSENSITIVE,
					  (rc_data && rc_data->scrollbar_flush) ? GTK_SHADOW_NONE : GTK_SHADOW_IN,
					  NULL, widget, "trough",
					  0, 0, -1, -1);
}

/* GtkVScrollbar */
static void
xeno_vscrollbar_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
	XenoScrollbarDims dims;
	GtkRange	*range;
	gint		trough_width;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_VSCROLLBAR (widget));
	g_return_if_fail (allocation != NULL);
	
	range = GTK_RANGE(widget);
	widget->allocation = *allocation;
	if (GTK_WIDGET_REALIZED (widget)) {
		xeno_scrollbar_config (widget, &dims, GTK_ORIENTATION_VERTICAL);
		
		if (range->step_back) {
			if (dims.back_size) {
				gdk_window_move_resize (range->step_back,
										dims.bt, dims.back_start,
										dims.breadth, dims.back_size);
				
				if (!gdk_window_is_visible (range->step_back))
					gdk_window_show (range->step_back);
			} else if (gdk_window_is_visible (range->step_back)) {
				gdk_window_hide (range->step_back);
			}
		}
		
		if (range->step_forw) {
			if (dims.forw_size) {
				gdk_window_move_resize (range->step_forw,
										dims.bt, dims.forw_start,
										dims.breadth, dims.forw_size);
				
				if (!gdk_window_is_visible (range->step_forw))
					gdk_window_show (range->step_forw);
			} else if (gdk_window_is_visible (range->step_forw)) {
				gdk_window_hide (range->step_forw);
			}
		}
		
		trough_width = dims.breadth + dims.bt * 2;
		gdk_window_move_resize (range->trough,
								allocation->x + (widget->allocation.width - trough_width)/2,
								allocation->y,
								trough_width,
								allocation->height);
		
		xeno_scrollbar_update (range, &dims, GTK_ORIENTATION_VERTICAL);
		gtk_widget_queue_draw (widget);
	}
}

static void
xeno_vscrollbar_realize	(GtkWidget	*widget)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_VSCROLLBAR (widget));
	
	old_vscrollbar_realize (widget);
	xeno_vscrollbar_size_allocate (widget, &(widget->allocation));
}

static gint
xeno_vscrollbar_trough_click (GtkRange *range, gint x, gint y, gfloat *jump_perc)
{
	XenoScrollbarDims dims;
	gint sy;
	
	g_return_val_if_fail (range != NULL, GTK_TROUGH_NONE);
	g_return_val_if_fail (GTK_IS_VSCROLLBAR (range), GTK_TROUGH_NONE);
	
	xeno_scrollbar_config ((GtkWidget *)range, &dims, GTK_ORIENTATION_VERTICAL);
	
	if (x < dims.bt || x >= dims.breadth - dims.bt || y < dims.start || y >= dims.start + dims.length)
		return GTK_TROUGH_NONE;
	
	if (jump_perc) {
		*jump_perc = ((gdouble)(y - dims.start)) / ((gdouble)dims.length);
		return GTK_TROUGH_JUMP;
	} else {
		gdk_window_get_position (range->slider, NULL, &sy);
		if (y < sy)
			return GTK_TROUGH_START;
	}
	return GTK_TROUGH_END;
}


static void
xeno_vscrollbar_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
	XenoRcData	*data;
	gint w, xt, yt;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_VSCROLLBAR (widget));
	g_return_if_fail (requisition != NULL);
	
	data = XENO_STYLE_IS_XENO(widget->style) ? XENO_STYLE_RC_DATA(widget->style) : NULL;
	
	w = RANGE_CLASS(widget)->slider_width;
	xt = XENO_STYLE_XTHICKNESS(widget->style);
	yt = XENO_STYLE_YTHICKNESS(widget->style);
	if (data) {
		w = data->scrollbar_width;
		if (data->scrollbar_flush || data->shadow_in[GTK_STATE_NORMAL] == XENO_SHADOW_NONE)
			xt = yt = 0;
	}
	
	requisition->width	= xt + xt + w;
	requisition->height	= yt + yt + w + w + w;
	widget->requisition = *requisition;
}

static void
xeno_vscrollbar_slider_update (GtkRange *range) {
	XenoScrollbarDims dims;
	
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_VSCROLLBAR (range));
	
	if (GTK_WIDGET_REALIZED(range)) {
		xeno_scrollbar_config ((GtkWidget *)range, &dims, GTK_ORIENTATION_VERTICAL);
		xeno_scrollbar_update (range, &dims, GTK_ORIENTATION_VERTICAL);
	}
}

static void
xeno_vscrollbar_motion	(GtkRange	*range,
						 gint		xdelta,
						 gint		ydelta)
{
	XenoScrollbarDims dims;
	gint top, bottom;
	gint y, height;
	
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_VSCROLLBAR(range));
	
	xeno_scrollbar_config ((GtkWidget *)range, &dims, GTK_ORIENTATION_VERTICAL);
	gdk_window_get_geometry (range->slider, NULL, &y, NULL, &height, NULL);
	top = dims.start;
	bottom = dims.start + dims.length - height;
	if (bottom == top)
		return;
	
	xeno_scrollbar_adjust (range, y + ydelta - top, bottom - top);
}


/* GtkHScrollbar */
static void
xeno_hscrollbar_motion	(GtkRange	*range,
						 gint		xdelta,
						 gint		ydelta)
{
	XenoScrollbarDims dims;
	gint left, right;
	gint x, width;
	
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_HSCROLLBAR(range));
	
	xeno_scrollbar_config ((GtkWidget *)range, &dims, GTK_ORIENTATION_HORIZONTAL);
	gdk_window_get_geometry (range->slider, &x, NULL, &width, NULL, NULL);
	left  = dims.start;
	right = dims.start + dims.length - width;
	if (left == right)
		return;
	
	xeno_scrollbar_adjust (range, x + xdelta - left, right - left);
}

static gint
xeno_hscrollbar_trough_click (GtkRange *range, gint x, gint y, gfloat *jump_perc)
{
	XenoScrollbarDims dims;
	gint	sx;
	
	g_return_val_if_fail (range != NULL, GTK_TROUGH_NONE);
	g_return_val_if_fail (GTK_IS_HSCROLLBAR (range), GTK_TROUGH_NONE);
	
	xeno_scrollbar_config ((GtkWidget *)range, &dims, GTK_ORIENTATION_HORIZONTAL);
	
	if (y < dims.bt || y >= dims.breadth - dims.bt || x < dims.start || x >= dims.start + dims.length)
		return GTK_TROUGH_NONE;
	
	if (jump_perc) {
		*jump_perc = ((gdouble)(x - dims.start)) / ((gdouble)dims.length);
		return GTK_TROUGH_JUMP;
	} else {
		gdk_window_get_position (range->slider, &sx, NULL);
		if (x < sx)
			return GTK_TROUGH_START;
	}
	return GTK_TROUGH_END;
}

static void
xeno_hscrollbar_size_request (GtkWidget *widget, GtkRequisition *requisition) {
	XenoRcData	*data;
	gint h, xt, yt;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_HSCROLLBAR (widget));
	g_return_if_fail (requisition != NULL);
	
	data = XENO_STYLE_IS_XENO(widget->style) ? XENO_STYLE_RC_DATA(widget->style) : NULL;
	
	h = RANGE_CLASS(widget)->slider_width;
	xt = XENO_STYLE_XTHICKNESS(widget->style);
	yt = XENO_STYLE_YTHICKNESS(widget->style);
	if (data) {
		h = data->scrollbar_width;
		if (data->scrollbar_flush || data->shadow_in[GTK_STATE_NORMAL] == XENO_SHADOW_NONE)
			xt = yt = 0;
	}
	
	requisition->width	= xt + xt + h + h + h;
	requisition->height	= yt + yt + h;
	widget->requisition = *requisition;
}

static void
xeno_hscrollbar_realize (GtkWidget *widget) {
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_HSCROLLBAR (widget));

	old_hscrollbar_realize (widget);
	xeno_hscrollbar_size_allocate (widget, &(widget->allocation));
}

static void
xeno_hscrollbar_slider_update (GtkRange *range) {
	XenoScrollbarDims dims;
	
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_HSCROLLBAR (range));
	
	if (GTK_WIDGET_REALIZED((GtkWidget *)range)) {
		xeno_scrollbar_config ((GtkWidget *)range, &dims, GTK_ORIENTATION_HORIZONTAL);
		xeno_scrollbar_update (range, &dims, GTK_ORIENTATION_HORIZONTAL);
	}
}

static void
xeno_hscrollbar_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
	XenoScrollbarDims dims;
	GtkRange	*range;
	gint		trough_height;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_HSCROLLBAR (widget));
	g_return_if_fail (allocation != NULL);
	
	range = GTK_RANGE(widget);
    widget->allocation = *allocation;
	if (GTK_WIDGET_REALIZED (widget)) {
		xeno_scrollbar_config (widget, &dims, GTK_ORIENTATION_HORIZONTAL);
		
		if (range->step_back) {
			if (dims.back_size) {
				gdk_window_move_resize (range->step_back,
										dims.back_start, dims.bt,
										dims.back_size, dims.breadth);
				if (!gdk_window_is_visible (range->step_back))
					gdk_window_show (range->step_back);
			} else if (gdk_window_is_visible (range->step_back)) {
				gdk_window_hide (range->step_back);
			}
		}
		
		if (range->step_forw) {
			if (dims.back_size) {
				gdk_window_move_resize (range->step_forw,
										dims.forw_start, dims.bt,
										dims.forw_size, dims.breadth);
				if (!gdk_window_is_visible (range->step_forw))
					gdk_window_show (range->step_forw);
			} else if (gdk_window_is_visible (range->step_forw)) {
				gdk_window_hide (range->step_forw);
			}
		}
		
		trough_height = dims.breadth + dims.bt * 2;
		gdk_window_move_resize (range->trough,
								allocation->x, allocation->y + (allocation->height - (trough_height))/2,
								allocation->width, trough_height);
		
		xeno_scrollbar_update (range, &dims, GTK_ORIENTATION_HORIZONTAL);
		gtk_widget_queue_draw (widget);
	}
}

static void
xeno_hscrollbar_draw_trough (GtkRange *range) {
	xeno_vscrollbar_draw_trough (range);
}


/* Panes */
static void xeno_hpaned_realize (GtkWidget *widget) {
	GdkCursor *crsr;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_HPANED(widget));

	old_hpaned_realize(widget);
	if (XENO_STYLE_IS_XENO(widget->style)) {
		crsr = gdk_cursor_new(GDK_SB_H_DOUBLE_ARROW);
		gdk_window_set_cursor(GTK_PANED(widget)->handle, crsr);
		gdk_cursor_destroy(crsr);
		gdk_window_set_events(GTK_PANED(widget)->handle,
							    gdk_window_get_events(GTK_PANED(widget)->handle)
							  | GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);
		xeno_hpaned_size_allocate (widget, &(widget->allocation));
	}
}

static void xeno_hpaned_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
	GtkPaned	*paned;
	gint		x, width, bw;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_HPANED(widget));

	if (XENO_STYLE_IS_XENO(widget->style)) {
		paned = ((GtkPaned*)widget);
		paned->handle_size = paned->gutter_size;
	
		old_hpaned_size_allocate (widget, allocation);
		if (GTK_WIDGET_REALIZED(widget)) {
			gdk_window_get_geometry(paned->handle, &x, NULL, &width, NULL, NULL);
			bw = ((GtkContainer *)widget)->border_width;
			gdk_window_move_resize (paned->handle,
									x + (paned->handle_size - 6)/2,
									bw,
									6,
									widget->allocation.height - bw - bw);
		}
		gtk_widget_queue_draw(widget);
	} else {
		old_hpaned_size_allocate (widget, allocation);
	}
}

static void xeno_vpaned_realize (GtkWidget *widget) {
	GdkCursor *crsr;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_VPANED(widget));

	old_vpaned_realize(widget);
	if (XENO_STYLE_IS_XENO(widget->style)) {
		crsr = gdk_cursor_new(GDK_SB_V_DOUBLE_ARROW);
		gdk_window_set_cursor(GTK_PANED(widget)->handle, crsr);
		gdk_cursor_destroy(crsr);
		gdk_window_set_events(GTK_PANED(widget)->handle,
							    gdk_window_get_events(GTK_PANED(widget)->handle)
							  | GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);
		xeno_vpaned_size_allocate (widget, &(widget->allocation));
	}
}

static void xeno_vpaned_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
	GtkPaned	*paned;
	gint		y, height, bw;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_VPANED(widget));

	if (XENO_STYLE_IS_XENO(widget->style)) {
		paned = ((GtkPaned*)widget);
		paned->handle_size = paned->gutter_size;
	
		old_vpaned_size_allocate (widget, allocation);
		if (GTK_WIDGET_REALIZED(widget)) {
			gdk_window_get_geometry(paned->handle, NULL, &y, NULL, &height, NULL);
			bw = ((GtkContainer *)widget)->border_width;
			gdk_window_move_resize (paned->handle,
									bw,
									y + (paned->handle_size - 6)/2,
									widget->allocation.width - bw - bw,
									6);
		}
		gtk_widget_queue_draw(widget);
	} else {
		old_vpaned_size_allocate (widget, allocation);
	}
}

static void xeno_paned_fake_event (GtkWidget *widget) {
	GdkEventExpose event;
	gint width, height;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_PANED(widget));
	
	event.type = GDK_EXPOSE;
	event.send_event = 1;
	event.window = GTK_PANED(widget)->handle;
	gdk_window_get_size (event.window, &width, &height);
	event.area.x = event.area.y = 0;
	event.area.width = width;
	event.area.height = height;
	event.count = 0;
	
	gdk_event_put ((GdkEvent *)&event);
}

static gint xeno_vpaned_enter_notify_event (GtkWidget *widget, GdkEventCrossing *event) {
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (event != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_VPANED(widget), FALSE);
	
	xeno_paned_fake_event(widget);
	return TRUE;
}

static gint xeno_vpaned_leave_notify_event (GtkWidget *widget, GdkEventCrossing *event) {
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (event != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_VPANED(widget), FALSE);
	
	xeno_paned_fake_event(widget);
	return TRUE;
}

static gint xeno_hpaned_enter_notify_event (GtkWidget *widget, GdkEventCrossing *event) {
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (event != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_HPANED(widget), FALSE);

	xeno_paned_fake_event(widget);
	return TRUE;
}

static gint xeno_hpaned_leave_notify_event (GtkWidget *widget, GdkEventCrossing *event) {
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (event != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_HPANED(widget), FALSE);
	
	xeno_paned_fake_event(widget);
	return TRUE;
}


/* GtkScale */

static gint xeno_hscale_expose_event(GtkWidget *widget, GdkEventExpose *event) {
	return xeno_vscale_expose_event(widget, event);
}

static gint xeno_vscale_expose_event(GtkWidget *widget, GdkEventExpose *event) {
	GtkRange *range;
	
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_SCALE (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);
	
	range = GTK_RANGE (widget);
	
	if (event->window == range->trough) {
		/* no declining to draw */
		gtk_range_draw_trough(range);
	} else if (event->window == range->slider) {
		gtk_range_draw_slider(range);
	} else if (event->window == range->step_forw) {
		gtk_range_draw_step_forw(range);
	} else if (event->window == range->step_back) {
		gtk_range_draw_step_back(range);
	} else if (event->window == widget->window) {
		gtk_range_draw_background(range);
	}
	return FALSE;
}

static void xeno_hscale_draw_focus (GtkWidget *widget) {
	xeno_vscale_draw_focus(widget);
}

static void xeno_vscale_draw_focus (GtkWidget *widget) {
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_SCALE (widget));
	
	if (GTK_WIDGET_DRAWABLE (widget))
		gtk_range_draw_slider (GTK_RANGE (widget));
}


/* GtkHScale */
static void xeno_hscale_draw_trough (GtkRange *range) {
	GtkStyle *style;
	int trough_y, trough_width, trough_height, xt, yt;
	
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_HSCALE (range));
	
	if (range->trough) {
		style = ((GtkWidget *)range)->style;
		
		if (!XENO_STYLE_IS_XENO(style)) {
			old_hscale_draw_trough (range);
			return;
		}
		
		gdk_window_get_size (range->trough, &trough_width, &trough_y);
		xt = XENO_STYLE_XTHICKNESS(style);
		yt = XENO_STYLE_YTHICKNESS(style);
		
		trough_height = (trough_y & 0x01) + yt+yt + 2;
		trough_y  = trough_y/2 - yt - 1;
		
		gtk_paint_flat_box (((GtkWidget *)range)->parent->style, range->trough,
							((GtkWidget *)range)->parent->state, GTK_SHADOW_NONE, NULL,
							(GtkWidget *)range, "scale trough", 0, 0, -1, -1);
		gtk_paint_box (style, range->trough,
					   ((GtkWidget*)range)->state==GTK_STATE_INSENSITIVE
					   ? GTK_STATE_INSENSITIVE
					   : GTK_STATE_ACTIVE,
					   GTK_SHADOW_IN, NULL,
					   (GtkWidget *)range, "trough", xt, trough_y, trough_width-xt-xt, trough_height);
		
		if (range->slider) {
			int	slider_x;
			gdk_window_get_position(range->slider, &slider_x, NULL);
			if (slider_x > (xt+xt)) {
				gdk_gc_set_clip_rectangle(style->bg_gc[GTK_STATE_SELECTED], NULL);
				gdk_draw_rectangle (range->trough, style->bg_gc[GTK_STATE_SELECTED], TRUE,
									xt+xt, trough_y + yt,
									slider_x - (xt+xt), trough_height - yt - yt);
			}
		}
	}
}

static void xeno_hscale_draw_slider (GtkRange *range) {
	GtkStateType state_type;
	
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_HSCALE (range));
	
	if (range->slider) {
		if ((range->in_child == RANGE_CLASS (range)->slider) ||
			(range->click_child == RANGE_CLASS (range)->slider))
			state_type = GTK_STATE_PRELIGHT;
		else
			state_type = GTK_STATE_NORMAL;
		
		gtk_paint_slider (GTK_WIDGET (range)->style, range->slider, state_type, 
						  GTK_SHADOW_OUT, NULL, GTK_WIDGET (range), "hscale",
						  0, 0, -1, -1, GTK_ORIENTATION_HORIZONTAL);
	}
}

/* GtkVScale */

static void xeno_vscale_draw_trough (GtkRange *range) {
	GtkStyle *style;
	int trough_x, trough_width, trough_height, xt, yt;

	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_VSCALE (range));
	
	if (range->trough) {
		style = ((GtkWidget *)range)->style;
		
		if (!XENO_STYLE_IS_XENO(style)) {
			old_vscale_draw_trough (range);
			return;
		}
		
		gdk_window_get_size (range->trough, &trough_x, &trough_height);
		style = ((GtkWidget *)range)->style;
		xt = XENO_STYLE_XTHICKNESS(style);
		yt = XENO_STYLE_YTHICKNESS(style);
		
		trough_width = (trough_x & 0x01) + xt+xt + 2;
		trough_x  = trough_x/2 - xt - 1;
		
		gtk_paint_flat_box (((GtkWidget *)range)->parent->style, range->trough,
							((GtkWidget *)range)->parent->state, GTK_SHADOW_NONE, NULL,
							(GtkWidget *)range, "scale trough", 0, 0, -1, -1);
		gtk_paint_box (style, range->trough,
					   ((GtkWidget*)range)->state==GTK_STATE_INSENSITIVE
					   ? GTK_STATE_INSENSITIVE
					   : GTK_STATE_ACTIVE,
					   GTK_SHADOW_IN, NULL,
					   (GtkWidget *)range, "trough", trough_x, yt, trough_width, trough_height - yt - yt);
		
		if (range->slider) {
			int	y;
			gdk_window_get_position(range->slider, NULL, &y);
			
			gdk_gc_set_clip_rectangle(style->bg_gc[GTK_STATE_SELECTED], NULL);
			gdk_draw_rectangle (range->trough, style->bg_gc[GTK_STATE_SELECTED], TRUE, 
								trough_x + xt, y,
								trough_width - xt - xt, trough_height - y - yt - yt);
		}
	}
}

static void xeno_vscale_draw_slider (GtkRange *range) {
	GtkStateType state_type;
	
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_VSCALE (range));
	
	if (range->slider) {
		if ((range->in_child == RANGE_CLASS (range)->slider) ||
			(range->click_child == RANGE_CLASS (range)->slider))
			state_type = GTK_STATE_PRELIGHT;
		else
			state_type = GTK_STATE_NORMAL;

		gtk_paint_slider (GTK_WIDGET (range)->style, range->slider, state_type,
						  GTK_SHADOW_OUT, NULL, GTK_WIDGET (range), "hscale",
						  0, 0, -1, -1, GTK_ORIENTATION_VERTICAL);
	}
}


/* FIXME: make work with Pango */

/*
 *	Combo box
 */

static void xeno_combo_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
	GtkCombo		*combo;
	GtkAllocation	child_allocation;
	gushort			xt, yt;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (allocation != NULL);
	g_return_if_fail (GTK_IS_COMBO(widget));

	old_combo_size_allocate(widget, allocation);

	if (XENO_STYLE_IS_XENO (widget->style)) {
		combo = GTK_COMBO(widget);
		xt = XENO_STYLE_XTHICKNESS(widget->style);
		yt = XENO_STYLE_YTHICKNESS(widget->style);
		
		child_allocation.height	= combo->entry->requisition.height - yt - yt;
		child_allocation.width	= (child_allocation.height - 1) | 1;
		child_allocation.x		= allocation->x
								+ allocation->width
								- child_allocation.width
								- ((GtkContainer *)widget)->border_width
								- xt;
		child_allocation.y		= combo->entry->allocation.y
								+ (combo->entry->allocation.height - combo->entry->requisition.height)/2
								+ yt;
	
		gtk_widget_size_allocate(combo->button, &child_allocation);
	
		child_allocation.x		= combo->entry->allocation.x;
		child_allocation.y		= combo->entry->allocation.y;
		child_allocation.width	= allocation->width - ((GtkContainer *)widget)->border_width * 2;
		child_allocation.height	= allocation->height - ((GtkContainer *)widget)->border_width * 2;
	
		gtk_widget_size_allocate (combo->entry, &child_allocation);
	}
}


/*
 *	Entry
 */

static void
xeno_combo_entry_resize (GtkWidget *widget)
{
	GtkEntry	*entry;
	gushort		text_area_width;
	gint		xoffset, max_offset;

	entry = GTK_ENTRY(widget);
	text_area_width	= widget->allocation.width
					- ((widget->requisition.height - 1) | 1)
					+ XENO_STYLE_YTHICKNESS(widget->style) * 2
					- XENO_STYLE_XTHICKNESS(widget->style) * 2;
	
	gdk_window_resize (entry->text_area,
					   text_area_width,
					   widget->requisition.height - XENO_STYLE_YTHICKNESS(widget->style) * 2);
	
	text_area_width -= XENO_STYLE_XTHICKNESS(widget->style);
	max_offset = entry->char_offset[entry->text_length] - text_area_width;
	if (max_offset < 0)
		max_offset = 0;
	
	if (entry->scroll_offset > max_offset)
		entry->scroll_offset = max_offset;
	
	xoffset = entry->char_offset[GTK_EDITABLE(entry)->current_pos];
	xoffset -= entry->scroll_offset;
	
	if (xoffset < 0)
		entry->scroll_offset += xoffset;
	else if (xoffset > text_area_width)
		entry->scroll_offset += xoffset - text_area_width + 1;
}

static void
xeno_entry_realize (GtkWidget *widget)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_ENTRY(widget));

	old_entry_realize(widget);

	if (   widget->parent
		&& GTK_IS_COMBO(widget->parent)
		&& XENO_STYLE_IS_XENO(widget->parent->style))
	{
		xeno_combo_entry_resize(widget);
	}
}

static void
xeno_entry_size_allocate (GtkWidget		*widget,
						  GtkAllocation	*allocation)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (allocation != NULL);
	g_return_if_fail (GTK_IS_ENTRY(widget));

	old_entry_size_allocate(widget, allocation);

	if (   widget->parent
		&& GTK_IS_COMBO(widget->parent)
		&& GTK_WIDGET_REALIZED(widget)
		&& XENO_STYLE_IS_XENO(widget->parent->style))
	{
		xeno_combo_entry_resize(widget);
	}
}


/* GtkCList */

static void
xeno_clist_realize (GtkWidget *widget)
{
	GdkGCValues	values;
	GdkGC		*gc;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_CLIST (widget));

	old_clist_realize (widget);

	if (XENO_STYLE_IS_XENO (widget->style)) {
		values.foreground = widget->style->base[GTK_STATE_SELECTED];
		values.function = GDK_XOR;
		values.subwindow_mode = GDK_INCLUDE_INFERIORS;
	
		gc = gdk_gc_new_with_values (widget->window, &values,
									 GDK_GC_FOREGROUND | GDK_GC_FUNCTION | GDK_GC_SUBWINDOW);
		if (gc) {
			gdk_gc_unref(GTK_CLIST(widget)->xor_gc);
			GTK_CLIST(widget)->xor_gc = gc;
		}
	}
}


/* GtkSpinButton */

static void
xeno_spin_button_changed (GtkEditable *editable)
{
	GtkEntry	*entry;
	gint		text_area_width;
	gint		char_width, text_width;
	
	g_return_if_fail (editable != NULL);
	g_return_if_fail (GTK_IS_ENTRY (editable));
	g_return_if_fail (GTK_IS_SPIN_BUTTON (editable));
	
	if (!XENO_STYLE_IS_XENO (((GtkWidget *)editable)->style)) {
		old_spin_button_changed (editable);
		return;
	}
	
	entry = GTK_ENTRY(editable);
	
	if (!entry->text_area)
		return;
	
	gdk_window_get_size (entry->text_area, &text_area_width, NULL);
	
	char_width = gdk_char_width (GTK_WIDGET(entry)->style->font, (gchar)'X');
	
	entry->scroll_offset = 0;
	
	/* Right-justify numbers */
	if (entry->text) {
		text_width = gdk_text_width (GTK_WIDGET(entry)->style->font,
									 gtk_entry_get_text (entry),
									 strlen (gtk_entry_get_text(entry)));

		entry->scroll_offset = -(text_area_width - text_width) + 4;
	} else {
		entry->scroll_offset = -text_area_width;
	}
	
	gtk_widget_queue_draw(GTK_WIDGET(entry));
}

static void
xeno_spin_button_size_allocate (GtkWidget		*widget,
								GtkAllocation	*allocation)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_SPIN_BUTTON (widget));
	g_return_if_fail (allocation != NULL);
	
	widget->allocation = *allocation;
	old_spin_button_size_allocate (widget, allocation);
	
	if (GTK_WIDGET_REALIZED (widget))
		xeno_spin_button_changed ((GtkEditable *)widget);
}

static void
xeno_spin_button_realize (GtkWidget *widget) {
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_SPIN_BUTTON(widget));
	
	old_spin_button_realize (widget);
	xeno_spin_button_changed ((GtkEditable *)widget);
}

static gint
xeno_spin_button_button_press_event (GtkWidget *widget, GdkEventButton *event)
{
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_SPIN_BUTTON(widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);
	
	if (!GTK_WIDGET_HAS_FOCUS (widget))
		gtk_widget_grab_focus(widget);
	
	return old_spin_button_button_press_event (widget, event);
}


/*
	GtkCheckButton
	
	The checkbox image is drawn as a glyph in the same font size as the text.
*/
#define XENO_CHECK_BUTTON_SPACING	4

static void
xeno_check_button_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
	GtkToggleButton *toggle_button;
	GtkWidget *style_widget;
	gint	ascent, descent;
	gint	xt, yt, ih, th;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_CHECK_BUTTON (widget));
	g_return_if_fail (requisition != NULL);
	
	if (XENO_STYLE_IS_XENO(widget->style)) {
		toggle_button = GTK_TOGGLE_BUTTON (widget);
		
		requisition->width = requisition->height = 0;
		if (GTK_BIN(widget)->child && GTK_WIDGET_VISIBLE(GTK_BIN(widget)->child))
			gtk_widget_size_request (GTK_BIN(widget)->child, requisition);
		
		if (toggle_button->draw_indicator) {
			xt = XENO_STYLE_XTHICKNESS(widget->style);
			yt = XENO_STYLE_YTHICKNESS(widget->style);
			if ((style_widget = GTK_BIN(widget)->child) == NULL)
				style_widget = widget;
			
			ascent = style_widget->style->font->ascent;
			descent = style_widget->style->font->descent;
			
			/*						:        [    ##############   ]    txt : */
			requisition->width	+=	ascent + xt*4 + XENO_CHECK_BUTTON_SPACING + 2
								+	GTK_CONTAINER(widget)->border_width * 2;
			th					=	descent + requisition->height;
			ih					=	ascent + yt*4;
			requisition->height =	MAX(th, ih) + GTK_CONTAINER(widget)->border_width * 2 + 2;
		}
	} else {
		old_check_button_size_request (widget, requisition);
	}
}

static void
xeno_check_button_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	GtkCheckButton	*check_button;
	GtkToggleButton	*toggle_button;
	GtkButton		*button;
	GtkAllocation	child_allocation;
	gint	xt, ascent, descent, bw;
	gint	l, r;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_CHECK_BUTTON (widget));
	g_return_if_fail (allocation != NULL);
	
	check_button = GTK_CHECK_BUTTON (widget);
	toggle_button = GTK_TOGGLE_BUTTON (widget);
	
	if (XENO_STYLE_IS_XENO(widget->style) && toggle_button->draw_indicator) {
		widget->allocation = *allocation;
		if (GTK_WIDGET_REALIZED (widget))
			gdk_window_move_resize (toggle_button->event_window,
									allocation->x, allocation->y,
									allocation->width, allocation->height);
		
		button = GTK_BUTTON (widget);
		
		if (GTK_BIN (button)->child && GTK_WIDGET_VISIBLE (GTK_BIN (button)->child)) {
			ascent	= GTK_BIN(widget)->child->style->font->ascent;
			descent	= GTK_BIN(widget)->child->style->font->descent;
			xt		= XENO_STYLE_XTHICKNESS(widget->style);
			bw		= GTK_CONTAINER(widget)->border_width;
			
			l		= 1 + xt + xt + ascent + xt + XENO_CHECK_BUTTON_SPACING;
			r		= xt + 1;
			
			child_allocation.x		= widget->allocation.x + l + bw;
			child_allocation.width	= widget->allocation.width - l - r - bw - bw;
			
			child_allocation.y		= widget->allocation.y + 1 + bw;
			child_allocation.height	= widget->allocation.height - 2 - bw - bw;
			
			gtk_widget_size_allocate (GTK_BIN (button)->child, &child_allocation);
		}
	} else {
		old_check_button_size_allocate (widget, allocation);
	}
}


/*
	GtkRadioButton
	
	The current radio button image is not scalable.
*/
static void
xeno_radio_button_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
	old_radio_button_size_request (widget, requisition);
}

static void
xeno_radio_button_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	old_radio_button_size_allocate (widget, allocation);
}

#endif
/* end */

