<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="1.0">
  
  <!-- This file is part of xgridfit.
       It is distributed under the GNU Public License, version 2.
       Copyright (c) 2006-7 by Peter S. Baker
  -->
  
  <xsl:template name="get-highest-function-number">
    <xsl:param name="current-function"/>
    <xsl:param name="current-max" select="0"/>
    <xsl:variable name="new-max">
      <xsl:choose>
        <xsl:when test="number($current-max) &gt;
			number($current-function/@num)">
          <xsl:value-of select="$current-max"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$current-function/@num"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:choose>
      <xsl:when test="$current-function/following-sibling::function[@num]">
        <xsl:call-template name="get-highest-function-number">
          <xsl:with-param name="current-function"
            select="$current-function/following-sibling::function[@num][1]"/>
          <xsl:with-param name="current-max" select="$new-max"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$new-max"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <xsl:template name="get-function-number">
    <xsl:param name="function-name"/>
    <xsl:choose>
      <xsl:when test="key('function-index',$function-name)/@num">
	<xsl:value-of select="key('function-index',$function-name)/@num"/>
      </xsl:when>
      <xsl:otherwise>
	<xsl:value-of select="number($auto-function-base) +
			      number($predefined-functions) +
			      count(key('function-index',
			      $function-name)/preceding-sibling::function[not(@num)])"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <!-- This is called from a <call-function> instruction, but it recursively surveys
       the <param> elements in the <function> being called, counting up those params
       that can be resolved at compile time to numbers between 0 and 255. These are
       "simple" params. Those that are resolved at compile time to larger or negative
       numbers, or must be resolved at run time, are not simple. -->
  <xsl:template match="param" mode="count-simple-params">
    <!-- A container for <with-param> elements. -->
    <xsl:param name="wps"/>
    <!-- The glyph (if any) that is an ancestor of the <call-function>. -->
    <xsl:param name="gl"/>
    <!-- Our running count (initialized to 0). -->
    <xsl:param name="count" select="0"/>
    <xsl:param name="call-macro-param-set"/>
    <xsl:variable name="nm" select="@name"/>
    <xsl:variable name="new-count">
      <xsl:choose>
        <!-- 1. A parameter that matches this one is being passed. -->
        <xsl:when test="$wps/with-param[@name=$nm or @param-id=$nm]">
          <!-- Try to resolve the "value" attribute to a number literal,
               including any offset. -->
          <xsl:variable name="v">
            <xsl:call-template name="get-number-literal">
              <xsl:with-param name="val"
                              select="$wps/with-param[@name=$nm or @param-id=$nm]/@value"/>
              <xsl:with-param name="offset"
                              select="$wps/with-param[@name=$nm or @param-id=$nm]/@offset"/>
              <xsl:with-param name="ancestor-glyph" select="$gl"/>
              <xsl:with-param name="permitted" select="'1xrnc'"/>
	      <xsl:with-param name="call-macro-param-set" select="$call-macro-param-set"/>
            </xsl:call-template>
          </xsl:variable>
          <!-- Is v within range? -->
          <xsl:choose>
            <xsl:when test="number($v) &gt;= 0 and number($v) &lt; 256">
              <xsl:value-of select="number($count) + 1"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="$count"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <!-- 2. No parameter has been passed, but this param declaration
                has a default value. -->
        <xsl:when test="@value">
          <xsl:variable name="v">
            <xsl:call-template name="get-number-literal">
              <xsl:with-param name="val" select="@value"/>
              <xsl:with-param name="permitted" select="'1xrnc'"/>
	      <xsl:with-param name="call-macro-param-set" select="$call-macro-param-set"/>
            </xsl:call-template>
          </xsl:variable>
          <xsl:choose>
            <xsl:when test="number($v) &gt;= 0 and number($v) &lt; 256">
              <xsl:value-of select="number($count) + 1"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="$count"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
          <!-- This script is going to fail, but it will be caught when the
               parameters are actually pushed. -->
          <xsl:value-of select="$count"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <!-- Are we done, or shall we recurse? -->
    <xsl:choose>
      <xsl:when test="following-sibling::param">
        <xsl:apply-templates select="following-sibling::param[1]"
                             mode="count-simple-params">
          <xsl:with-param name="wps" select="$wps"/>
          <xsl:with-param name="gl" select="$gl"/>
          <xsl:with-param name="count" select="$new-count"/>
	  <xsl:with-param name="call-macro-param-set" select="$call-macro-param-set"/>
        </xsl:apply-templates>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$new-count"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <!-- The <param-set> template pushes a set of function parameters onto the
       stack in preparation for a function call. Various complications: we
       determine what the parameters are and their proper order by looking
       into the function definition itself, so we lose the present context.
       Also, we'd like to optimize if possible by pushing all the parameters
       with a single PUSH command. Finally, we may omit a <with-param> element
       from the <param-set> if the corresponding <param> in the function
       definition has a default value; so we need to be ready to look for
       that default if the <with-param> is omitted here. -->
  <xsl:template match="param-set|call-function" mode="function">
    <xsl:param name="call-macro-param-set"/>
    <xsl:param name="f">
      <xsl:choose>
	<xsl:when test="ancestor-or-self::call-function/@name">
	  <xsl:value-of select="ancestor-or-self::call-function/@name"/>
	</xsl:when>
	<xsl:otherwise>
	  <xsl:value-of select="ancestor-or-self::call-function/@function-id"/>
	</xsl:otherwise>
      </xsl:choose>
    </xsl:param>
    <xsl:param name="all-params" select="key('function-index',$f)/param |
					 key('function-index',$f)/params/param"/>
    <!-- Check the with-param elements for validity, since the schema does little -->
    <xsl:apply-templates select="with-param" mode="check-func"/>
    <!-- These variables allow us to carry some stuff from this
         context along when the for-each loop changes the context. -->
    <xsl:variable name="wps" select="."/>
    <xsl:variable name="ancestor-glyph" select="ancestor::glyph"/>
    <xsl:variable name="ancestor-function" select="ancestor::function"/>
    <xsl:variable name="ancestor-prep" select="ancestor::pre-program"/>
    <xsl:variable name="p">
      <xsl:choose>
        <xsl:when test="ancestor::function">
          <xsl:text>1xmpfvnc</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text>1xmpvnc</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:variable name="simple-param-count">
      <xsl:apply-templates select="$all-params[1]"
                           mode="count-simple-params">
        <xsl:with-param name="wps" select="."/>
        <xsl:with-param name="gl" select="ancestor::glyph"/>
	<xsl:with-param name="call-macro-param-set" select="$call-macro-param-set"/>
      </xsl:apply-templates>
    </xsl:variable>
    <!-- Is this clear enough? This variable is true when we CAN optimize by
         pushing all the parameters with a single PUSHB command. The main test
         is that the number of "simple" params (those that can be resolved to
         numbers between 0 and 255 at compile time) is equal to the number of
         paramters to push. This optimiztion can be forced by the attribute
         opt="yes" or inhibited by the attribute opt="no" but is not influenced
         by the absence of an opt attribute. -->
    <xsl:variable name="add-mode-eligible"
                  select="boolean((@opt = 'yes' or
                          number($simple-param-count) =
                          count($all-params)) and
                          (not(@opt) or @opt != 'no'))"/>
    <!-- We're going to loop backwards through the params in the function element.
         As we go we try to match up with-param elements in the present param-set
         with the param elements in the function. It is an error if any are
         missing. -->
    <xsl:for-each select="$all-params">
      <xsl:sort select="position()" order="descending"/>
      <!-- The value of $am determines whether we want the PUSH instruction
           output for this iteration of the loop. Criteria are, first, that
           it has been determined above that it's all right to push all these
           parameters with a single PUSH instruction, and, second, that this
           is not the first parameter we're pushing (the *last* in the function's
           param list); for of course the first number to be pushed must be
           preceded by the PUSH instruction. -->
      <xsl:variable name="am" select="boolean($add-mode-eligible and
                                      following-sibling::param)"/>
      <xsl:variable name="exp">
        <xsl:choose>
          <xsl:when test="$add-mode-eligible">
            <xsl:value-of select="$simple-param-count"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="1"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:variable>
      <xsl:variable name="nm" select="@name"/>
      <!-- The first choice is a value that is passed as a parameter. The second
           choice is the default value defined in the function itself. The third
           choice is failure. -->
      <xsl:choose>
        <xsl:when test="$wps/with-param[@name=$nm or @param-id=$nm]">
          <xsl:call-template name="push-value">
            <xsl:with-param name="val"
                            select="$wps/with-param[@name=$nm or @param-id=$nm]/@value"/>
            <xsl:with-param name="ancestor-glyph" select="$ancestor-glyph"/>
            <xsl:with-param name="ancestor-function" select="$ancestor-function"/>
            <xsl:with-param name="ancestor-prep" select="$ancestor-prep"/>
            <xsl:with-param name="add-mode" select="$am"/>
            <xsl:with-param name="offset"
                            select="$wps/with-param[@name=$nm or @param-id=$nm]/@offset"/>
            <xsl:with-param name="permitted" select="$p"/>
            <xsl:with-param name="expect" select="$exp"/>
	    <xsl:with-param name="call-macro-param-set"
			    select="$call-macro-param-set"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:when test="@value">
          <xsl:call-template name="push-value">
            <xsl:with-param name="val" select="@value"/>
            <xsl:with-param name="ancestor-glyph" select="$ancestor-glyph"/>
            <xsl:with-param name="ancestor-function" select="$ancestor-function"/>
            <xsl:with-param name="ancestor-prep" select="$ancestor-prep"/>
            <xsl:with-param name="add-mode" select="$am"/>
            <xsl:with-param name="permitted" select="'1xn'"/>
            <xsl:with-param name="expect" select="$exp"/>
	    <xsl:with-param name="call-macro-param-set"
			    select="$call-macro-param-set"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <!-- Oops! A parameter is missing. -->
          <xsl:call-template name="error-message">
            <xsl:with-param name="msg">
              <xsl:text>Parameter "</xsl:text>
              <xsl:value-of select="current()/@name"/>
              <xsl:text>" missing in call to function "</xsl:text>
              <xsl:value-of select="$f"/>
              <xsl:text>"</xsl:text>
            </xsl:with-param>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:for-each>
  </xsl:template>

  <xsl:template match="call-function">
    <xsl:param name="call-macro-param-set"/>
    <xsl:variable name="fid">
      <xsl:choose>
	<xsl:when test="@name">
	  <xsl:value-of select="@name"/>
	</xsl:when>
	<xsl:when test="@function-id">
	  <xsl:value-of select="@function-id"/>
	</xsl:when>
	<xsl:otherwise>
	  <xsl:call-template name="error-message">
	    <xsl:with-param name="msg">
	      <xsl:text>Encountered &lt;call-function&gt; element without name or function-id</xsl:text>
	    </xsl:with-param>
	  </xsl:call-template>
	</xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <!-- If we're calling this from a function, we need to preserve
         the variable $var-function-stack-count and restore it after
         we're done. -->
    <xsl:if test="ancestor::function">
      <xsl:call-template name="number-command">
        <xsl:with-param name="num" select="$var-function-stack-count"/>
        <xsl:with-param name="cmd" select="'RS'"/>
      </xsl:call-template>
    </xsl:if>
    <!-- If the glyph program, function or pre-program we are calling from
         contains variables, we have to protect them from overwriting by
         saving the variable frame: push var-frame-bottom and copy
         var-frame-top to var-frame bottom. If function being called
         uses variables, it will set var-frame-top. -->
    <xsl:variable name="have-variable"
                  select="boolean(ancestor::glyph/variables/variable|
			  ancestor::glyph/variable|
                          ancestor::function/variables/variable|
			  ancestor::function/variable|
                          ancestor::pre-program/variables/variable|
			  ancestor::pre-program/variable)"/>
    <xsl:if test="$have-variable">
      <xsl:call-template name="number-command">
        <xsl:with-param name="num" select="$var-frame-bottom"/>
        <xsl:with-param name="cmd" select="'RS'"/>
      </xsl:call-template>
    </xsl:if>
    <!-- push the parameters onto the stack. -->
    <xsl:choose>
      <xsl:when test="param-set">
	<xsl:apply-templates select="param-set" mode="function">
	  <xsl:with-param name="call-macro-param-set"
			  select="$call-macro-param-set"/>
	</xsl:apply-templates>
      </xsl:when>
      <xsl:otherwise>
	<xsl:apply-templates select="." mode="function">
	  <xsl:with-param name="call-macro-param-set"
			  select="$call-macro-param-set"/>
	</xsl:apply-templates>
      </xsl:otherwise>
    </xsl:choose>
    <!-- continue saving variable frame. -->
    <xsl:if test="$have-variable">
      <xsl:call-template name="push-num">
        <xsl:with-param name="num" select="$var-frame-bottom"/>
      </xsl:call-template>
      <xsl:call-template name="number-command">
        <xsl:with-param name="num" select="$var-frame-top"/>
        <xsl:with-param name="cmd" select="'RS'"/>
      </xsl:call-template>
      <xsl:call-template name="simple-command">
        <xsl:with-param name="cmd" select="'WS'"/>
      </xsl:call-template>
    </xsl:if>
    <!-- Push the index of this function. This always matches the number that
         was used to define the function when the fgpm table was set up. -->
    <xsl:call-template name="push-num">
      <xsl:with-param name="num">
        <xsl:call-template name="get-function-number">
          <xsl:with-param name="function-name" select="$fid"/>
        </xsl:call-template>
      </xsl:with-param>
    </xsl:call-template>
    <!-- Call via CALL or LOOPCALL, depending on whether we have two sets of
         parameters, or more, or fewer. -->
    <xsl:choose>
      <xsl:when test="count(param-set) &gt; 1">
        <xsl:call-template name="push-num">
          <xsl:with-param name="num" select="count(param-set)"/>
        </xsl:call-template>
        <xsl:call-template name="simple-command">
          <xsl:with-param name="cmd" select="'SWAP'"/>
        </xsl:call-template>
        <xsl:call-template name="simple-command">
          <xsl:with-param name="cmd" select="'LOOPCALL'"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="simple-command">
          <xsl:with-param name="cmd" select="'CALL'"/>
        </xsl:call-template>
      </xsl:otherwise>
    </xsl:choose>
    <!-- Now restore variable frame. Copy var-frame-bottom to var-frame-top
         and then copy the number we saved to the stack before the call into
         var-frame-bottom. -->
    <xsl:if test="$have-variable">
      <xsl:call-template name="push-num">
        <xsl:with-param name="num" select="$var-frame-top"/>
      </xsl:call-template>
      <xsl:call-template name="number-command">
        <xsl:with-param name="num" select="$var-frame-bottom"/>
        <xsl:with-param name="cmd" select="'RS'"/>
      </xsl:call-template>
      <xsl:call-template name="simple-command">
        <xsl:with-param name="cmd" select="'WS'"/>
      </xsl:call-template>
      <xsl:call-template name="stack-top-to-storage">
        <xsl:with-param name="loc" select="$var-frame-bottom"/>
      </xsl:call-template>
    </xsl:if>
    <!-- If we're calling this from a function, we need to restore
         the previous value of $var-function-stack-count (it's supposed
         to be on top of the stack after the function returns). -->
    <xsl:if test="ancestor::function">
      <xsl:call-template name="stack-top-to-storage">
        <xsl:with-param name="loc" select="$var-function-stack-count"/>
      </xsl:call-template>
    </xsl:if>
    <!-- If the function we are calling has the attribute return="yes" then
         we collect the return value from var-return-value. We'd like to
         store it in a variable (or whatever) referenced by "result-to", but
         if that attribute is missing we issue a warning and leave the value
         on the stack. -->
    <xsl:choose>
      <xsl:when test="key('function-index',$fid)/@return = 'yes'">
        <xsl:choose>
          <xsl:when test="@result-to">
            <xsl:call-template name="number-command">
              <xsl:with-param name="num" select="$var-return-value"/>
              <xsl:with-param name="cmd" select="'RS'"/>
            </xsl:call-template>
            <xsl:call-template name="store-value">
              <xsl:with-param name="vname" select="@result-to"/>
	      <xsl:with-param name="call-macro-param-set"
			      select="$call-macro-param-set"/>
            </xsl:call-template>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="warning">
              <xsl:with-param name="msg">
                <xsl:text>Function </xsl:text>
                <xsl:value-of select="$fid"/>
                <xsl:text> returns a value, but I can</xsl:text>
                <xsl:value-of select="$newline"/>
                <xsl:text>find no place to put it (no "result-to" attribute).</xsl:text>
                <xsl:value-of select="$newline"/>
                <xsl:text>It is being left on the stack.</xsl:text>
              </xsl:with-param>
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <xsl:if test="@result-to">
          <xsl:call-template name="warning">
            <xsl:with-param name="msg">
              <xsl:text>The return value of a function cannot be read </xsl:text>
              <xsl:text>unless the</xsl:text>
              <xsl:value-of select="$newline"/>
              <xsl:text>attribute return="yes" is present. </xsl:text>
              <xsl:text>I am ignoring the "result-to"</xsl:text>
              <xsl:value-of select="$newline"/>
              <xsl:text>attribute of this call to function </xsl:text>
              <xsl:value-of select="$fid"/>
              <xsl:text>.</xsl:text>
            </xsl:with-param>
          </xsl:call-template>
        </xsl:if>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="function">
    <xsl:variable name="omit-initial-newline-i">
      <xsl:choose>
        <xsl:when test="@num and not(preceding-sibling::function[@num])">
          <xsl:text>1</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text>0</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:variable name="omit-initial-newline" select="boolean(number($omit-initial-newline-i))"/>
    <xsl:call-template name="number-command">
      <xsl:with-param name="num">
        <xsl:call-template name="get-function-number">
          <xsl:with-param name="function-name">
	    <xsl:choose>
	      <xsl:when test="@name">
		<xsl:value-of select="@name"/>
	      </xsl:when>
	      <xsl:when test="@id">
		<xsl:value-of select="@id"/>
	      </xsl:when>
	      <xsl:otherwise>
		<xsl:call-template name="error-message">
		  <xsl:with-param name="msg">
		    <xsl:text>Encountered &lt;function&lt; lacking name and id attributes.</xsl:text>
		  </xsl:with-param>
		</xsl:call-template>
	      </xsl:otherwise>
	    </xsl:choose>
	  </xsl:with-param>
        </xsl:call-template>
      </xsl:with-param>
      <xsl:with-param name="cmd" select="'FDEF'"/>
      <xsl:with-param name="with-leading-newline" select="not($omit-initial-newline)"/>
    </xsl:call-template>
    <!-- First save the current depth of the stack in the Storage Location
         var-function-stack-count. This will help us find
         parameters when we need them, even if stuff has been pushed onto the
         stack in the meantime. -->
    <xsl:if test="param|params/param">
      <xsl:call-template name="simple-command">
        <xsl:with-param name="cmd" select="'DEPTH'"/>
      </xsl:call-template>
      <xsl:call-template name="stack-top-to-storage">
        <xsl:with-param name="loc" select="$var-function-stack-count"/>
      </xsl:call-template>        
    </xsl:if>
    <!-- If we have declared variables, place the new top of the variable
         frame in var-frame-top. -->
    <xsl:if test="variable or variables/variable">
      <xsl:call-template name="push-num">
        <xsl:with-param name="num" select="$var-frame-top"/>
      </xsl:call-template>
      <xsl:call-template name="number-command">
        <xsl:with-param name="num" select="$var-frame-bottom"/>
        <xsl:with-param name="cmd" select="'RS'"/>
      </xsl:call-template>
      <xsl:call-template name="number-command">
        <xsl:with-param name="num" select="count(variable|variables/variable)"/>
        <xsl:with-param name="cmd" select="'ADD'"/>
      </xsl:call-template>
      <xsl:call-template name="simple-command">
        <xsl:with-param name="cmd" select="'WS'"/>
      </xsl:call-template>
      <!-- Also, initialize any that want initializing. -->
      <xsl:apply-templates select="variable|variables/variable" mode="initialize"/>
    </xsl:if>
    <!-- If the function is declared as expecting a return value, initialize
         var-return-value to zero. This is a safety measure, since it is
         impractical (right now at least) to check that the function contains
         an instruction that writes to that location. -->
    <xsl:if test="@return = 'yes'">
      <xsl:call-template name="push-num">
        <xsl:with-param name="num" select="$var-return-value"/>
        <xsl:with-param name="expect" select="2"/>
      </xsl:call-template>
      <xsl:call-template name="push-num">
        <xsl:with-param name="num" select="0"/>
        <xsl:with-param name="add-mode" select="true()"/>
      </xsl:call-template>
      <xsl:call-template name="simple-command">
        <xsl:with-param name="cmd" select="'WS'"/>
      </xsl:call-template>
    </xsl:if>
    <!-- Now execute instructions. -->
    <xsl:apply-templates/>
    <!-- Pop all the parameters off the stack, and we're done. -->
    <xsl:for-each select="param|params/param">
      <xsl:call-template name="simple-command">
        <xsl:with-param name="cmd" select="'POP'"/>
      </xsl:call-template>
    </xsl:for-each>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'ENDF'"/>
    </xsl:call-template>
  </xsl:template>

  <xsl:template match="legacy-functions">
    <xsl:for-each select="*">
      <xsl:apply-templates select=".">
	<xsl:with-param name="with-leading-newline"
			select="boolean(preceding-sibling::*)"/>
      </xsl:apply-templates>
    </xsl:for-each>
  </xsl:template>

  <xsl:template match="macro">
    <xsl:param name="call-macro-param-set"/>
    <xsl:apply-templates>
      <xsl:with-param name="call-macro-param-set"
		      select="$call-macro-param-set"/>
    </xsl:apply-templates>
  </xsl:template>

  <xsl:template match="with-param" mode="check-macro">
    <xsl:if test="not(@value) and not(set) and not(range) and not(line)">
      <xsl:call-template name="error-message">
	<xsl:with-param name="msg">
	  <xsl:text>In call to macro "</xsl:text>
	  <xsl:choose>
	    <xsl:when test="@name">
	      <xsl:value-of select="@name"/>
	    </xsl:when>
	    <xsl:when test="@macro-id">
	      <xsl:value-of select="@macro-id"/>
	    </xsl:when>
	  </xsl:choose>
	  <xsl:value-of select="ancestor::call-macro/@macro-id"/>
	  <xsl:text>" &lt;with-param&gt; element "</xsl:text>
	  <xsl:choose>
	    <xsl:when test="@name">
	      <xsl:value-of select="@name"/>
	    </xsl:when>
	    <xsl:otherwise>
	      <xsl:value-of select="@param-id"/>
	    </xsl:otherwise>
	  </xsl:choose>
	  <xsl:text>" must have a value attribute or contain &lt;line&gt;, &lt;set&gt; or &lt;range&gt;</xsl:text>
	</xsl:with-param>
      </xsl:call-template>
    </xsl:if>
  </xsl:template>

  <xsl:template match="with-param" mode="check-func">
    <xsl:variable name="fid">
      <xsl:choose>
	<xsl:when test="ancestor::call-function/@name">
	  <xsl:value-of select="ancestor::call-function/@name"/>
	</xsl:when>
	<xsl:otherwise>
	  <xsl:value-of select="ancestor::call-function/@function-id"/>
	</xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:if test="not(@value)">
      <xsl:call-template name="error-message">
	<xsl:with-param name="msg">
	  <xsl:text>In call to function "</xsl:text>
	  <xsl:value-of select="$fid"/>
	  <xsl:text>" &lt;with-param&gt; element "</xsl:text>
	  <xsl:choose>
	    <xsl:when test="@name">
	      <xsl:value-of select="@name"/>
	    </xsl:when>
	    <xsl:otherwise>
	      <xsl:value-of select="@param-id"/>
	    </xsl:otherwise>
	  </xsl:choose>
	  <xsl:text>" must have a value attribute</xsl:text>
	</xsl:with-param>
      </xsl:call-template>
    </xsl:if>
  </xsl:template>

  <xsl:template match="param-set" mode="macro">
    <xsl:param name="mid"/>
    <xsl:apply-templates select="with-param" mode="check-macro"/>
    <xsl:apply-templates select="key('macro-index',$mid)">
      <xsl:with-param name="call-macro-param-set" select="with-param"/>
    </xsl:apply-templates>
  </xsl:template>

  <xsl:template match="params"></xsl:template>

  <xsl:template match="call-macro">
    <xsl:variable name="mid">
      <xsl:choose>
	<xsl:when test="@name">
	  <xsl:value-of select="@name"/>
	</xsl:when>
	<xsl:when test="@macro-id">
	  <xsl:value-of select="@macro-id"/>
	</xsl:when>
	<xsl:otherwise>
	  <xsl:call-template name="error-message">
	    <xsl:with-param name="msg">
	      <xsl:text>Encountered a &lt;call-macro&gt; without name or macro-id.</xsl:text>
	    </xsl:with-param>
	  </xsl:call-template>
	</xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:if test="not(key('macro-index',$mid))">
      <xsl:call-template name="error-message">
	<xsl:with-param name="msg">
	  <xsl:text>Can't find macro </xsl:text>
	  <xsl:value-of select="$mid"/>
	</xsl:with-param>
      </xsl:call-template>
    </xsl:if>
    <xsl:choose>
      <xsl:when test="param-set">
	<xsl:apply-templates select="param-set" mode="macro">
	  <xsl:with-param name="mid" select="$mid"/>
	</xsl:apply-templates>
      </xsl:when>
      <xsl:when test="with-param">
	<xsl:apply-templates select="with-param" mode="check-macro"/>
	<xsl:apply-templates select="key('macro-index',$mid)">
	  <xsl:with-param name="call-macro-param-set" select="with-param"/>
	</xsl:apply-templates>
      </xsl:when>
      <xsl:otherwise>
	<xsl:apply-templates select="key('macro-index',$mid)"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

</xsl:stylesheet>
