#include "kj.h"

#define MAXLINELEN	512
#define MAXARGS		32

/* ----------------------------------------------------------------------
 *  Stolen from XMMS :)
 */
void kj_del_directory(const char *dirname)
{
	DIR *dir;
	struct dirent *dirent;
	char *file;

	if((dir = opendir(dirname)))
	{
		while((dirent = readdir(dir)))
		{
			if (strcmp(dirent->d_name, ".") && strcmp(dirent->d_name, ".."))
			{
				file = (char *) g_malloc(strlen(dirname) + strlen(dirent->d_name) + 2);
				sprintf(file, "%s/%s", dirname, dirent->d_name);
				if (unlink(file) == -1)
					if (errno == EISDIR)
						kj_del_directory(file);
				g_free(file);
			}
		}
	}
	rmdir(dirname);
}


/* ----------------------------------------------------------------------
 *  Stolen from XMMS :)  Modified to also look for a file with a specified
 *  extension when ext = 1.  Used to find the .rc file.
 */
gchar *kj_find_file_recursively(const char *dirname, const char *file, const char ext)
{
	DIR *dir;
	struct dirent *dirent;
	gchar *result, *ending;
	gchar *found;
	struct stat buf;
	int f;

	if((dir = opendir(dirname)))
	{
		while((dirent = readdir(dir)))
		{
			/* Don't scan "." and ".." */
			if (strcmp(dirent->d_name, ".") == 0 || strcmp(dirent->d_name, "..") == 0)
				continue;
			/* We need this in order to find out if file is directory */
			found = g_strdup_printf("%s/%s", dirname, dirent->d_name);
			if (stat(found, &buf) != 0)
			{	/* Error occured */
				g_free(found);
				closedir(dir);
				return NULL;
			}
			if (S_ISDIR(buf.st_mode))
			{	/* Found directory -- descend into it */
				result = kj_find_file_recursively(found, file, ext);
				if (result != NULL)
				{	/* Found file there -- exit */
					g_free(found);
					closedir(dir);
					return result;
				}
			}
			else
			{	/* Normal file -- maybe just what we are looking for? */
				if(ext)
					{
					ending = strrchr(dirent->d_name, '.');
					f = (ending ? !strcasecmp(ending, file) : 0);
					}
				else f = !strcasecmp(dirent->d_name, file);
				if (f)
				{	/* Yes! */
					if (strlen(found) > FILENAME_MAX)
					{	/* No good! File + path too long */
						g_free(found);
						closedir(dir);
						return NULL;
					}
					else
					{
						closedir(dir);
						return found;
					}
				}
			}
			g_free(found);
		}
		closedir(dir);
	}
	return NULL;
}


/* ---------------------------------------------------------------------- */
k_image *kj_read_skin(gchar *path, gchar *fname, int pix)
{
k_image *img = NULL;
char *filename;

	if((filename = kj_find_file_recursively(path, fname, 0)))
		{
		if((img = kj_read_image(filename, pix)) == NULL)
			printf("Warning, kj image `%s' not opened.\n", fname);
		g_free(filename);
		}
	return img;
}


/* ---------------------------------------------------------------------- */
void set_area(int f, k_area *area, int argc, char **argv)
{
	if(argc < 5) return;
	area->type = f;
	area->focus = area->pressed = area->bm = 0;
	area->x1 = atoi(argv[1]);
	area->y1 = atoi(argv[2]);
	area->x2 = atoi(argv[3]);
	area->y2 = atoi(argv[4]);
	if(area->x1 > area->x2 || area->y1 > area->y2) printf("WARNING: bad area!\n");
	if(argc > 6)
		{
		if(!strcasecmp(argv[6], "BMP1")) area->bm = 0;
		else if(!strcasecmp(argv[6], "BMP2")) area->bm = 1;
		else if(!strcasecmp(argv[6], "BMP3")) area->bm = 2;
		}
}


/* ---------------------------------------------------------------------- */
void set_colour(GdkColor *col, int argc, char **argv)
{
	if(argc < 3) return;
	col->red = (atoi(argv[1]) << 8) | 0xff;
	col->green = (atoi(argv[2]) << 8) | 0xff;
	col->blue = (atoi(argv[3]) << 8) | 0xff;
	gdk_color_alloc(gdk_colormap_get_system(), col);
}


/* ---------------------------------------------------------------------- */
k_res_area resource_areas[] = {
	{ "VolumeControlButton", BUTTON_VOLUME, A_SLIDER },
	{ "CloseButton", BUTTON_CLOSE, A_BUTTON },
	{ "MinimizeButton", BUTTON_MINIMIZE, A_BUTTON },
	{ "AboutButton", BUTTON_ABOUT, A_BUTTON },
	{ "OpenFileButton", BUTTON_OPEN, A_BUTTON },
	{ "StopButton", BUTTON_STOP, A_BUTTON },
	{ "PlayButton", BUTTON_PLAY, A_BUTTON },
	{ "RewindButton", BUTTON_REWIND, A_BUTTON },
	{ "ForwardButton", BUTTON_FASTFORWARD, A_BUTTON },
	{ "PreviousSongButton", BUTTON_PREVIOUSSONG, A_BUTTON },
	{ "NextSongButton", BUTTON_NEXTSONG, A_BUTTON },
	{ "EqualizerButton", BUTTON_EQUALIZER, A_BUTTON },
	{ "PauseButton", BUTTON_PAUSE, A_BUTTON },
	{ "PreferencesButton", BUTTON_OPTIONS, A_BUTTON },
	{ "RepeatButton", BUTTON_REPEAT, A_TOGGLE },
	{ "SeekRegion", REGION_SEEK, A_SLIDER },
	{ "MP3KhzWindow", REGION_MP3KHZ, A_TEXT },
	{ "MP3KbpsWindow", REGION_MP3KBPS, A_TEXT },
	{ "MP3TimeWindow", REGION_MP3TIME, A_TEXT },
	{ "VolumeText", REGION_VOLTEXT, A_TEXT },
	{ "FilenameWindow", REGION_FNAME, A_TEXT },
	{ "CurrentTrackWindow", REGION_TRACK, A_TEXT },
	{ "EqualizerResetButton", BUTTON_EQURESET, A_BUTTON },
	{ "EqualizerOnButton", BUTTON_EQUON, A_BUTTON },
	{ "EqualizerOffButton", BUTTON_EQUOFF, A_BUTTON },
	{ "ShuffleButton", BUTTON_SHUFFLE, A_BUTTON },
	{ "PlayListButton", BUTTON_PLAYLIST, A_BUTTON },
	{ "DockModeButton", BUTTON_DOCKMODE, A_BUTTON },
	{ "UnDockModeButton", BUTTON_UNDOCK, A_BUTTON },
	{ "SpectrumAnalyzerButton", BUTTON_SANALYZER, A_TEXT },
	{ "OscilloscopeButton", BUTTON_OSCILLO, A_TEXT },	/* need to fix for default skin */
	{ "AnalyzerWindow", REGION_ANALYZER, A_ANALYZER },
	{ "PitchControlButton", BUTTON_PITCH, A_SLIDER },
	{ "PitchText", REGION_PITCHTEXT, A_TEXT },

	{ "PlaylistWindowText", PLREGION_TEXT, A_PLIST },
	{ "PlaylistWindowNbSelectedTrack", PLREGION_SELTRACK, A_TEXT },
	{ "PlaylistWindowNbTotalTracks", PLREGION_TTLTRACK, A_TEXT },
	{ "PlaylistWindowLinkButton", PLBUTTON_LINK, A_TOGGLE },
	{ "PlaylistWindowCloseButton", PLBUTTON_CLOSE, A_BUTTON },
	{ "PlaylistWindowOpenButton", PLBUTTON_OPEN, A_BUTTON },
	{ "PlaylistWindowMinimizeButton", PLBUTTON_MINIMIZE, A_BUTTON },
	{ "PlaylistWindowUpButton", PLBUTTON_UP, A_BUTTON },
	{ "PlaylistWindowDownButton", PLBUTTON_DOWN, A_BUTTON },
	{ "PlaylistWindowSortButton", PLBUTTON_SORT, A_BUTTON },
	{ "PlaylistWindowShuffleButton", PLBUTTON_SHUFFLE, A_BUTTON },
	{ "PlaylistWindowSortInverseButton", PLBUTTON_INVERSE, A_BUTTON },
	{ "PlaylistWindowAddButton", PLBUTTON_ADD, A_BUTTON },
	{ "PlaylistWindowResetButton", PLBUTTON_RESET, A_BUTTON },
	{ "PlaylistWindowDelButton", PLBUTTON_DELETE, A_BUTTON },
	{ "PlaylistWindowLoadPlaylistButton", PLBUTTON_LOAD, A_BUTTON },
	{ "PlaylistWindowSavePlaylistButton", PLBUTTON_SAVE, A_BUTTON },
	{ "PlaylistWindowSelectionUpButton", PLBUTTON_SELUP, A_BUTTON },
	{ "PlaylistWindowSelectionDownButton", PLBUTTON_SELDOWN, A_BUTTON },
	{ "PlaylistWindowPreviousButton", PLBUTTON_PREVIOUS, A_BUTTON },
	{ "PlaylistWindowPlayButton", PLBUTTON_PLAY, A_BUTTON },
	{ "PlaylistWindowNextButton", PLBUTTON_NEXT, A_BUTTON },
	{ "PlaylistWindowPauseButton", PLBUTTON_PAUSE, A_BUTTON },
	{ "PlaylistWindowStopButton", PLBUTTON_STOP, A_BUTTON },
	{ "PlaylistWindowAboutButton", PLBUTTON_ABOUT, A_BUTTON },
	{ "PlaylistWindowNbSelectedTrackTime", PLREGION_SELTIME, A_TEXT },
	{ "PlaylistWindowPreferencesButton", PLBUTTON_PREFS, A_BUTTON },
	{ "PlaylistWindowPageUpButton", PLBUTTON_PGUP, A_BUTTON },
	{ "PlaylistWindowPageDownButton", PLBUTTON_PGDOWN, A_BUTTON },

	{ NULL, -1, -1 }
};

/* ---------------------------------------------------------------------- */
void set_value(gchar *path, k_resource *res, k_area *areas, int argc, char **argv)
{
k_res_area *ra;

	if(argc < 1) return;
	if(!strcasecmp(argv[0], "BackgroundImage")) res->background = kj_read_skin(path, argv[1], 2);
	else if(!strcasecmp(argv[0], "BackgroundImageInactive")) res->inactive = kj_read_skin(path, argv[1], 2);
	else if(!strcasecmp(argv[0], "BackgroundImagePressed1")) res->pressed[0] = kj_read_skin(path, argv[1], 2);
	else if(!strcasecmp(argv[0], "BackgroundImagePressed2")) res->pressed[1] = kj_read_skin(path, argv[1], 2);
	else if(!strcasecmp(argv[0], "BackgroundImagePressed3")) res->pressed[2] = kj_read_skin(path, argv[1], 2);

	else if(!strcasecmp(argv[0], "FontImage")) res->Font.img = kj_read_skin(path, argv[1], 1);
	else if(!strcasecmp(argv[0], "TimeFontImage")) res->TimeFont.img = kj_read_skin(path, argv[1], 1);
	else if(!strcasecmp(argv[0], "VolumeFontImage")) res->VolumeFont.img = kj_read_skin(path, argv[1], 1);
	else if(!strcasecmp(argv[0], "PitchFontImage")) res->PitchFont.img = kj_read_skin(path, argv[1], 1);

	else if(!strcasecmp(argv[0], "FontSize"))
		{ res->Font.size[0] = atoi(argv[1]); res->Font.size[1] = atoi(argv[2]); }
	else if(!strcasecmp(argv[0], "FontSpacing")) res->Font.spacing = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "FontTransparent")) res->Font.transparent = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "TimeFontSize"))
		{ res->TimeFont.size[0] = atoi(argv[1]); res->TimeFont.size[1] = atoi(argv[2]); }
	else if(!strcasecmp(argv[0], "TimeFontSpacing")) res->TimeFont.spacing = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "TimeFontTransparent")) res->TimeFont.transparent = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "VolumeFontSize"))
		{ res->VolumeFont.size[0] = atoi(argv[1]); res->VolumeFont.size[1] = atoi(argv[2]); }
	else if(!strcasecmp(argv[0], "VolumeFontSpacing")) res->VolumeFont.spacing = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "VolumeFontTransparent")) res->VolumeFont.transparent = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "PitchFontSize"))
		{ res->PitchFont.size[0] = atoi(argv[1]); res->PitchFont.size[1] = atoi(argv[2]); }
	else if(!strcasecmp(argv[0], "PitchFontSpacing")) res->PitchFont.spacing = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "PitchFontTransparent")) res->PitchFont.transparent = atoi(argv[1]);

	else if(!strcasecmp(argv[0], "About"))
		{
		if(argc > 1 && res->about_str_pos < MAX_ABOUT_STR)
			{
			if(res->about_str[res->about_str_pos]) g_free(res->about_str[res->about_str_pos]);
			res->about_str[res->about_str_pos] = g_strdup(argv[1]);
			res->about_str_pos++;
			if(res->about_str_pos > res->about_str_num) res->about_str_num = res->about_str_pos;
			}
		}

	else if(!strcasecmp(argv[0], "VolumeControlImage")) res->volume = kj_read_skin(path, argv[1], 2);
	else if(!strcasecmp(argv[0], "VolumeControlImagePosition")) res->volume_position = kj_read_skin(path, argv[1], 0);
	else if(!strcasecmp(argv[0], "VolumeControlImageXSize")) res->VolumeControlImageXSize = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "VolumeControlImageNb")) res->VolumeControlImageNb = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "PitchControlImage")) res->pitch = kj_read_skin(path, argv[1], 2);
	else if(!strcasecmp(argv[0], "PitchControlImagePosition")) res->pitch_position = kj_read_skin(path, argv[1], 0);
	else if(!strcasecmp(argv[0], "PitchControlImageXSize")) res->PitchControlImageXSize = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "PitchControlImageNb")) res->PitchControlImageNb = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "SeekImage")) res->seek = kj_read_skin(path, argv[1], 0);
	else if(!strcasecmp(argv[0], "VolumeControlType"))
		{
		if(!strcasecmp(argv[1], "BAR")) res->VolumeControlType = 0;
		else if(!strcasecmp(argv[1], "BMP")) res->VolumeControlType = 1;
		else printf("Unknown volume control type: %s\n", argv[1]);
		}
	else if(!strcasecmp(argv[0], "EqualizerBmp"))
		{
		res->equalizer = kj_read_skin(path, argv[3], 2);
		res->equ_band_width = atoi(argv[1]);
		res->equ_bmp_bands = atoi(argv[2]);
		}
	else if(!strcasecmp(argv[0], "EqualizerWindow"))
		{
		set_area(A_EQUALIZER, areas + REGION_EQUALIZER, argc, argv);
		res->equ_num_bands = atoi(argv[6]);
		res->equ_band_space = atoi(argv[7]);
		}

	else if(!strcasecmp(argv[0], "DockModeRCFile")) res->dck_rc_fname = g_strdup(argv[1]);
	else if(!strcasecmp(argv[0], "DockModePositionXY"))
		{ res->DockPos[0] = atoi(argv[1]); res->DockPos[1] = atoi(argv[2]); }

	else if(!strcasecmp(argv[0], "WinshadeModeRCFile")) res->wsh_rc_fname = g_strdup(argv[1]);

	else if(!strcasecmp(argv[0], "AnalyzerColor")) set_colour(&(res->analyser_colour), argc, argv);

	else if(!strcasecmp(argv[0], "PlaylistBmp")) res->pl_background = kj_read_skin(path, argv[1], 2);
	else if(!strcasecmp(argv[0], "PlaylistBmpPressed")) res->pl_pressed = kj_read_skin(path, argv[1], 2);
	else if(!strcasecmp(argv[0], "PlaylistWindowFontYSpacing")) res->pl_fontyspace = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "PlaylistWindowColor")) set_colour(&(res->pl_colour), argc, argv);
	else if(!strcasecmp(argv[0], "PlaylistWindowCurrentTrackColor")) set_colour(&(res->pl_curtrack_colour), argc, argv);
	else if(!strcasecmp(argv[0], "PlaylistWindowCurrentSelectionColor")) set_colour(&(res->pl_cursel_colour), argc, argv);
	else if(!strcasecmp(argv[0], "PlaylistWindowFontSize")) res->pl_fontsize = atoi(argv[1]);
	else if(!strcasecmp(argv[0], "PlaylistWindowFontName")) res->pl_fontname = g_strdup(argv[1]);

	else
		{
		for(ra = resource_areas; ra->name != NULL; ra++)
			if(!strcasecmp(argv[0], ra->name))
				{
				set_area(ra->type, areas + ra->area, argc, argv);
				break;
				}
#if 0
		if(ra->name == NULL)
			{
			int i;
			printf("Not implemented: ");
			for(i = 0; i < argc; i++) printf("`%s' ", argv[i]);
			printf("\n");
			}
#endif
		}
}


/* ---------------------------------------------------------------------- */
void read_rc_file(gchar *path, gchar *fname, k_resource *res, k_area *areas)
{
FILE *fp;
char *ptr, cmd_buf[MAXLINELEN], *cmd_argv[MAXARGS], *inc_rc_file;
int cmd_argc, quote, in_space;

	if((fp = fopen(fname, "r")) == NULL)
		{
		printf("Error opening rc file `%s'\n", fname);
		exit(-1);
		}

	while(fgets(cmd_buf, MAXLINELEN, fp) != NULL)
		{
		ptr = cmd_buf + strlen(cmd_buf)-2;
		if(*ptr == 13) *ptr = 0;
		else if(*(ptr+1) == 10) *(ptr+1) = 0;

		quote = cmd_argc = 0;
		in_space = 1;

		for(ptr = cmd_buf; *ptr; ptr++)
			{
			if(!quote)
				{
				if(*ptr == ';') break;
				else if(*ptr == '#') break;	/* comment */
				else if(*ptr == '`') *ptr = '\"';
				else if (*ptr == ' ')
					{
					*ptr = 0;
					in_space = 1;
					}
				else if(in_space)
					{
					if(cmd_argc >= MAXARGS) break;
					else
						{
						cmd_argv[cmd_argc] = ptr;
						cmd_argc++;
						in_space = 0;
						if(*ptr == '\"')
							{
							quote = 1;
							cmd_argv[cmd_argc-1] = ptr+1;		/* remove quote */
							}
						if(cmd_argc > 1 && !strcasecmp(cmd_argv[0], "About")) break;
						}
					}
				}
			else
				{
				if(*ptr == '\"')
					{
					*ptr = 0;
					quote = 0;
					in_space = 1;
					}
				else if(*ptr == '`') *ptr = '\"';
				}
			}

		if(!cmd_argc) continue;

#if 0
		{
		int i;
		printf(">> ");
		for(i = 0; i < cmd_argc; i++) printf("'%s' ", cmd_argv[i]);
		printf("\n");
		}
#endif

		if(!strcasecmp(cmd_argv[0], "IncludeRCFile") && cmd_argc > 1) 
			{
			if(!(inc_rc_file = kj_find_file_recursively(path, cmd_argv[1], 0)))
				printf("WARNING: file `%s' not found.\n", cmd_argv[1]);
			else
				{
				res->about_str_pos = 0;
				read_rc_file(path, inc_rc_file, res, areas);
				g_free(inc_rc_file);
				}
			}
		else
			set_value(path, res, areas, cmd_argc, cmd_argv);
		}

	fclose(fp);
}


/* ---------------------------------------------------------------------- */
void set_area_digideck(int f, k_area *area, int argc, char **argv)
{
	if(argc < 5) return;
	area->type = f;
	area->focus = area->pressed = area->bm = 0;
	area->x1 = atoi(argv[1]);
	area->y1 = atoi(argv[2]);
	area->x2 = area->x1 + atoi(argv[3]);
	area->y2 = area->y1 + atoi(argv[4]);
	if(area->x1 > area->x2 || area->y1 > area->y2) printf("WARNING: bad area!\n");
	if(argc > 6)
		{
		if(!strcasecmp(argv[6], "BMP1")) area->bm = 0;
		else if(!strcasecmp(argv[6], "BMP2")) area->bm = 1;
		else if(!strcasecmp(argv[6], "BMP3")) area->bm = 2;
		}
}


/* ---------------------------------------------------------------------- */
void set_value_digideck(gchar *path, k_resource *res, k_area *areas, gchar *section, int argc, char **argv)
{
int d = 1;

	if(!strcasecmp(section, "coordinates"))
		{
		if(!strcasecmp(argv[0], "quit")) set_area_digideck(A_BUTTON, areas + BUTTON_CLOSE, argc, argv);
		else if(!strcasecmp(argv[0], "about")) set_area_digideck(A_BUTTON, areas + BUTTON_ABOUT, argc, argv);
		else if(!strcasecmp(argv[0], "minimize")) set_area_digideck(A_BUTTON, areas + BUTTON_MINIMIZE, argc, argv);
		else if(!strcasecmp(argv[0], "prev")) set_area_digideck(A_BUTTON, areas + BUTTON_PREVIOUSSONG, argc, argv);
		else if(!strcasecmp(argv[0], "stop")) set_area_digideck(A_BUTTON, areas + BUTTON_STOP, argc, argv);
		else if(!strcasecmp(argv[0], "pause")) set_area_digideck(A_BUTTON, areas + BUTTON_PAUSE, argc, argv);
		else if(!strcasecmp(argv[0], "play")) set_area_digideck(A_BUTTON, areas + BUTTON_PLAY, argc, argv);
		else if(!strcasecmp(argv[0], "next")) set_area_digideck(A_BUTTON, areas + BUTTON_NEXTSONG, argc, argv);
		else if(!strcasecmp(argv[0], "open")) set_area_digideck(A_BUTTON, areas + BUTTON_OPEN, argc, argv);
		else if(!strcasecmp(argv[0], "volumedown")) set_area_digideck(A_BUTTON, areas + BUTTON_VOLDOWN, argc, argv);
		else if(!strcasecmp(argv[0], "volumeup")) set_area_digideck(A_BUTTON, areas + BUTTON_VOLUP, argc, argv);
		else if(!strcasecmp(argv[0], "playlist")) set_area_digideck(A_BUTTON, areas + BUTTON_PLAYLIST, argc, argv);
		else if(!strcasecmp(argv[0], "shuffle")) set_area_digideck(A_BUTTON, areas + BUTTON_SHUFFLE, argc, argv);
		else if(!strcasecmp(argv[0], "repeat")) set_area_digideck(A_TOGGLE, areas + BUTTON_REPEAT, argc, argv);
		else if(!strcasecmp(argv[0], "seekbar")) set_area_digideck(A_BAR, areas + REGION_SEEKBAR, argc, argv);
		else d = 0;
		}
	else d = 0;

#if 1
	if(!d)
		{
		int i;
		printf("Not implemented: %s - ", section);
		for(i = 0; i < argc; i++) printf("`%s' ", argv[i]);
		printf("\n");
		}
#endif
}


/* ---------------------------------------------------------------------- */
void read_digideck_skin(gchar *path, gchar *fname, k_resource *res, k_area *areas)
{
FILE *fp;
char *ptr, cmd_buf[MAXLINELEN], *cmd_argv[MAXARGS], *section = NULL;
int cmd_argc, quote, in_space;
gulong tcol;

	if((fp = fopen(fname, "r")) == NULL)
		{
		printf("Error opening digideck skin file `%s'\n", fname);
		exit(-1);
		}

	res->background = kj_read_skin(path, "mainup.bmp", 1);
	res->pressed[0] = kj_read_skin(path, "maindown.bmp", 1);

	tcol = kj_get_pixel(res->background, 0, 0);
	kj_mask_colour(res->background, tcol);
	tcol = kj_get_pixel(res->pressed[0], 0, 0);
	kj_mask_colour(res->pressed[0], tcol);

	while(fgets(cmd_buf, MAXLINELEN, fp) != NULL)
		{
		ptr = cmd_buf + strlen(cmd_buf)-2;
		if(*ptr == 13) *ptr = 0;
		else if(*(ptr+1) == 10) *(ptr+1) = 0;

		if(cmd_buf[0] == '[')
			{
			if((ptr = strchr(cmd_buf, ']')))
				{
				*ptr = '\0';
				if(section) g_free(section);
				section = g_strdup(cmd_buf + 1);
				}
			}
		else if(cmd_buf[0] != '#' && section != NULL)
			{
			if((ptr = strchr(cmd_buf, '=')))
				{
				*ptr++ = '\0';
				quote = 0;
				cmd_argc = in_space = 1;
				cmd_argv[0] = cmd_buf;

				for( ; *ptr; ptr++)
					{
					if(!quote)
						{
						if(*ptr == ';') break;
						else if(*ptr == '#') break;	/* comment */
						else if(*ptr == '`') *ptr = '\"';
						else if (*ptr == ' ' || *ptr == ',')
							{
							*ptr = 0;
							in_space = 1;
							}
						else if(in_space)
							{
							if(cmd_argc >= MAXARGS) break;
							else
								{
								cmd_argv[cmd_argc] = ptr;
								cmd_argc++;
								in_space = 0;
								if(*ptr == '\"')
									{
									quote = 1;
									cmd_argv[cmd_argc-1] = ptr+1;		/* remove quote */
									}
								}
							}
						}
					else
						{
						if(*ptr == '\"')
							{
							*ptr = 0;
							quote = 0;
							in_space = 1;
							}
						else if(*ptr == '`') *ptr = '\"';
						}
					}

				set_value_digideck(path, res, areas, section, cmd_argc, cmd_argv);
				}
			}
		}

	printf("Digideck skin... still implementing...\n");
	if(section) g_free(section);
	fclose(fp);
}


/* ---------------------------------------------------------------------- */
static void set_font_skin_type(k_font *font, int type)
{
	font->type = type;

		/* generate transparent mask */
	if(font->img && font->transparent)
		{
		font->tcol = kj_get_pixel(font->img, font->img->width-1, font->img->height-1);
		kj_mask_colour(font->img, font->tcol);
		}
}


/* ----------------------------------------------------------------------
 * Ugly hack to generate visualisation colours.  K-Jofol does not normally
 * support multi-colour vis.
 */
static void gen_vis_colour(GdkColor *orig, GdkColor *dst)
{
int i, dr, dg, db, sr, sg, sb;

		/* peak colour */
	dst[0].red = (3*0xffff + (long)orig->red)/4;
	dst[0].green = (3*0xffff + (long)orig->green)/4;
	dst[0].blue = (3*0xffff + (long)orig->blue)/4;
	gdk_color_alloc(gdk_colormap_get_system(), dst);

		/* fire/line colour */
	dr = orig->red/3;
	dg = orig->green/3;
	db = orig->blue/3;
	sr = (0xffff + (long)orig->red)/2;
	sg = (0xffff + (long)orig->green)/2;
	sb = (0xffff + (long)orig->blue)/2;

	for(i = 0; i < 24; i++)
		{
		dst[i+1].red = dr + ((sr - dr) * i)/24;
		dst[i+1].green = dg + ((sg - dg) * i)/24;
		dst[i+1].blue = db + ((sb - db) * i)/24;
		gdk_color_alloc(gdk_colormap_get_system(), dst + i + 1);
		}
}


/* ---------------------------------------------------------------------- */
void free_resource(k_resource *res)
{
int i;

	for(i = 0; i < MAX_ABOUT_STR; i++)
		if(res->about_str[i])
			{
			g_free(res->about_str[i]);
			res->about_str[i] = NULL;
			}
	res->about_str_num = res->about_str_pos = 0;

	if(res->dck_rc_fname) g_free(res->dck_rc_fname);
	if(res->wsh_rc_fname) g_free(res->wsh_rc_fname);
	if(res->background) kj_free_image(res->background);
	if(res->inactive) kj_free_image(res->inactive);
	if(res->pressed[0]) kj_free_image(res->pressed[0]);
	if(res->pressed[1]) kj_free_image(res->pressed[1]);
	if(res->pressed[2]) kj_free_image(res->pressed[2]);
	if(res->volume) kj_free_image(res->volume);
	if(res->volume_position) kj_free_image(res->volume_position);
	if(res->pitch) kj_free_image(res->pitch);
	if(res->pitch_position) kj_free_image(res->pitch_position);
	if(res->seek) kj_free_image(res->seek);
	if(res->seek_sel) gdk_image_destroy(res->seek_sel);
	if(res->seek_unsel) gdk_image_destroy(res->seek_unsel);
	if(res->Font.img) kj_free_image(res->Font.img);
	if(res->TimeFont.img) kj_free_image(res->TimeFont.img);
	if(res->VolumeFont.img) kj_free_image(res->VolumeFont.img);
	if(res->PitchFont.img) kj_free_image(res->PitchFont.img);
	if(res->equalizer) kj_free_image(res->equalizer);
	if(res->pl_background) kj_free_image(res->pl_background);
	if(res->pl_pressed) kj_free_image(res->pl_pressed);
	if(res->pl_fontname) g_free(res->pl_fontname);

	res->dck_rc_fname = NULL;
	res->wsh_rc_fname = NULL;
	res->background = NULL;
	res->inactive = NULL;
	res->pressed[0] = NULL;
	res->pressed[1] = NULL;
	res->pressed[2] = NULL;
	res->volume = NULL;
	res->volume_position = NULL;
	res->pitch = NULL;
	res->pitch_position = NULL;
	res->seek = NULL;
	res->seek_sel = NULL;
	res->seek_unsel = NULL;
	res->Font.img = NULL;
	res->TimeFont.img = NULL;
	res->VolumeFont.img = NULL;
	res->PitchFont.img = NULL;
	res->dock_mode = 0;
	res->equalizer = NULL;
	res->pl_background = NULL;
	res->pl_pressed = NULL;
	res->pl_fontname = NULL;
	memset(res->areas, 0, sizeof(k_area) * NUM_AREAS);
}


/* ---------------------------------------------------------------------- */
int load_resource(char *path, char *rc_fname, k_resource *res)
{
char *ending, *tmpdir, *unzip, *tmp, *rc_file;
k_area *pos;
int ret;

	if(path && res->res_name && !strcasecmp(res->res_name, path)) return 1;
	free_resource(res);

	if(path && strncmp(path, P_tmpdir, strlen(P_tmpdir)))
		{
		if(res->res_name) g_free(res->res_name);
		res->res_name = g_strdup(path);
		}

	if(!path) path = res->res_name;

		/* defaults? (at least K-Nine seems to think so...) */
	res->Font.transparent = res->TimeFont.transparent
		= res->VolumeFont.transparent = res->PitchFont.transparent = 1;

	ending = strrchr(path, '.');
	if(ending && (!strcasecmp(ending, ".zip")))
		{
		tmpdir = tempnam(NULL, NULL);
		unzip = getenv("UNZIPCMD");
		if(!unzip) unzip = "unzip";
		tmp = g_strdup_printf("%s >/dev/null -o -j \"%s\" -d %s", unzip, path, tmpdir);
		system(tmp);
		g_free(tmp);
		ret = load_resource(tmpdir, rc_fname, res);
		kj_del_directory(tmpdir);
		free(tmpdir);
		return ret;
		}

#if DIGIDECK
	if((rc_file = kj_find_file_recursively(path, "skin.ini", 0)))
		{
		read_digideck_skin(path, rc_file, res, res->areas);
		}
	else
#endif
		{
		if(rc_fname) rc_file = kj_find_file_recursively(path, rc_fname, 0);
		else rc_file = kj_find_file_recursively(path, ".rc", 1);
		if(!rc_file)
			{
			printf("ERROR: RC file not found.\n");
			return 0;
			}

		read_rc_file(path, rc_file, res, res->areas);
		}

		/* grab region for seek and unseek from dithered images */
	pos = res->areas + REGION_SEEK;
	if(pos->type != A_NONE && res->background && res->pressed[pos->bm])
		{
		res->seek_sel = gdk_image_get(res->pressed[pos->bm]->pix, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
		res->seek_unsel = gdk_image_get(res->background->pix, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
		}

		/* generate square mask for window */
	if(res->background && !res->background->mask)
		{
		GdkGC *gc;
		GdkColor c;
		res->background->mask = gdk_pixmap_new(root_window, res->background->width, res->background->height, 1);
		gc = gdk_gc_new(res->background->mask);
		c.pixel = 1;
		gdk_gc_set_foreground(gc, &c);
		gdk_draw_rectangle(res->background->mask, gc, TRUE, 0, 0, res->background->width, res->background->height);
		gdk_gc_destroy(gc);
		}

		/* set font type and mask */
	set_font_skin_type(&(res->Font), 1);
	set_font_skin_type(&(res->TimeFont), 2);
	set_font_skin_type(&(res->VolumeFont), 2);
	set_font_skin_type(&(res->PitchFont), 2);

	gen_vis_colour(&(res->analyser_colour), res->vis_colour);

	g_free(rc_file);

	return 1;
}

