/*

    xpuyopuyo - pnetgame.c    Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Send and receive a runtime game packets
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <xpuyopuyo.h>


#if USE_NETWORK /* Allow network support? */


#include <stdlib.h>
#include <stdio.h>
#include <config.h>
#include <pnetint.h>
#include <pconfig.h>
#include <pplayer.h>
#include <pgame.h>


int pnet_send_rocks(pconfig *c, pplayer *p, int numrocks) {

   if(c->socket == NULL) return(0);
   if(numrocks <= 0) return(0);

   /* Send configuration */   
   if(pnet_send_header(c, c->socket, P_NET_SEND_ROCKS_ID, sizeof(word)) < 0) return(-1);
   if(pnet_send_word(c, c->socket, numrocks) < 0) return(-1);
   
   #ifdef P_NET_DEBUG_COMM
   printf("send_rocks:  Sent %d rocks\n", numrocks);
   #endif /* P_NET_DEBUG_COMM */
   
   return(0);

}


int pnet_send_status(pconfig *c, pplayer *p) {

   byte packet[P_NET_STATUS_SIZE];
   int i;

   if(c->socket == NULL) return(0);

   /* Setup status record */
   *(dword *)(packet) = htonl(p->rocks);
   *(dword *)(packet + 4) = htonl(p->score);
   *(dword *)(packet + 8) = htonl(p->scorerocks);
   *(dword *)(packet + 12) = htonl(p->f_pieces);
   *(dword *)(packet + 16) = htonl(p->f_rksent);
   *(dword *)(packet + 20) = htonl(p->f_rkrecv);
   
   /* Send clump information */
   for(i = 0; i < P_PLAYER_NUM_CLUMPS; i++) {
      *(dword *)(packet + 4 * i + 24) = htonl(p->num_clumps[i]);
   }

   /* Send status information */
   if(pnet_send_header(c, c->socket, P_NET_STATUS_ID, P_NET_STATUS_SIZE) < 0) return(-1);
   if(pnet_send_buffer(c, c->socket, packet, P_NET_STATUS_SIZE) < 0) return(-1);
   
   #ifdef P_NET_DEBUG_COMM
   printf("send_status:  Sent status record\n");
   #endif /* P_NET_DEBUG_COMM */
   
   return(0);

}


int pnet_send_defeat(pconfig *c, pplayer *p) {

   if(c->socket == NULL) return(0);
   
   /* Send final status */
   if(pnet_send_status(c, p) < 0) return(-1);

   /* Send defeat information */
   if(pnet_send_header(c, c->socket, P_NET_I_LOST, 0) < 0) return(-1);
   
   #ifdef P_NET_DEBUG_COMM
   printf("send_defeat:  Sent surrender\n");
   #endif /* P_NET_DEBUG_COMM */
   
   return(0);

}


int pnet_send_end(pconfig *c) {

   if(c->socket == NULL) return(0);

   /* Send end gambit */   
   if(pnet_send_header(c, c->socket, P_NET_END_ID, 0) < 0) return(-1);
   
   #ifdef P_NET_DEBUG_COMM
   printf("send_end:  Sent endgame\n");
   #endif /* P_NET_DEBUG_COMM */
   
   return(0);

}


int pnet_send_pause(pconfig *c, int paused) {

   word type;

   if(c->socket == NULL) return(0);

   /* Send end gambit */   
   type = paused ? P_NET_PAUSE_ID : P_NET_UNPAUSE_ID;
   if(pnet_send_header(c, c->socket, type, 0) < 0) return(-1);
   
   #ifdef P_NET_DEBUG_COMM
   printf("send_pause:  Sent %spause\n", paused ? "" : "un");
   #endif /* P_NET_DEBUG_COMM */
   
   return(0);

}


int pnet_send_next(pconfig *c, pplayer *p) {

   if(c->socket == NULL) return(0);

   /* Send our next piece to opponent */   
   if(pnet_send_header(c, c->socket, P_NET_SEND_NEXT_ID, P_NET_SEND_NEXT_SIZE) < 0) return(-1);
   if(pnet_send_dword(c, c->socket, p->next.block[0]) < 0) return(-1);
   if(pnet_send_dword(c, c->socket, p->next.block[1]) < 0) return(-1);
   
   #ifdef P_NET_DEBUG_COMM
   printf("send_pause:  Sent our next piece\n");
   #endif /* P_NET_DEBUG_COMM */
   
   return(0);

}


static void pnet_send_board_set_data(pconfig *c, pplayer *p, byte *buffer, int x1, int y1, int x2, int y2) {

   pfield *piece;
   word value;
   int x;
   int y;
   int i;

   /* Copy game field to temporary structure */   
   piece = p->field->piece;
   for(y = y1, i = 0; y <= y2; y++) {
      for(x = x1; x <= x2; x++, i++) {
         value = P_CLEAR;
         if(x >= piece->x && x < piece->x + 2 && y >= piece->y && y < piece->y + 2) {
            value = p_field_get(piece, x - piece->x, y - piece->y);
         }
         if(P_IS_CLEAR(value)) value = p_field_get(p->field, x, y);
         *(word *)(buffer + i * sizeof(word)) = htons(value);
      }
   }
   
}


int pnet_send_board(pconfig *c, pplayer *p) {

   byte *packet;

   /* Allocate temporary packet structure */
   packet = (byte *)malloc(p->field->size * sizeof(word));
   if(packet == NULL) {
      pnet_set_error("send_board", "cannot allocate memory for packet");
      return(-1);
   }
   
   /* Copy game field to temporary structure */   
   pnet_send_board_set_data(c, p, packet, 0, 0, p->field->width - 1, p->field->height - 1);
   
   /* Send gameboard */
   if(pnet_send_header(c, c->socket, P_NET_BOARD_ID, p->field->size * sizeof(word)) < 0) return(-1);
   if(pnet_send_buffer(c, c->socket, packet, p->field->size * sizeof(word)) < 0) return(-1);
   
   #ifdef P_NET_DEBUG_COMM
   printf("send_board:  Sent %dx%d gameboard from player %d\n", p->field->width, p->field->height, p->number);
   #endif /* P_NET_DEBUG_COMM */
   
   /* Release and return */
   free(packet);
   return(0);
   
}


int pnet_send_subboard(pconfig *c, pplayer *p, int x1, int y1, int x2, int y2) {

   byte *packet;
   word numpix;

   /* Nothing to do if range is empty */
   if(x2 < x1 || y2 < y1) return(0);

   /* Allocate temporary packet structure */
   if(x1 < 0) x1 = 0;
   if(y1 < 0) y1 = 0;
   if(x2 >= p->field->width)  x2 = p->field->width - 1;
   if(y2 >= p->field->height) y2 = p->field->height - 1;
   numpix = (x2 - x1 + 1) * (y2 - y1 + 1);

   packet = (byte *)malloc(numpix * sizeof(word));
   if(packet == NULL) {
      pnet_set_error("send_board", "cannot allocate memory for packet");
      return(-1);
   }
   
   /* Copy game field to temporary structure */   
   pnet_send_board_set_data(c, p, packet, x1, y1, x2, y2);
   
   /* Send gameboard */
   if(pnet_send_header(c, c->socket, P_NET_SUBBOARD_ID, 4 * sizeof(word) + numpix * sizeof(word)) < 0) return(-1);
   if(pnet_send_word(c, c->socket, x1) < 0) return(-1);
   if(pnet_send_word(c, c->socket, y1) < 0) return(-1);
   if(pnet_send_word(c, c->socket, x2) < 0) return(-1);
   if(pnet_send_word(c, c->socket, y2) < 0) return(-1);
   if(pnet_send_buffer(c, c->socket, packet, numpix * sizeof(word)) < 0) return(-1);
   
   #ifdef P_NET_DEBUG_COMM
   printf("send_subboard:  Sent %dx%d gameboard from player %d\n", x2 - x1 + 1, y2 - y1 + 1, p->number);
   #endif /* P_NET_DEBUG_COMM */
   
   /* Release and return */
   free(packet);
   return(0);
   
}


static int pnet_recv_rocks(pconfig *c, pplayer *p, word type, word size) {

   word value;

   if(pnet_recv_word(c, c->socket, &value) < 0) return(-1);

   P_PLAYER(c, P_OTHER_PLAYER(p->number))->rocks += value;

   #ifdef P_NET_DEBUG_COMM
   printf("recv_game:  Got %ld rocks\n", value);
   #endif /* P_NET_DEBUG_COMM */
   
   return(0);
      
}


static int pnet_recv_status(pconfig *c, pplayer *p, word type, word size) {

   byte packet[P_NET_STATUS_SIZE];
   int i;

   if(pnet_recv_buffer(c, c->socket, packet, P_NET_STATUS_SIZE) < 0) return(-1);

   p->rocks = ntohl(*(dword *)(packet));
   p->score = ntohl(*(dword *)(packet + 4));
   p->scorerocks = ntohl(*(dword *)(packet + 8));
   p->f_pieces = ntohl(*(dword *)(packet + 12));
   p->f_rksent = ntohl(*(dword *)(packet + 16));
   p->f_rkrecv = ntohl(*(dword *)(packet + 20));

   /* Process clump information */
   for(i = 0; i < P_PLAYER_NUM_CLUMPS; i++) {
      p->num_clumps[i] = ntohl(*(dword *)(packet + 4 * i + 24));
   }

   #ifdef P_NET_DEBUG_COMM
   printf("recv_game:  Got status packet\n");
   #endif /* P_NET_DEBUG_COMM */
   
   return(0);

}


static void pnet_recv_board_get_data(pconfig *c, pplayer *p, const byte *buffer, int x1, int y1, int x2, int y2) {

   word value;
   int x;
   int y;
   int i;

   /* Load information into the board */
   for(y = y1, i = 0; y <= y2; y++) {
      for(x = x1; x <= x2; x++, i++) {
         value = ntohs(*(dword *)(buffer + i * sizeof(word)));
         if(value != p_field_get(p->field, x, y)) {
            p_field_set(p->field, x, y, value);
            p_field_redraw_point(p->field, x, y);
         }
      }
   }

}


static int pnet_recv_board(pconfig *c, pplayer *p, word type, word size) {

   byte *packet;

   /* Allocate memory for the data packet */
   packet = (unsigned char *)malloc(p->field->size * sizeof(word));
   if(packet == NULL) {
      pnet_set_error("recv_game", "cannot allocate memory for packet");
      return(-1);
   }

   /* Receive the packet information */
   if(pnet_recv_buffer(c, c->socket, packet, p->field->size * sizeof(word)) < 0) return(-1);

   /* Load information into the board */
   pnet_recv_board_get_data(c, p, packet, 0, 0, p->field->width - 1, p->field->height - 1);

   #ifdef P_NET_DEBUG_COMM
   printf("recv_game:  Got %dx%d gameboard for player %d\n", p->field->width, p->field->height, p->number);
   #endif /* P_NET_DEBUG_COMM */

   /* Done. */
   free(packet);
   return(0);

}


static int pnet_recv_subboard(pconfig *c, pplayer *p, word type, word size) {

   byte *packet;
   word numpix;
   word x1;
   word y1;
   word x2;
   word y2;

   /* Get the dimensions to work with */
   if(pnet_recv_word(c, c->socket, &x1) < 0) return(-1);
   if(pnet_recv_word(c, c->socket, &y1) < 0) return(-1);
   if(pnet_recv_word(c, c->socket, &x2) < 0) return(-1);
   if(pnet_recv_word(c, c->socket, &y2) < 0) return(-1);
   numpix = (x2 - x1 + 1) * (y2 - y1 + 1);

   /* Allocate memory for the packet; receive the packet */   
   packet = (unsigned char *)malloc(numpix * sizeof(word));
   if(packet == NULL) {
      pnet_set_error("recv_game", "cannot allocate memory for packet");
      return(-1);
   }
   if(pnet_recv_buffer(c, c->socket, packet, numpix * sizeof(word)) < 0) return(-1);

   /* Load data into playing field */
   pnet_recv_board_get_data(c, p, packet, x1, y1, x2, y2);
   
   #ifdef P_NET_DEBUG_COMM
   printf("recv_game:  Got %dx%d gameboard for player %d\n", x2 - x1 + 1, y2 - y1 + 1, p->number);
   #endif /* P_NET_DEBUG_COMM */

   /* Done. */
   free(packet);
   return(0);

}


static int pnet_recv_next(pconfig *c, pplayer *p, word type, word size) {

   dword value;
   int i;

   /* Read each piece */
   for(i = 0; i < 2; i++) {
      if(pnet_recv_dword(c, c->socket, &value) < 0) return(-1);
      p->next.block[i] = value;
   }

   #ifdef P_NET_DEBUG_COMM
   printf("recv_game:  Got next piece for opponent\n");
   #endif /* P_NET_DEBUG_COMM */
   
   return(0);

}


static int pnet_recv_pause(pconfig *c, pplayer *p, word type, word size) {

   p_game_pause_(c, type == P_NET_PAUSE_ID);
   
   #ifdef P_NET_DEBUG_COMM
   printf("recv_game:  Received %spause from opponent\n", type == P_NET_PAUSE_ID ? "" : "un");
   #endif /* P_NET_DEBUG_COMM */
   
   return(0);
   
}


static int pnet_recv_defeat(pconfig *c, pplayer *p, word type, word size) {

   #ifdef P_NET_DEBUG_COMM
   printf("recv_game:  Received defeat from opponent\n");
   #endif /* P_NET_DEBUG_COMM */

   return(P_NET_RES_LOST);

}


static int pnet_recv_endgame(pconfig *c, pplayer *p, word type, word size) {

   p_game_end_(c);
   
   #ifdef P_NET_DEBUG_COMM
   printf("recv_game:  Received endgame from opponent\n");
   #endif /* P_NET_DEBUG_COMM */

   return(P_NET_RES_ENDGAME);

}


int pnet_recv_game(pconfig *c, pplayer *p) {

   word type;
   word size;
   int sel;
   int res;

   if(c->socket == NULL) return(0);

   /* See if any data is available */   
   sel = pnet_ready_to_recv(c, c->socket);
   res = 0;

   while(sel > 0 && res == 0) {

      /* Something is available; what is it? */
      if(pnet_recv_header(c, c->socket, &type, &size) < 0) return(-1);

      /* Check what type */
      switch(type) {
      case P_NET_SEND_ROCKS_ID:    /* Opponent is sending some rocks */
         res = pnet_recv_rocks(c, p, type, size);
         break;
         
      case P_NET_STATUS_ID:    /* Opponent is sending their current status */
         res = pnet_recv_status(c, p, type, size);
         break;
         
      case P_NET_BOARD_ID:    /* Opponent is sending their current board */
         res = pnet_recv_board(c, p, type, size);
         break;

      case P_NET_SUBBOARD_ID: /* Opponent is sending their current board */
         res = pnet_recv_subboard(c, p, type, size);
         break;

      case P_NET_SEND_NEXT_ID:/* Opponent sends us their next piece */
         res = pnet_recv_next(c, p, type, size);
         break;
         
      case P_NET_PAUSE_ID:    /* Opponent requested a pause */
      case P_NET_UNPAUSE_ID:  /* Opponent requested an un-pause */
         res = pnet_recv_pause(c, p, type, size);
         break;

      case P_NET_I_LOST:      /* Opponent just lost the game! */
         res = pnet_recv_defeat(c, p, type, size);
         break;

      case P_NET_END_ID:      /* Opponent sent endgame */
         res = pnet_recv_endgame(c, p, type, size);
         break;

      default:                /* Unknown packet */
         pnet_recv_null(c, c->socket, size);
         break;

      }

      /* See if any data is available */   
      if(res == 0) {
         sel = pnet_ready_to_recv(c, c->socket);
      }
      
   } /* End while loop */

   if(sel < 0) return(-1);
   return(res);

}


int pnet_recv_for_unpause(pconfig *c) {

   word type;
   word size;
   int sel;
   int res;

   if(c->socket == NULL) return(0);

   /* See if any data is available */   
   sel = pnet_ready_to_recv(c, c->socket);
   res = 0;

   while(res == 0 && sel > 0) {

      /* Something is available; what is it? */
      if(pnet_recv_header(c, c->socket, &type, &size) < 0) return(-1);

      /* Check what type */
      switch(type) {

      case P_NET_UNPAUSE_ID:  /* Opponent requested an un-pause */
         res = pnet_recv_pause(c, NULL, type, size);
         break;

      case P_NET_END_ID:      /* Opponent sent endgame */
         res = pnet_recv_endgame(c, NULL, type, size);
         break;

      default:                /* Unknown packet */
         pnet_recv_null(c, c->socket, size);
         break;

      }

      /* See if any data is available */   
      if(res == 0) {
         sel = pnet_ready_to_recv(c, c->socket);
      }
      
   } /* End while loop */

   if(sel < 0) return(-1);
   return(res);

}


#endif /* Allow network? */
