/*-
# MOTIF/X-BASED MISSING LINK(tm)
#
#  xmlink.c
#
###
#
#  Copyright (c) 1994 - 2007	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/11/15 X/Windows
  Version 5: 95/10/01 Xt/Motif
  Version 4: 94/08/30 Xt
*/

#include "file.h"
#ifdef WINVER
#include "MlinkP.h"
#define TITLE "wmlink"

static MlinkRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->mlink.started = b
#else
#include "xwin.h"
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#ifdef MOUSEBITMAPS
#include "pixmaps/mouse-l.xbm"
#include "pixmaps/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Mlink.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "pixmaps/mlink.t.xpm"
#include "pixmaps/mlink.p.xpm"
#include "pixmaps/mlink.s.xpm"
#include "pixmaps/mlink.m.xpm"
#include "pixmaps/mlink.l.xpm"
#include "pixmaps/mlink.xpm"
#define RESIZE_XPM(s) ((char **) ((s)<=32)?\
(((s)<=22)?(((s)<=16)?mlink_t_xpm:mlink_p_xpm):\
(((s)<=24)?mlink_s_xpm:mlink_m_xpm)):\
(((s)<=48)?mlink_l_xpm:mlink_xpm))
#endif
#include "pixmaps/mlink.xbm"
#define DEFINE_XBM (char *) mlink_bits, mlink_width, mlink_height
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef WINVER
static const char aboutHelp[] = {
"Mlink Version 7.3.2\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd@tux.org>\n"
"The latest version is at: "
"http://www.tux.org/~bagleyd/puzzles.html\n"
};

static const char optionsHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-tile {color}] [-face{0|1|2|3|4|5|6|7} {color}]\n"
"[-{border|bd} {color}] [-[no]install]\n"
"[-picture {filename}] [-delay msecs] [-[no]sound]\n"
"[-bumpSound {filename}] [-moveSound {filename}]\n"
"[-{font|fn} {fontname}] [-tiles {int}] [-faces {int}]\n"
"[-[no]orient] [-[no]middle] [-base {int}]\n"
"[-userName {string}] [-scoreFile {filename}] [-scores]\n"
"[-version]\n"
};
#endif

#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
"Ideal's The Missing Link, a 4x4 puzzle,  riding on the wave "
"of Rubik's Cube craze.  There are  many variations on this\n"
"theme, including the \"6x6\" puzzle and Babylon Tower.  What "
"makes the Missing Link a harder puzzle to solve is that the\n"
"2 center positions rotate together.\n"
};

static const char featuresHelp[] = {
"Press \"mouse-left\" button to move a tile. Release "
"\"mouse-left\" button in the same row would move the tile towards\n"
"the space (unless there is no space in that row).  "
"Release on a different row will move the tiles up or down.\n"
"\n"
"Click \"mouse-center\" button, or press \"P\" or \"p\" "
"keys, to toggle the practice mode (in practice mode the\n"
"record should say \"practice\").  This is good for learning "
"moves and experimenting.\n"
"\n"
"Click \"mouse-right\" button, or press \"Z\" or \"z\" "
"keys, to randomize the puzzle (this must be done first\n"
"to set a new record).\n"
"\n"
"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"C\" or \"c\" keys to clear the puzzle.\n"
"\n"
"\"S\" or \"s\" keys reserved for the auto-solver "
"(not implemented).\n"
"\n"
"Press \"O\" or \"o\" keys to toggle orient mode.  Slightly "
"more challenging on (default is off).\n"
"\n"
"Press \"M\" or \"m\" keys to toggle the middle mode.  All "
"the middle tiles rotate together.  This is considerably\n"
"more challenging on (default is on).\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of tiles.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of tiles.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Missing Link as:\n"
"  8    Up\n"
"  ^\n"
"4< >6  Left, Right\n"
"  v\n"
"  2    Down\n"
"\n"
"Use the shift key and the up or down arrow keys to rotate "
"all the tiles.\n"
"\n"
"Use the control key and the keypad or arrow keys to reduce "
"or enlarge the complexity of the puzzle.\n"
};

static const char referencesHelp[] = {
"L. E. Horden, Sliding Piece Puzzles (Recreations in "
"Mathematics Series), Oxford University\n"
"Press 1986, Plate X.\n"
"you are about to find the solution: The Missing Link, "
"Ideal Toy Corporation, 1981.\n"
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, only implemented "
"for tiles = 4 and faces = 4.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "mlink.scores"
#endif

#define MAXTILES 8
#define NEVER (-1)
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} PuzzleRecord;

static PuzzleRecord puzzleRecord[2][2]
[MAXFACES - MINFACES + 1][MAXTILES - MINTILES + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";

#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
static Widget moves, record, message;
static Widget orientizeSwitch, middleSwitch, practiceSwitch, tile, face, speed;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
static Arg arg[3];
#else
static char titleDsp[TITLELEN] = "";
#endif
static char scoreFileName[FILENAMELEN] = SCOREFILE;
static char fileName[FILENAMELEN];
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char userNameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap pixmap = None;
static Widget topLevel, puzzle;
static char *progDsp;
static char userNameDsp[USERNAMELEN] = "";

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w, XmNlabelString, xmstr, NULL);
}
#endif

static void
PrintRecords(void)
{
	int i, j, k, l;

	(void) printf("                    MLINK  HALL OF FAME\n\n");
	(void) printf("MIDDLE ORIENT FACE TILE USER            MOVES\n");
	for (l = 0; l < 2; l++)
		for (k = 0; k < 2; k++)
			for (j = 0; j < MAXFACES - MINFACES + 1; j++)
				for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
					if (puzzleRecord[l][k][j][i].score > 0)
						(void) printf("%6d%7d%5d%5d %-12s%9d\n",
							l, k, j + 1, i + 1,
							puzzleRecord[l][k][j][i].name,
							puzzleRecord[l][k][j][i].score);
				}
}
#endif

static void
InitRecords(void)
{
	int i, j, k, l;

	for (l = 0; l < 2; l++)
		for (k = 0; k < 2; k++)
			for (j = 0; j < MAXFACES - MINFACES + 1; j++)
				for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
					puzzleRecord[l][k][j][i].score = NEVER;
					(void) strncpy(puzzleRecord[l][k][j][i].name,
						NOACCESS, USERNAMELEN);
				}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int i, j, k, l, n;
	char userName[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USERNAMELEN);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USERNAMELEN);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (l = 0; l < 2; l++)
		for (k = 0; k < 2; k++)
			for (j = 0; j < MAXFACES - MINFACES + 1; j++)
				for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
					(void) fscanf(fp, "%d %s\n", &n, userName);
					if (n <= puzzleRecord[l][k][j][i].score ||
							puzzleRecord[l][k][j][i].score <= NEVER) {
						puzzleRecord[l][k][j][i].score = n;
						(void) strncpy(puzzleRecord[l][k][j][i].name,
							userName, USERNAMELEN);
					}
				}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int i, j, k, l;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Can not write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILENAMELEN];

		(void) strncpy(lockFile, fileName, FILENAMELEN - 6);
		(void) strcat(lockFile, ".lock");
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (l = 0; l < 2; l++) {
			for (k = 0; k < 2; k++) {
				for (j = 0; j < MAXFACES - MINFACES + 1; j++) {
					for (i = 0; i < MAXTILES - MINTILES + 1; i++)
						(void) fprintf(fp, "%d %s\n",
							puzzleRecord[l][k][j][i].score, puzzleRecord[l][k][j][i].name);
					(void) fprintf(fp, "\n");
				}
				(void) fprintf(fp, "\n");
			}
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
PrintRecord(int tiles, int faces, Boolean orient, Boolean middle,
		Boolean practice)
{
	int i = tiles - MINTILES, j = faces - MINFACES;
	int k = (orient) ? 1 : 0, l = (middle) ? 1 : 0;

	if (practice) {
		(void) strncpy(recordDsp, "practice", MESSAGELEN);
	} else if (tiles > MAXTILES) {
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	} else if (puzzleRecord[l][k][j][i].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			puzzleRecord[l][k][j][i].score,
			puzzleRecord[l][k][j][i].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static void
PrintStatus(char *msg, int nMoves
#ifndef HAVE_MOTIF
		, int tiles, int faces, Boolean middle
#endif
		)
{
#ifdef HAVE_MOTIF
	PrintState(message, msg);
	(void) sprintf(buff, "%d", nMoves);
	PrintState(moves, buff);
#else
	if (middle)
		(void) sprintf(titleDsp, "%s: %dx%d norm@ (%d/%s) - %s",
			progDsp, tiles, faces, nMoves, recordDsp, msg);
	else
		(void) sprintf(titleDsp, "%s: %dx%d ind@ (%d/%s) - %s",
			progDsp, tiles, faces, nMoves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
	XtVaSetValues(XtParent(puzzle), XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
HandleSolved(int counter, int tiles, int faces, Boolean orient, Boolean middle)
{
	int i = tiles - MINTILES, j = faces - MINFACES;
	int k = (orient) ? 1 : 0, l = (middle) ? 1 : 0;

	if (tiles <= MAXTILES && (counter < puzzleRecord[l][k][j][i].score ||
			puzzleRecord[l][k][j][i].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		puzzleRecord[l][k][j][i].score = counter;
		(void) strncpy(puzzleRecord[l][k][j][i].name, userNameDsp,
			USERNAMELEN);
		if (tiles < 4 || faces < 2) {
			puzzleRecord[!l][k][j][i].score = counter;
			(void) strncpy(puzzleRecord[!l][k][j][i].name,
				userNameDsp, USERNAMELEN);
		}
		WriteRecords();
		PrintRecord(tiles, faces, orient, middle, False);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
MlinkWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int tiles, faces;
	Boolean orient, middle, practice;
	char *userName, *scoreFile;

#ifdef WINVER
	InitializePuzzle(w, brush);

	tiles = w->mlink.tiles;
	faces = w->mlink.faces;
	orient = w->mlink.orient;
	middle = w->mlink.middle;
	practice = w->mlink.practice;
	userName = w->mlink.userName;
	scoreFile = w->mlink.scoreFile;
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
#else
	Boolean scoreOnly, versionOnly;

	XtVaGetValues(w,
		XtNuserName, &userName,
		XtNscoreFile, &scoreFile,
		XtNtiles, &tiles,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNmiddle, &middle,
		XtNpractice, &practice,
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf(aboutHelp);
		exit(0);
	}
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
	if (scoreOnly) {
		InitRecords();
		ReadRecords();
		PrintRecords();
		exit(0);
	}
#ifdef HAVE_MOTIF
	if (tiles > MAXTILES)
		XtVaSetValues(tile, XmNmaximum, tiles, NULL);
	XmScaleSetValue(tile, tiles);
	XmScaleSetValue(face, faces);
	XmToggleButtonSetState(orientizeSwitch, orient, True);
	XmToggleButtonSetState(middleSwitch, middle, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
	{
		int delay;

		XtVaGetValues(w, XtNdelay, &delay, NULL);
		XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay - 1);
	}
#endif
#endif
	SET_STARTED(w, False);
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USERNAMELEN);
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USERNAMELEN);
#else
		char *login = getlogin();

		if (login == NULL) {
			(void) strcpy(userNameDsp, "");
		} else {
			(void) sprintf(userNameDsp, "%s", login);
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(userNameDsp, "%s", "guest");
#endif
	}
	PrintRecord(tiles, faces, orient, middle, practice);
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, tiles, faces, middle
#endif
		);
}

#ifdef WINVER
void
setPuzzle(MlinkWidget w, int reason)
#else
static void
CallbackPuzzle(Widget w, caddr_t clientData, mlinkCallbackStruct *callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int tiles, faces;
	Boolean orient, middle, practice, cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	tiles = w->mlink.tiles;
	faces = w->mlink.faces;
	orient = w->mlink.orient;
	middle = w->mlink.middle;
	practice = w->mlink.practice;
	cheat = w->mlink.cheat;
#else
	XtVaGetValues(w,
		XtNtiles, &tiles,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNmiddle, &middle,
		XtNpractice, &practice,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
	case ACTION_HIDE:
#ifdef WINVER
		ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
		(void) XIconifyWindow(XtDisplay(topLevel),
			XtWindow(topLevel),
			XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
		break;
#ifndef WINVER
	case ACTION_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(practiceDialog);
#else
		XtVaSetValues(puzzle, XtNmenu, ACTION_PRACTICE, NULL);
#endif
		break;
	case ACTION_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(randomizeDialog);
#else
		XtVaSetValues(puzzle, XtNmenu, ACTION_RANDOMIZE, NULL);
#endif
		break;
#endif
	case ACTION_SOLVE_MESSAGE:
#ifdef WINVER
		(void) MessageBox(w->core.hWnd, solveHelp,
			"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
		XtManageChild(solveDialog);
#else
		(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
		break;
	case ACTION_RESTORE:
		if (practice) {
			(void) strncpy(recordDsp, "practice", MESSAGELEN);
#ifdef HAVE_MOTIF
			PrintState(record, recordDsp);
#endif
		}
		movesDsp = 0;
		break;
	case ACTION_RESET:
		movesDsp = 0;
		break;
	case ACTION_BLOCKED:
		(void) strncpy(messageDsp, "Blocked", MESSAGELEN);
		break;
	case ACTION_SPACE:
#if 0
		/* Too annoying */
		(void) strncpy(messageDsp, "A space can not slide",
			MESSAGELEN);
#endif
		break;
	case ACTION_IGNORE:
		(void) strncpy(messageDsp, "Randomize to start",
			MESSAGELEN);
		break;
	case ACTION_MOVED:
		movesDsp++;
		SET_STARTED(w, True);
		break;
	case ACTION_CONTROL:
		return;
	case ACTION_SOLVED:
		if (practice)
			movesDsp = 0;
		else if (cheat)
			(void) sprintf(messageDsp,
				"No cheating %s!!", userNameDsp);
		else if (HandleSolved(movesDsp, tiles, faces,
				orient, middle))
			(void) sprintf(messageDsp,
				"Congratulations %s!!", userNameDsp);
		else
			(void) strncpy(messageDsp, "Solved!",
				MESSAGELEN);
		SET_STARTED(w, False);
		break;
	case ACTION_COMPUTED:
		SET_STARTED(w, False);
		break;
	case ACTION_PRACTICE:
		movesDsp = 0;
		practice = !practice;
		if (!practice)
			(void) strncpy(messageDsp, "Randomize to start",
				MESSAGELEN);
		PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
		w->mlink.practice = practice;
		w->mlink.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, practice,
			XtNstart, False, NULL);
#endif
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
		break;
	case ACTION_RANDOMIZE:
		movesDsp = 0;
#ifdef WINVER
		w->mlink.practice = False;
		w->mlink.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, False,
			XtNstart, False, NULL);
#endif
		break;
	case ACTION_ORIENTIZE:
		movesDsp = 0;
		orient = !orient;
		PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
		w->mlink.orient = orient;
#else
		XtVaSetValues(w, XtNorient, orient, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(orientizeSwitch, orient, True);
#endif
#endif
		break;
	case ACTION_MIDDLE:
		movesDsp = 0;
		middle = !middle;
		PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
		w->mlink.middle = middle;
#else
		XtVaSetValues(w, XtNmiddle, middle, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(middleSwitch, middle, True);
#endif
#endif
		break;
	case ACTION_INCX:
		movesDsp = 0;
		tiles++;
		PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
		w->mlink.tiles = tiles;
#else
		XtVaSetValues(w, XtNtiles, tiles, NULL);
#ifdef HAVE_MOTIF
		if (tiles > MAXTILES)
			XtVaSetValues(tile, XmNmaximum, tiles, NULL);
		XmScaleSetValue(tile, tiles);
#endif
#endif
		break;
	case ACTION_DECX:
		movesDsp = 0;
		tiles--;
		PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
		w->mlink.tiles = tiles;
#else
		XtVaSetValues(w, XtNtiles, tiles, NULL);
#ifdef HAVE_MOTIF
		XmScaleSetValue(tile, tiles);
		if (tiles >= MAXTILES)
			XtVaSetValues(tile, XmNmaximum, tiles, NULL);
#endif
#endif
		break;
	case ACTION_INCY:
		movesDsp = 0;
		faces++;
		PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
		w->mlink.faces = faces;
#else
		XtVaSetValues(w, XtNfaces, faces, NULL);
#ifdef HAVE_MOTIF
		XmScaleSetValue(face, faces);
#endif
#endif
		break;
	case ACTION_DECY:
		movesDsp = 0;
		faces--;
		PrintRecord(tiles, faces, orient, middle, practice);
#ifdef WINVER
		w->mlink.faces = faces;
#else
		XtVaSetValues(w, XtNfaces, faces, NULL);
#ifdef HAVE_MOTIF
		XmScaleSetValue(face, faces);
#endif
#endif
		break;
	case ACTION_UNDO:
		movesDsp--;
		SET_STARTED(w, True);
		break;
	case ACTION_REDO:
		movesDsp++;
		SET_STARTED(w, True);
		break;
#ifdef HAVE_MOTIF
	case ACTION_SPEED:
		{
			int oldDelay, delay;

			XtVaGetValues(puzzle,
				XtNdelay, &delay, NULL);
			oldDelay = delay;
			if (delay > MAXSPEED - MINSPEED)
				delay = MAXSPEED - MINSPEED;
			if (delay < 0)
				delay = 0;
			if (delay != oldDelay) {
				XtVaSetValues(w,
					XtNdelay, delay, NULL);
			}
			XmScaleSetValue(speed,
				MAXSPEED + MINSPEED - delay - 1);
		}
		return;
#endif
	}
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, tiles, faces, middle
#endif
		);
}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
	case WM_COMMAND:
		if (LOWORD(wParam) == IDOK) {
			(void) EndDialog(hDlg, TRUE);
			return TRUE;
		}
		break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;
	static Boolean mousePressed = False;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.mlink.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			EnterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.mlink.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			LeavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
	case WM_CREATE:
		Initialize(&widget, brush);
		break;
	case WM_DESTROY:
		DestroyPuzzle(&widget, brush);
		break;
	case WM_SIZE:
		ResizePuzzle(&widget);
		(void) InvalidateRect(hWnd, NULL, TRUE);
		break;
	case WM_PAINT:
		widget.core.hDC = BeginPaint(hWnd, &paint);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		ExposePuzzle(&widget);
		(void) EndPaint(hWnd, &paint);
		break;
	case WM_RBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		RandomizePuzzle(&widget);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONDOWN:
		mousePressed = True;
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		SelectPuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			((GetKeyState(VK_SHIFT) >> 1) ||
			(GetKeyState(VK_CAPITAL) & 1)));
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_MOUSEMOVE:
		if (!(wParam & MK_LBUTTON)) {
			mousePressed = False;
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ReleasePuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) ||
				(GetKeyState(VK_CAPITAL) & 1)));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		}
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		MotionPuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			((GetKeyState(VK_SHIFT) >> 1) ||
			(GetKeyState(VK_CAPITAL) & 1)));
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONUP:
		mousePressed = False;
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		ReleasePuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			((GetKeyState(VK_SHIFT) >> 1) ||
			(GetKeyState(VK_CAPITAL) & 1)));
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
	case WM_MOUSEWHEEL:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		{
			int zDelta = ((short) HIWORD(wParam));
			POINT cursor, origin;

			origin.x = 0, origin.y = 0;
			ClientToScreen(hWnd, &origin);
			(void) GetCursorPos(&cursor);
			if (zDelta > (WHEEL_DELTA >> 1)) {
				MovePuzzleInput(&widget, cursor.x - origin.x,
					TOP,
					((GetKeyState(VK_SHIFT) >> 1) ||
					(GetKeyState(VK_CAPITAL) & 1)),
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
				if (GetKeyState(VK_CONTROL) >> 1) {
					SizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
				}
			} else if (zDelta < -(WHEEL_DELTA >> 1)) {
				MovePuzzleInput(&widget, cursor.x - origin.x,
					BOTTOM,
					((GetKeyState(VK_SHIFT) >> 1) ||
					(GetKeyState(VK_CAPITAL) & 1)),
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
				if (GetKeyState(VK_CONTROL) >> 1) {
					SizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
				}
			}
		}
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#endif
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case ACTION_GET:
			GetPuzzle(&widget);
			ResizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_WRITE:
			WritePuzzle(&widget);
			break;
		case ACTION_EXIT:
			DestroyPuzzle(&widget, brush);
			break;
		case ACTION_HIDE:
			HidePuzzle(&widget);
			break;
		case ACTION_UNDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			UndoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_REDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			RedoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_CLEAR:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ClearPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_PRACTICE:
			PracticePuzzle(&widget);
			SizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_RANDOMIZE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			RandomizePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_SOLVE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			SolvePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_ORIENTIZE:
			OrientizePuzzle(&widget);
			SizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_MIDDLE:
			(void) MiddlePuzzle(&widget);
			SizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_LEFT:
		case ACTION_RIGHT:
		case ACTION_SHIFT_UP:
		case ACTION_SHIFT_DOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			(void) MovePuzzleInput(&widget, 0,
				(int) LOWORD(wParam) - ACTION_SHIFT_UP,
				!((int) wParam % 2), FALSE);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_UP:
		case ACTION_DOWN:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				(void) MovePuzzleInput(&widget,
					cursor.x - origin.x,
					(int) LOWORD(wParam) - ACTION_UP,
					FALSE, FALSE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_DECY:
		case ACTION_INCX:
		case ACTION_INCY:
		case ACTION_DECX:
			MovePuzzleInput(&widget, 0,
				(int) LOWORD(wParam) - ACTION_DECY,
				FALSE, TRUE);
			SizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_SPEED:
			SpeedPuzzle(&widget);
			break;
		case ACTION_SLOW:
			SlowPuzzle(&widget);
			break;
		case ACTION_SOUND:
			SoundPuzzle(&widget);
			break;
		case ACTION_DESCRIPTION:
			(void) MessageBox(hWnd, descriptionHelp,
				"Description", MB_OK);
			break;
		case ACTION_FEATURES:
			(void) MessageBox(hWnd, featuresHelp,
				"Features", MB_OK);
			break;
		case ACTION_REFERENCES:
			(void) MessageBox(hWnd, referencesHelp,
				"References", MB_OK);
			break;
		case ACTION_ABOUT:
			(void) DialogBox(widget.core.hInstance,
				"About", hWnd, (DLGPROC) About);
			break;
		}
		break;
	default:
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#if 0
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#else
		512 + 10, 512 + 10 + 34,
#endif
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char *programName)
{
	unsigned int i;

	(void) fprintf(stderr, "usage: %s\n", programName);
	for (i = 0; i < strlen(optionsHelp); i++) {
		if (i == 0 || optionsHelp[i - 1] == '\n')
			(void) fprintf(stderr, "\t");
		(void) fprintf(stderr, "%c", (optionsHelp[i]));
	}
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*mlink.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*mlink.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*mlink.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*mlink.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*mlink.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*mlink.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*mlink.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*mlink.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*mlink.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*mlink.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-tile", (char *) "*mlink.tileColor", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*mlink.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*mlink.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*mlink.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*mlink.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*mlink.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*mlink.faceColor5", XrmoptionSepArg, NULL},
	{(char *) "-face6", (char *) "*mlink.faceColor6", XrmoptionSepArg, NULL},
	{(char *) "-face7", (char *) "*mlink.faceColor7", XrmoptionSepArg, NULL},
	{(char *) "-install", (char *) "*mlink.install", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noinstall", (char *) "*mlink.install", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-picture", (char *) "*mlink.picture", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*mlink.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*mlink.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*mlink.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-bumpSound", (char *) "*mlink.bumpSound", XrmoptionSepArg, NULL},
	{(char *) "-moveSound", (char *) "*mlink.moveSound", XrmoptionSepArg, NULL},
	{(char *) "-base", (char *) "*mlink.base", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*mlink.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*mlink.font", XrmoptionSepArg, NULL},
	{(char *) "-tiles", (char *) "*mlink.tiles", XrmoptionSepArg, NULL},
	{(char *) "-faces", (char *) "*mlink.faces", XrmoptionSepArg, NULL},
	{(char *) "-orient", (char *) "*mlink.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*mlink.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-middle", (char *) "*mlink.middle", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomiddle", (char *) "*mlink.middle", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*mlink.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*mlink.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-userName", (char *) "*mlink.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*mlink.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*mlink.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*mlink.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
CallbackPuzzlePractice(Widget w, XtPointer clientData,
		XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(puzzle, XtNmenu, ACTION_PRACTICE, NULL);
	}
}

static void
CallbackPuzzleRandomize(Widget w, XtPointer clientData,
		XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(puzzle, XtNmenu, ACTION_RANDOMIZE, NULL);
	}
}

static void
TileSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int tiles = cbs->value, faces, old;
	Boolean orient, middle, practice;

	XtVaGetValues(puzzle,
		XtNtiles, &old,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNmiddle, &middle,
		XtNpractice, &practice, NULL);
	if (old == tiles)
		return;
	XtVaSetValues(puzzle,
		XtNtiles, tiles, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, middle, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
FaceSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int tiles, faces = cbs->value, old;
	Boolean orient, middle, practice;

	XtVaGetValues(puzzle,
		XtNtiles, &tiles,
		XtNfaces, &old,
		XtNorient, &orient,
		XtNmiddle, &middle,
		XtNpractice, &practice, NULL);
	if (old == faces)
		return;
	XtVaSetValues(puzzle,
		XtNfaces, faces, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, middle, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
OrientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct *cbs)
{
	int tiles, faces;
	Boolean orient = cbs->set, middle, practice, oldOrient;
	char *picture;

	XtVaGetValues(puzzle,
		XtNtiles, &tiles,
		XtNfaces, &faces,
		XtNorient, &oldOrient,
		XtNmiddle, &middle,
		XtNpicture, &picture,
		XtNpractice, &practice, NULL);
	if (oldOrient == orient)
		return;
	if (!orient && picture && *picture) {
		XmToggleButtonSetState(orientizeSwitch, oldOrient, True);
		return;
	}
	XtVaSetValues(puzzle,
		XtNorient, orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, middle, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
MiddleToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct *cbs)
{
	int tiles, faces;
	Boolean orient, middle = cbs->set, practice, oldMid;

	XtVaGetValues(puzzle,
		XtNtiles, &tiles,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNmiddle, &oldMid,
		XtNpractice, &practice, NULL);
	if (oldMid == middle)
		return;
	XtVaSetValues(puzzle,
		XtNmiddle, middle, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, middle, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PracticeToggle(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct *cbs)
{
	int tiles, faces;
	Boolean orient, middle, practice = cbs->set, oldPract;

	XtVaGetValues(puzzle,
		XtNtiles, &tiles,
		XtNfaces, &faces,
		XtNorient, &orient,
		XtNmiddle, &middle,
		XtNpractice, &oldPract, NULL);
	if (oldPract == practice)
		return;
	XtVaSetValues(puzzle,
		XtNpractice, practice,
		XtNstart, False, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, faces, orient, middle, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGELEN);
	PrintState(message, messageDsp);
}

static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int delay = MAXSPEED + MINSPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(puzzle,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(puzzle, XtNdelay, delay, NULL);
	}
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value + ACTION_GET;

	if (val == ACTION_EXIT)
		exit(0);
	XtVaSetValues(puzzle, XtNmenu, val, NULL);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value + ACTION_UNDO;

	XtVaSetValues(puzzle, XtNmenu, val, NULL);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}

static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
	int pixmapSize;
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, tilesRowCol, speedRowCol, switchRowCol;
	Widget messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString;
	XmString clearString, practiceString, randomizeString, solveString;
	XmString orientizeString, middleString;
	XmString speedString, slowString, soundString;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Mlink",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#ifdef HAVE_MOTIF
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu",
		0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
	practiceString = XmStringCreateSimple((char *) "Practice");
	randomizeString = XmStringCreateSimple((char *) "RandomiZe");
	solveString = XmStringCreateSimple((char *) "Auto-Solve");
	orientizeString = XmStringCreateSimple((char *) "Orientize");
	middleString = XmStringCreateSimple((char *) "Middle");
	speedString = XmStringCreateSimple((char *) ">Speed");
	slowString = XmStringCreateSimple((char *) "<Slow");
	soundString = XmStringCreateSimple((char *) "@Sound");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu",
		1, playCB,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'Z', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, orientizeString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, middleString, 'M', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
	XmStringFree(practiceString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(orientizeString);
	XmStringFree(middleString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure you want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) CallbackPuzzlePractice);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackPuzzleRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move tile", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	tilesRowCol = XtVaCreateManagedWidget("tilesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	tile = XtVaCreateManagedWidget("tile",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Tiles", 6,
		XmNminimum, MINTILES,
		XmNmaximum, MAXTILES,
		XmNvalue, DEFAULTTILES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(tile, XmNvalueChangedCallback,
		(XtCallbackProc) TileSlider, (XtPointer) NULL);
	face = XtVaCreateManagedWidget("face",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Faces", 6,
		XmNminimum, MINFACES,
		XmNmaximum, MAXFACES,
		XmNvalue, DEFAULTFACES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(face, XmNvalueChangedCallback,
		(XtCallbackProc) FaceSlider, (XtPointer) NULL);
	speedRowCol = XtVaCreateManagedWidget("speedRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	speed = XtVaCreateManagedWidget("Animation Speed",
		xmScaleWidgetClass, speedRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED - DEFAULTDELAY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);
	switchRowCol = XtVaCreateManagedWidget("switchRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	orientizeSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTORIENT, NULL);
	XtAddCallback(orientizeSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) OrientToggle, (XtPointer) NULL);
	middleSwitch = XtVaCreateManagedWidget("Middle rigid",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTMIDDLE, NULL);
	XtAddCallback(middleSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) MiddleToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTPRACTICE, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PracticeToggle, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Missing Link! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	puzzle = XtCreateManagedWidget("mlink",
		mlinkWidgetClass, mainPanel, NULL, 0);
#else
	puzzle = XtCreateManagedWidget("mlink",
		mlinkWidgetClass, topLevel, NULL, 0);
#endif
	XtVaGetValues(puzzle,
		XtNpixmapSize, &pixmapSize, NULL);
#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			RESIZE_XPM(pixmapSize), &pixmap, NULL,
			&xpmAttributes);
	}
	if (pixmap == (Pixmap) NULL)
#endif
		pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			DEFINE_XBM);
	XtVaSetValues(topLevel,
#ifdef HAVE_MOTIF
		XmNkeyboardFocusPolicy, XmPOINTER, /* not XmEXPLICIT */
#else
		XtNinput, True,
#endif
		XtNiconPixmap, pixmap, NULL);
	XtAddCallback(puzzle, XtNselectCallback,
		(XtCallbackProc) CallbackPuzzle, (XtPointer) NULL);
	Initialize(puzzle);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(puzzle), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle),
		XCreateFontCursor(XtDisplay(puzzle), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
