use Test::More tests => 41;

use strict;
use File::Spec;
use Zim::Store::Files;
use Zim;

$| = 1;
$Zim::CODESET = 'utf8';
$SIG{__WARN__} = sub {
	print STDERR @_ unless $_[0] =~ /^(#|Searching|Indexing)/;
};

my @clean = qw#
	.zim.cache
	link
	Test/bar.txt
	new copy move Move
#;

# Clean
for (@clean) {
	my $f = Zim::File->new('t', 'notebook', split '/', $_);
	$f->remove if -e $f;
}


# Test a bunch of stuff in the "Files" backend

my $root = File::Spec->rel2abs( File::Spec->catdir(qw/t notebook/) );
my $rep = Zim::Store::Files->new(dir => $root);

is($rep->{format}, 'wiki', 'format'); # 1

print "\n## 2 file(foo:bar) resulted in: ".$rep->file('foo:bar')->path."\n";
ok($rep->file('foo:bar')->path =~ m#/t/notebook/foo/bar.txt$#,
	'filename 1' ); # 2

my $page = $rep->get_page(':foo:bar');
is($page->status, 'new', 'status new page'); # 3
ok(@{$page->get_parse_tree} > 2, 'template'); # 4 
is($page->{format}, 'Zim::Formats::Wiki', 'format class'); # 5

ok($rep->file('Test:wiki')->path =~ m#/t/notebook/Test/wiki.txt$#,
	'filename 2' ); # 6

print "\n## 7 - resolve_name test:WIKI => :Test:wiki\n";
my $name = $rep->resolve_name('test:WIKI');
is($name, ':Test:wiki', 'resolve_name'); # 7

$page = $rep->get_page(':Test:foo:bar');
print "\n## 8 - resolve_name Dus => :Test:foo:Dus\n";
$name = $page->resolve_name('Dus');
is($name, ':Test:foo:Dus', 'resolve_name 1'); # 8

$page = $rep->get_page(':Test:foo:bar');
print "\n## 9 - resolve_name wIKi => :Test:wiki\n";
$name = $page->resolve_name('wIKi');
is($name, ':Test:wiki', 'resolve_name 2'); # 9

$page = $rep->get_page(':Test:foo:bar');
print "\n## 10 - resolve_name Test:Dus => :Test:Dus\n";
$name = $page->resolve_name('Test:Dus');
is($name, ':Test:Dus', 'resolve_name 3'); # 10

$page = $rep->get_page(':Test:foo:bar');
print "\n## 11 - resolve_name test:Dus => :Test:Dus\n";
$name = $page->resolve_name('test:Dus');
is($name, ':Test:Dus', 'resolve_name 4'); # 11

$page = $rep->get_page(':Test:foo:bar');
print "\n## 12 - resolve_name test:WIKI:bar => :Test:wiki:bar\n";
$name = $page->resolve_name('test:WIKI:bar');
is($name, ':Test:wiki:bar', 'resolve_name 5'); # 12

$page = $rep->get_page(':Test:foo:bar');
print "\n## 13 - resolve_name :Dus => :Dus\n";
$name = $page->resolve_name(':Dus');
is($name, ':Dus', 'resolve_name 6'); # 13

$page = $rep->get_page(':Test:foo');
print "\n## 14 - resolve_name .bar => Test:foo:bar\n";
my (undef, $l) = $page->parse_link('.bar');
$name = $page->resolve_name($l);
is($name, ':Test:foo:bar', 'resolve_name 7'); # 14

#$page = $rep->open_page('test:utf8-acchars');

# Clean again
for (@clean) {
	my $f = Zim::File->new('t', 'notebook', split '/', $_);
	$f->remove if -e $f;
}

my @pages = $rep->list_pages(':');
#warn "Pages: ", map(">>$_<< ", @pages), "\n";
is_deeply(\@pages, [qw/Test:/], 'list_pages 1'); # 15

# Clean again
for (@clean) {
	my $f = Zim::File->new('t', 'notebook', split '/', $_);
	$f->remove if -e $f;
}

SKIP: {
	my ($old, $new) = map Zim::File->new(@$_),
		[qw/t notebook Test/], [qw/t notebook link/] ;
	skip('symlink not supported '.$!, 1)
		unless eval { symlink("",""); 1 }
		and symlink($old => $new) ;
	@pages = $rep->list_pages(':');
	is_deeply(\@pages, [qw/link: Test:/], 'list_pages symlink'); # 16
}

@pages = sort $rep->list_pages(':Test:');
#warn "Pages: ", map(">>$_<< ", @pages), "\n";
is_deeply(\@pages, [qw/foo: wiki/], 'list_pages 2'); # 17

# repeating previous test to catch bad caching
@pages = sort $rep->list_pages(':Test:');
#warn "Pages: ", map(">>$_<< ", @pages), "\n";
is_deeply(\@pages, [qw/foo: wiki/], 'list_pages 3'); # 18

# test caching in combo with new content
print "# sleeping 3 seconds ...\n";
print "# The following test may fail depending on IO buffering\n";
sleep 3; # make sure mtime is changed
Zim::File->new(qw/t notebook Test bar.txt/)->write("=== Bar ===\n\nfoo bar !\n" );
@pages = sort $rep->list_pages(':Test:');
#warn "Pages: ", map(">>$_<< ", @pages), "\n";
is_deeply(\@pages, [qw/bar foo: wiki/], 'list_pages 4'); # 19

# create / move / copy / delete
$page = $rep->get_page(':new');
is($page->status, 'new', 'new page 1'); # 20

$page->set_parse_tree(['Page', {}, 'test 1 2 3']);
$page = $rep->get_page(':new');
ok($page->exists, 'create page 1'); # 21

$page = $rep->get_page(':New:Foo');
is($page->status, 'new', 'new page 2'); # 22

$page->set_parse_tree(['Page', {}, 'test 1 2 3']);
$page = $rep->get_page(':New:Foo');
ok($page->exists, 'create page 2'); # 23

$page = $rep->get_page(':new');
$page->move( $rep->get_page(':move') ); # no root wrapper, so need object
$page = $rep->get_page(':new');
ok(! $page->exists, 'page move 1'); # 24
$page = $rep->get_page(':move');
is($page->get_parse_tree()->[2][2], "test 1 2 3\n", 'page move 2'); # 25
	# Returns like ['Page', {}, ['Para', {}, 'test 1 2 3']]

$page->copy( $rep->get_page(':copy') ); # no root wrapper, so need object
$page = $rep->get_page(':move');
is($page->get_parse_tree()->[2][2], "test 1 2 3\n", 'page copy 1');  #26 
$page = $rep->get_page(':copy');
is($page->get_parse_tree()->[2][2], "test 1 2 3\n", 'page copy 2'); # 27 

$rep->get_page(':move')->move( $rep->get_page(':Move') );
is($rep->resolve_name('move'), ':Move', 'move is case sensitive'); # 28

$rep->get_page(':Move')->delete;
$rep->get_page(':copy')->delete;
$page = $rep->get_page(':Move');
ok(!$page->exists, 'delete page 1'); # 29
$page = $rep->get_page(':copy');
ok(!$page->exists, 'delete page 2'); # 30

# Test dispatching to child rep
$rep = Zim->new(dir => $root);

$page = $rep->get_page(':test:wiki');
is($page->{store}{namespace}, ':', 'Store dispatch 1'); # 31

$page = $rep->get_page(':foo:bar');
is($page->{store}{namespace}, ':', 'Store dispatch 2'); # 32

# add child rep
my $node = File::Spec->catdir($root, 'Foo');
$rep->add_child(':foo', 'Files', dir => $node);

$page = $rep->get_page(':test:wiki');
is($page->{store}{namespace}, ':', 'Store dispatch 3'); # 33

$page = $rep->get_page(':foo:bar');
is($page->{store}{namespace}, ':foo:', 'Store dispatch 4'); # 34

$page = $rep->get_page(':Foo');
is($page->{store}{namespace}, ':foo:', 'Store dispatch 5'); # 35

$page = $rep->get_page(':fooBar');
is($page->{store}{namespace}, ':', 'Store dispatch 6'); # 36

ok(! -e $node, 'No pre-mature creation of dir'); # 37

# search
my @res;
my $cb = sub { push @res, @_ };
$rep->search({string => "\x{2022}\x{2022} Search Me \x{2022}\x{2022}"}, $cb);
#use Data::Dumper; print Dumper \@res;
ok(scalar( grep {$$_[0] eq ':Test:foo:bar' and $$_[1] > 0} @res),
	"utf8 search pattern" ); # 38

# Relative path
$page = $rep->resolve_page(':test:wiki');
my $path = File::Spec->catfile($root, 'foo.pdf');
is($page->relative_path($path), '../foo.pdf', 'relative_path 1'); # 39
$path = File::Spec->catfile($root, 'Test', 'foo.pdf');
warn "### ", $page->relative_path($path), "\n";
is($page->relative_path($path), './foo.pdf', 'relative_path 2'); # 40
$path = Zim::File->abs_path(
	File::Spec->catfile($root, '..', '..', 'foo.pdf') );
is($page->relative_path($path), $path, 'relative_path 3'); # 41

