/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: plug.c,v 1.89 2004/07/22 09:47:36 bazsi Exp $
 *
 * Author: Balazs Scheidler <bazsi@balabit.hu>
 * Auditor:
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/plugsession.h>
#include <zorp/thread.h>
#include <zorp/streamfd.h>
#include <zorp/proxy.h>
#include <zorp/poll.h>
#include <zorp/policy.h>
#include <zorp/thread.h>
#include <zorp/log.h>
#include <zorp/registry.h>
#include <zorp/sockaddr.h>
#include <zorp/io.h>
#include <zorp/fastpath.h>

#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <arpa/inet.h>

#define PLUG_DEFAULT_BUFSIZE    1500

#define PLUG_DEBUG "plug.debug"
#define PLUG_ERROR "plug.error"
#define PLUG_POLICY "plug.policy"
#define PLUG_SESSION "plug.session"

typedef struct _PlugProxy
{
  ZProxy super;
  ZSockAddr *master_client, *master_server;
  ZPoll *poll;
  GAsyncQueue *session_queue;
  ZPlugSessionData session_data;
  ZDispatchEntry *secondary_dispatch;
  guint secondary_mask;
  guint secondary_sessions;
} PlugProxy;

extern ZClass PlugProxy__class;
            
#define ENABLE_PKT_STATS 1

static gboolean
plug_packet_stat_event(ZProxy *s, ZSessionVars *vars, 
                       guint64 client_bytes, guint64 client_pkts, 
                       guint64 server_bytes, guint64 server_pkts);
                       
static void plug_proxy_free(ZObject *s);

void
plug_config_set_defaults(PlugProxy *self)
{
  z_proxy_enter(self);

  self->session_data.copy_to_server = TRUE;
  self->session_data.copy_to_client = TRUE;
  self->session_data.timeout = 600000;
  self->session_data.buffer_size = PLUG_DEFAULT_BUFSIZE;
  self->session_data.packet_stats = plug_packet_stat_event;
  
  /* secondary sessions are not permitted */
  self->secondary_mask = ZS_MATCH_ALL;
  self->secondary_sessions = 10;
  
  self->poll = z_poll_new();
  
  z_proxy_leave(self);
}

void
plug_register_vars(PlugProxy *self)
{
  z_proxy_enter(self);

  z_proxy_var_new(&self->super,
                  "secondary_mask",
                  Z_VAR_GET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG | Z_VAR_TYPE_INT,
                  &self->secondary_mask);

  z_proxy_var_new(&self->super,
                  "secondary_sessions",
                  Z_VAR_GET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG | Z_VAR_TYPE_INT,
                  &self->secondary_sessions);

  z_proxy_var_new(&self->super,
                  "timeout",
                  Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_TYPE_INT,
                  &self->session_data.timeout);

  z_proxy_var_new(&self->super,
                  "copy_to_client",
                  Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_TYPE_INT,
                  &self->session_data.copy_to_client);

  z_proxy_var_new(&self->super,
                  "copy_to_server",
                  Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_TYPE_INT,
                  &self->session_data.copy_to_server);

  z_proxy_var_new(&self->super,
                  "shutdown_soft",
                  Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_TYPE_INT,
                  &self->session_data.shutdown_soft);

  z_proxy_var_new(&self->super,
                  "packet_stats_interval_packet",
                  Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_TYPE_INT,
                  &self->session_data.packet_stats_interval_packet);

  z_proxy_var_new(&self->super,
                  "packet_stats_interval_time",
                  Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_TYPE_INT,
                  &self->session_data.packet_stats_interval_time);

  z_proxy_var_new(&self->super,
                  "buffer_size",
                  Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_TYPE_INT,
                  &self->session_data.buffer_size);

  /* Zorp 1.4 compatibility */
  z_proxy_var_new(&self->super,
                  "packet_stats_interval",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_SET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG,
                  "packet_stats_interval_packet");

  z_proxy_leave(self);
}

static gboolean
plug_packet_stat_event(ZProxy *s, ZSessionVars *vars, 
                       guint64 client_bytes, guint64 client_pkts, 
                       guint64 server_bytes, guint64 server_pkts)
{
  PlugProxy *self = (PlugProxy *) s;
  ZPolicyObj *res;
  gboolean called;
  guint resc;

  z_policy_lock(self->super.thread);
  z_proxy_vars_set_active_session(s, vars);
  res = z_policy_call(self->super.handler, "packetStats",
                      z_policy_var_build("iiii",
                                         (guint32) client_bytes,
                                         (guint32) client_pkts,
                                         (guint32) server_bytes,
                                         (guint32) server_pkts),
                      &called,
                      self->super.session_id);
  z_proxy_vars_set_active_session(s, NULL);
  if (called)
    {
      resc = Z_REJECT;
      if (res)
        {
          if (!z_policy_var_parse(res, "i", &resc))
            {
              /*LOG
                This message is logged when the policy layer returned a
                non-integer value in its packetStats() function. packetStats()
                is expected to return Z_REJECT or Z_ACCEPT.
               */
              z_proxy_log(self, PLUG_POLICY, 1, "Invalid return value of packetStats(), integer required;");
            }
          else if (resc != Z_ACCEPT)
            {
              /*LOG
                This message indicates that the verdict returned by the
                packetStats() function requests to terminate the session.
               */
              z_proxy_log(self, PLUG_POLICY, 1, "packetStats() requested to abort session; verdict='%d'", resc);
            }
        }
    }
  else
    resc = Z_ACCEPT;
  z_policy_var_unref(res);
  z_policy_unlock(self->super.thread);
  return resc == Z_ACCEPT;
}

static gboolean
plug_request_stack_event(PlugProxy *self, ZStackedProxy **stacked)
{
  ZPolicyObj *res;
  gboolean called;
  gboolean rc = TRUE;

  z_proxy_enter(self);
  z_policy_lock(self->super.thread);
  *stacked = NULL;
  res = z_policy_call(self->super.handler,
                      "requestStack",
                      NULL,
                      &called,
                      self->super.session_id);
  if (res)
    {
      if (res != z_policy_none)
        {
          *stacked = z_proxy_stack_object(&self->super, res);
        }
    }
  else if (called)
    res = FALSE;
    
  z_policy_var_unref(res);
  z_policy_unlock(self->super.thread);
  z_proxy_leave(self);
  return rc;
}

/* FIXME: make sure the proxy instance has not yet been freed yet, see
 * comment at plug_start_secondary_sessions */
static gboolean
plug_secondary_accept(ZConnection *conn, gpointer user_data)
{
  PlugProxy *self = (PlugProxy *) user_data;
  
  if (!conn)
    {
      /*LOG
       This message is logged when secondary sessions are enabled and there
       was an error accepting the new connection. Probably the OS function
       accept() returned an error.
       */
      z_proxy_log(self, PLUG_ERROR, 3, "Error accepting secondary connection; conn='NULL'");
      return FALSE;
    }
    
  if (z_proxy_get_state(&self->super) != ZPS_WORKING)
    {
      return FALSE;
    }
  if (self->session_data.num_sessions >= self->secondary_sessions)
    {
      /*LOG
        This message is logged when secondary sessions are enabled and a new
        connection is accepted while the maximum number of parallel
        secondary sessions are running. The connection is not accepted by
        this proxy as a result, thus processing will continue on the slow
        path (e.g. another running secondary-session enabled proxy is
        searched for or a new proxy is started).
       */
      z_proxy_log(self, PLUG_DEBUG, 6, 
                  "Maximum number of secondary sessions reached; secondary_sessions='%d'", 
                  self->secondary_sessions);
      return FALSE;
    }
    
  if (!z_sockaddr_inet_check(conn->remote))
    {
      gchar buf[256];
      
      /* we are not interested in non-ipv4 secondary connections */
      
      /*LOG
        This message indicates that a new connection was accepted, but
        it is not an IPv4 connection, thus Plug rejects it.
       */
      z_proxy_log(self, PLUG_DEBUG, 6, 
                  "Incoming secondary connection with non-IPv4 client address; %s", 
                  z_connection_format(conn, buf, sizeof(buf)));
      return FALSE;
    }
    
  if (!z_proxy_check_secondary(&self->super, self->secondary_mask, 
                               self->master_client, self->master_server, 
                               conn->remote, conn->dest))
    {
      return FALSE;
    }
  g_async_queue_push(self->session_queue, conn);
  z_poll_wakeup(self->poll);
  return TRUE;
}

static gboolean
plug_enable_secondary_sessions(PlugProxy *self)
{
  ZSockAddr *listen;
  ZDispatchParams params;
  guint protocol;
  
  z_proxy_enter(self);

  if (self->secondary_mask == ZS_MATCH_ALL)
    {
      z_proxy_leave(self);
      return TRUE;
    }
  
  if (!z_proxy_get_addresses(&self->super, &protocol, &self->master_client, &self->master_server, NULL, NULL, &listen))
    {
      /*LOG
        This message indicates that although secondary sessions are enabled,
        we failed to initialize it, as an error occurred when querying the
        listener address which started this proxy.
       */
      z_proxy_log(self, PLUG_ERROR, 1, "Internal error starting secondary sessions, no listen address set;");
      z_proxy_leave(self);
      return FALSE;
    }
  
  /* we increment the refcount of self, which will be freed when the
   * dispatch entry is unregistered
   */
   
  z_proxy_ref(&self->super);
  
  memset(&params, 0, sizeof(params));
  self->secondary_dispatch = 
    z_dispatch_register(self->super.session_id, 
                        protocol, listen, NULL, 
                        ZD_PRI_NORMAL, 
                        &params, 
                        plug_secondary_accept, self, (GDestroyNotify) z_proxy_unref);
  z_sockaddr_unref(listen);
  
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
plug_disable_secondary_sessions(PlugProxy *self)
{
  if (self->secondary_dispatch)
    {
      z_dispatch_unregister(self->secondary_dispatch);
      self->secondary_dispatch = NULL;
    }
  return TRUE;
}

static gboolean
plug_start_secondary_session(PlugProxy *self, ZConnection *conn)
{
  ZStream *server_stream;
  ZPlugSession *session;
  ZStackedProxy *stacked;
  gchar buf1[128], buf2[128];
  gint fd;
  
  fd = z_stream_get_fd(conn->stream);
  
  /*LOG
    This message indicates that a new secondary session is started with the
    specified parameters.
   */
  z_proxy_log(self, PLUG_SESSION, 3, 
              "Secondary session, client connection accepted; client_fd='%d', client_address='%s', client_local='%s'", 
              fd, z_sockaddr_format(conn->remote, buf1, sizeof(buf1)), z_sockaddr_format(conn->dest, buf2, sizeof(buf2)));
  
  server_stream = z_proxy_fastpath_connect_server(&self->super, conn);
  
  if (!server_stream)
    return FALSE;
  
  if (!plug_request_stack_event(self, &stacked))
    {
      z_stream_close(server_stream, NULL);
      z_stream_unref(server_stream);
      return FALSE;
    }
  
  session = z_plug_session_new(&self->super, &self->session_data, conn->stream, server_stream, stacked);
  if (!session)
    {
      z_stream_close(server_stream, NULL);
      z_stream_unref(server_stream);
      z_stacked_proxy_destroy(stacked);
      return FALSE;
    }
  z_stream_unref(server_stream);
    
  if (!z_plug_session_start(session, self->poll))
    {
      z_plug_session_free(session);
    }
  return TRUE;
}

static gboolean
plug_start_main_session(PlugProxy *self)
{
  ZPlugSession *session;
  ZStackedProxy *stacked;
  
  z_proxy_enter(self);
  
  if (!plug_request_stack_event(self, &stacked))
    {
      z_proxy_leave(self);
      return FALSE;
    }
  
  session = z_plug_session_new(&self->super, &self->session_data, self->super.endpoints[EP_CLIENT], self->super.endpoints[EP_SERVER], stacked);
  
  if (!session)
    {
      z_proxy_leave(self);
      return FALSE;
    }
  z_stream_unref(self->super.endpoints[EP_CLIENT]);
  z_stream_unref(self->super.endpoints[EP_SERVER]);
  self->super.endpoints[EP_CLIENT] = self->super.endpoints[EP_SERVER] = NULL;
  if (!z_plug_session_start(session, self->poll))
    {
      z_plug_session_free(session);
      z_proxy_leave(self);
      return FALSE;
    }
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
plug_main_loop(PlugProxy *self)
{
  while ((g_async_queue_length(self->session_queue) || 
          z_plug_sessions_running(&self->session_data)) && 
         z_poll_iter_timeout(self->poll, -1))
    {
    
      while (g_async_queue_length(self->session_queue) > 0)
        {
          ZConnection *conn = (ZConnection *) g_async_queue_pop(self->session_queue);
          
          if (!plug_start_secondary_session(self, conn))
            z_connection_destroy(conn, TRUE);
          else
            z_connection_destroy(conn, FALSE);
        }
      z_plug_sessions_purge(&self->session_data);
    }
  z_plug_sessions_purge(&self->session_data);
  return TRUE;
}

static void
plug_main(ZProxy *s)
{
  PlugProxy *self = Z_CAST(s, PlugProxy);
  
  z_proxy_enter(self);
  
  if (!z_proxy_connect_server(&self->super, NULL, 0))
    {
      z_proxy_leave(self);
      return;
    }
    
  if (!plug_start_main_session(self))
    {
      z_proxy_leave(self);
      return;
    }
    
  if (!plug_enable_secondary_sessions(self))
    {
      z_proxy_leave(self);
      return;
    }

  plug_main_loop(self);
  
  /* disable secondary sessions */
  plug_disable_secondary_sessions(self);
  
  /* we might have some pending secondary connections which were accepted while
   * we switched into SHUTTING_DOWN state, handle those as well.
   */
  
  plug_main_loop(self);
  
  z_proxy_leave(self);
}

static gboolean
plug_config(ZProxy *s)
{
  PlugProxy *self = (PlugProxy *) s;

  z_proxy_enter(self);

  plug_config_set_defaults(self);
  plug_register_vars(self);
  if (Z_SUPER(s, ZProxy)->config(s))
    {
      z_proxy_leave(self);
      return TRUE;
    }
  z_proxy_leave(self);
  return FALSE;
}

ZProxyFuncs plug_proxy_funcs =
{
  { 
    Z_FUNCS_COUNT(ZProxy),
    plug_proxy_free,
  },
  plug_config,
  NULL,
  plug_main,
  NULL,
  NULL,
  NULL
};

ZClass PlugProxy__class = 
{
  Z_CLASS_HEADER,
  &ZProxy__class,
  "PlugProxy",
  sizeof(PlugProxy),
  &plug_proxy_funcs.super
};

ZProxy *
plug_proxy_new(ZProxyParams *params)
{
  PlugProxy *self;
  
  z_enter();
  self = Z_CAST(z_proxy_new(Z_CLASS(PlugProxy), params), PlugProxy);
  
  self->session_queue = g_async_queue_new();
  z_proxy_start(&self->super);
  z_leave();
  return &self->super;
}
    
static void
plug_proxy_free(ZObject *s)
{
  PlugProxy *self = Z_CAST(s, PlugProxy);
  
  z_proxy_enter(self);
  
  if (self->poll)
    {
      z_poll_unref(self->poll);
      self->poll = NULL;
    }
    

  if (self->session_queue)
    g_async_queue_unref(self->session_queue);

  z_sockaddr_unref(self->master_client);
  z_sockaddr_unref(self->master_server);
  z_proxy_free_method(s);
  z_proxy_leave(self);
}

gint
zorp_module_init(void)
{
  z_registry_add("plug", ZR_PROXY, plug_proxy_new);
  return TRUE;
}
